/*
 * Copyright (c) 2008 Nokia Corporation
 * Contact: integration@maemo.org
 *
 * Authors: Hallyson Melo <hallyson.melo@indt.org.br>
 *          Kyller Gorgonio <kyllercg@gmail.com>
 *          Andre Rodrigues <andre.rodrigues@signove.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <hildon/hildon.h>
#include <string.h>

#include "nm_bt_setup.h"
#include "nm_bt_utils.h"
#include "nm_bt_search_dialog.h"
#include "nm_ifaces_utils.h"
#include "nm_settings.h"
#include "nm_environment_manager.h"


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

static void nm_bt_setup_defaults (NmBtSetup *, gboolean);
static void nm_bt_setup_create_widget(NmBtSetup *);
static void nm_bt_setup_check_initial_state (NmBtSetup *self);
static void nm_bt_setup_enable_cb (GtkToggleButton *, gpointer);
static void nm_bt_setup_enable_options (NmBtSetup *, gboolean);
static gboolean nm_bt_setup_search (GtkButton *, gpointer);
static int nm_bt_setup_enable_dhcpd(NmBtSetup *, NmSettings *, gboolean);
static void nm_bt_setup_request_pin_code_cb(NmBtUtils *, gchar *, gpointer);
static void nm_bt_setup_device_paired_cb(NmBtUtils *, gboolean, gpointer);


typedef struct _NmBtSetupPrivate NmBtSetupPrivate;

struct _NmBtSetupPrivate
{
    GtkWidget *checkbutton;

    GtkWidget *ip_label;
    GtkWidget *gw_label;
    GtkWidget *mask_label;
    GtkWidget *dhcpd_label;

    GtkWidget *ip_entry;
    GtkWidget *gw_entry;
    GtkWidget *mask_entry;
    GtkWidget *dhcpd_entry;

    GtkWidget *btaddr_search;
    GtkWidget *btname_entry;
    GtkWidget *btaddr_frame;

    NmBtUtils *bt_utils;
    gchar *bt_addr;

    gchar *env;
};

#define NM_BT_SETUP_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), NM_BT_SETUP_TYPE, NmBtSetupPrivate))

static void nm_bt_setup_class_init (NmBtSetupClass *klass);
static void nm_bt_setup_init       (NmBtSetup *self);
static void nm_bt_setup_dispose    (GObject *object);
static void nm_bt_setup_finalize   (GObject *object);

G_DEFINE_TYPE (NmBtSetup, nm_bt_setup, NM_CONN_WIDGET_TYPE)

static void
nm_bt_setup_class_init (NmBtSetupClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);
    NmConnWidgetClass *conn_widget_class = NM_CONN_WIDGET_CLASS (klass);

    g_type_class_add_private (klass, sizeof (NmBtSetupPrivate));

    object_class->dispose = nm_bt_setup_dispose;
    object_class->finalize = nm_bt_setup_finalize;

    conn_widget_class->apply_changes = nm_bt_setup_apply_changes;
}

static void
nm_bt_setup_init (NmBtSetup *self)
{
    NmBtSetupPrivate *priv;

    priv = NM_BT_SETUP_GET_PRIVATE (self);

    /* Init BT data, move to BT stuff later */
    priv->bt_addr = NULL;
    priv->bt_utils = nm_bt_utils_new ();

#ifdef FREMANTLE
    g_signal_connect (priv->bt_utils, "request-pin-code", G_CALLBACK(nm_bt_setup_request_pin_code_cb),
                      self);
    g_signal_connect (priv->bt_utils, "device-paired", G_CALLBACK(nm_bt_setup_device_paired_cb),
                      self);
#endif

    nm_bt_setup_create_widget (self);
/*    nm_bt_setup_check_initial_state (self);*/
}

static void
nm_bt_setup_request_pin_code_cb(NmBtUtils *bt_utils, gchar *pin_code, gpointer user_data)
{
	gchar *text = g_strdup_printf("Please enter the following PIN code: %s",
								  pin_code);
	GtkWidget *note = hildon_note_new_information(GTK_WINDOW(user_data), text);

	gtk_widget_show_all(note);
	//gtk_widget_destroy (GTK_WIDGET (note));
	g_free(text);
}

static void
nm_bt_setup_device_paired_cb(NmBtUtils *bt_utils, gboolean paired, gpointer user_data)
{
    NmBtSetupPrivate *priv;
    NmSettings *settings;
    gchar *pand = NULL;

    priv = NM_BT_SETUP_GET_PRIVATE (user_data);

	if (paired) {

        if ((pand = nm_bt_utils_check_pand (priv->bt_utils,
                                             priv->bt_addr)) == NULL) {

            hildon_banner_show_information (NULL, NULL,
                                             "Unable to connect. Check if PAND is running on host!");
            return;
        }

		settings = nm_settings_get_context();

		if (!nm_bt_utils_connect (priv->bt_utils, priv->bt_addr,
								   nm_settings_get_value (settings, priv->env,
														   NM_BT_SETTINGS_GROUP,
														   NM_SETTINGS_GATEWAY))) {

			hildon_banner_show_information (NULL, NULL,
											 "Could not connect to PC");
            return;
		}

		hildon_banner_show_information(NULL, NULL, "Bluetooth network enabled");
	}
	else {

		hildon_banner_show_information(NULL, NULL, "Could not pair PC");
	}

}

static void
nm_bt_setup_dispose (GObject *object)
{
    G_OBJECT_CLASS (nm_bt_setup_parent_class)->dispose (object);
}

static void
nm_bt_setup_finalize (GObject *object)
{
    NmBtSetupPrivate *priv;

    priv = NM_BT_SETUP_GET_PRIVATE (object);

    g_free (priv->bt_addr);
    g_object_unref (priv->bt_utils);

    G_OBJECT_CLASS (nm_bt_setup_parent_class)->finalize (object);
}

GtkWidget*
nm_bt_setup_new (gchar *env)
{
    GtkWidget *self = g_object_new (NM_BT_SETUP_TYPE, NULL);

    NmBtSetupPrivate *priv;

    priv = NM_BT_SETUP_GET_PRIVATE (self);

    priv->env = g_strdup(env);

    g_debug("[%s] - env: %s", __FUNCTION__, priv->env);

    return self;
}

void
nm_bt_setup_enable (NmBtSetup *self, gboolean enable)
{
    NmBtSetupPrivate *priv;

    priv = NM_BT_SETUP_GET_PRIVATE (self);

    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(priv->checkbutton), enable);
}

gboolean
nm_bt_setup_is_enabled (NmBtSetup *self)
{
    NmBtSetupPrivate *priv;

    priv = NM_BT_SETUP_GET_PRIVATE (self);
    return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(priv->checkbutton));
}

static void
nm_bt_setup_create_widget(NmBtSetup *self)
{
    GtkWidget *table;
    GtkWidget *hbox;
    NmBtSetupPrivate *priv;

    priv = NM_BT_SETUP_GET_PRIVATE (self);

    table = gtk_table_new (7, 2, TRUE);
    hbox = gtk_hbox_new (FALSE, 15);

    priv->checkbutton = gtk_check_button_new_with_label ("enable");
    g_signal_connect (priv->checkbutton, "toggled", G_CALLBACK(nm_bt_setup_enable_cb),
                      self);
    
    /* DHCPD server */
    priv->dhcpd_label = gtk_label_new(_("DHCP Server: "));      
    gtk_misc_set_alignment(GTK_MISC(priv->dhcpd_label), 1, 0.5);
    priv->dhcpd_entry = gtk_check_button_new ();
    
    /* IP address */
    priv->ip_label = gtk_label_new(_("IP address: "));
    gtk_misc_set_alignment(GTK_MISC(priv->ip_label), 1, 0.5);
    priv->ip_entry = gtk_entry_new();

    /* Gateway */
    priv->gw_label = gtk_label_new(_("Gateway: "));
    gtk_misc_set_alignment(GTK_MISC(priv->gw_label), 1, 0.5);
    priv->gw_entry = gtk_entry_new();

    /* Network mask */
    priv->mask_label = gtk_label_new(_("Network mask: "));
    gtk_misc_set_alignment(GTK_MISC(priv->mask_label), 1, 0.5);
    priv->mask_entry = gtk_entry_new();

    /* Bluetooth address */
    priv->btaddr_frame = gtk_frame_new (_(" Bluetooth address "));
    priv->btaddr_search = gtk_button_new_with_label (_("Search"));
    g_signal_connect (priv->btaddr_search, "clicked",
                      G_CALLBACK (nm_bt_setup_search), self);

    priv->btname_entry = gtk_entry_new ();
    gtk_entry_set_editable (GTK_ENTRY (priv->btname_entry), FALSE);

    gtk_table_attach_defaults (GTK_TABLE (table), priv->checkbutton, 0, 1, 0, 1);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->dhcpd_label, 0, 1, 1, 2);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->dhcpd_entry, 1, 2, 1, 2);
    gtk_table_attach_defaults (GTK_TABLE (table), priv->ip_label, 0, 1, 2, 3);
    gtk_table_attach_defaults (GTK_TABLE (table), priv->ip_entry, 1, 2, 2, 3);
    gtk_table_attach_defaults (GTK_TABLE (table), priv->gw_label, 0, 1, 3, 4);
    gtk_table_attach_defaults (GTK_TABLE (table), priv->gw_entry, 1, 2, 3, 4);
    gtk_table_attach_defaults (GTK_TABLE (table), priv->mask_label, 0, 1, 4, 5);
    gtk_table_attach_defaults (GTK_TABLE (table), priv->mask_entry, 1, 2, 4, 5);

    gtk_box_pack_start (GTK_BOX (hbox), priv->btaddr_search, FALSE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), priv->btname_entry, FALSE, TRUE, 0);
    gtk_container_add (GTK_CONTAINER (priv->btaddr_frame), hbox);

    gtk_table_attach_defaults (GTK_TABLE (table), priv->btaddr_frame, 0, 2, 5, 7);

    GtkWidget *scroll;
    scroll = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
                                    GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
    gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scroll),
                                           table);

    gtk_container_add(GTK_CONTAINER (self), scroll);

    gtk_widget_set_sensitive (priv->btaddr_frame, FALSE);
    gtk_widget_set_sensitive (priv->btaddr_search, FALSE);
    nm_bt_setup_enable_options (self, FALSE);

    gtk_widget_show_all(GTK_WIDGET(self));
}

static void
nm_bt_setup_check_initial_state (NmBtSetup *self)
{
    NmBtSetupPrivate *priv;

    priv = NM_BT_SETUP_GET_PRIVATE (self);

    if (nm_bt_utils_radio_is_enabled (priv->bt_utils) &&
        nm_bt_utils_has_network_connection (priv->bt_utils)) {

        gchar *name = nm_bt_utils_get_network_connection_end_point (priv->bt_utils);

        if (name) {

            gtk_entry_set_text (GTK_ENTRY (priv->btname_entry), name);
        }
        nm_bt_setup_enable (self, TRUE);
    } else {

        nm_bt_setup_enable (self, FALSE);
    }

    nm_bt_setup_enable_cb (GTK_TOGGLE_BUTTON(priv->checkbutton), self);
/*    nm_bt_setup_enable_options (self, FALSE);*/
}

static void
nm_bt_setup_defaults (NmBtSetup *self, gboolean pressed)
{
    NmBtSetupPrivate *priv;
    NmSettings *settings  = nm_settings_get_context();

    priv = NM_BT_SETUP_GET_PRIVATE (self);

    if (pressed) {
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (priv->dhcpd_entry),
                             nm_settings_get_boolean (settings, priv->env,
                                                      NM_BT_SETTINGS_GROUP,
                                                      NM_SETTINGS_DHCPD));
        gtk_entry_set_text(GTK_ENTRY (priv->ip_entry),
                           nm_settings_get_value (settings, priv->env, NM_BT_SETTINGS_GROUP, NM_SETTINGS_IP));
        gtk_entry_set_text(GTK_ENTRY (priv->gw_entry),
                           nm_settings_get_value (settings, priv->env, NM_BT_SETTINGS_GROUP, NM_SETTINGS_GATEWAY));
        gtk_entry_set_text(GTK_ENTRY (priv->mask_entry),
                           nm_settings_get_value (settings, priv->env, NM_BT_SETTINGS_GROUP, NM_SETTINGS_MASK));
        
        gchar *host = nm_settings_get_value (settings, priv->env, NM_BT_SETTINGS_GROUP, NM_SETTINGS_HOST);
        
        if (host != NULL) {
            gtk_entry_set_text(GTK_ENTRY (priv->btname_entry), host);
        }
        
        g_free(host);
        
        priv->bt_addr = nm_settings_get_value (settings, priv->env, NM_BT_SETTINGS_GROUP, NM_SETTINGS_MAC);
    } else {
        gtk_entry_set_text(GTK_ENTRY (priv->ip_entry), "");
        gtk_entry_set_text(GTK_ENTRY (priv->gw_entry), "");
        gtk_entry_set_text(GTK_ENTRY (priv->mask_entry), "");
    }

}

static void
nm_bt_setup_enable_cb (GtkToggleButton *button, gpointer data)
{
    NmBtSetupPrivate *priv;
    gboolean pressed = gtk_toggle_button_get_active (button);

    priv = NM_BT_SETUP_GET_PRIVATE (data);
    gtk_widget_set_sensitive (priv->btaddr_frame, pressed);
    gtk_widget_set_sensitive (priv->btaddr_search, pressed);
    
    nm_bt_setup_defaults(data, pressed);

    // FIXME: remove this if the combo box is not needed for something else
    nm_bt_setup_enable_options (data, pressed);

    nm_conn_widget_settings_changed (data);
}

static void
nm_bt_setup_enable_options (NmBtSetup *self, gboolean enable)
{
    NmBtSetupPrivate *priv;
    priv = NM_BT_SETUP_GET_PRIVATE (self);

    gtk_widget_set_sensitive (priv->dhcpd_label, enable);
    gtk_widget_set_sensitive (priv->dhcpd_entry, enable);
    gtk_widget_set_sensitive (priv->ip_label, enable);
    gtk_widget_set_sensitive (priv->ip_entry, enable);
    gtk_widget_set_sensitive (priv->gw_label, enable);
    gtk_widget_set_sensitive (priv->gw_entry, enable);
    gtk_widget_set_sensitive (priv->mask_label, enable);
    gtk_widget_set_sensitive (priv->mask_entry, enable);
}

gboolean
nm_bt_setup_apply_changes (NmConnWidget *self)
{
    NmBtSetupPrivate *priv;
    NmBtSetup *btsetup = NM_BT_SETUP (self);
    gchar *pand = NULL;
    NmSettings *settings = nm_settings_get_context();

    priv = NM_BT_SETUP_GET_PRIVATE (self);

    g_debug("[%s]", __FUNCTION__);
    nm_bt_setup_save_changes(btsetup);
    
    // Disable the dhcp server
    nm_bt_setup_enable_dhcpd(btsetup, settings, FALSE);

    if (nm_bt_setup_is_enabled (btsetup)) {

        if (!nm_bt_utils_radio_is_enabled (priv->bt_utils)) {

            gboolean res = nm_bt_utils_radio_enable (priv->bt_utils, TRUE);
            if (!res) {

                hildon_banner_show_information (NULL, NULL,
                                                 _("Error enabling BT radio"));
                return FALSE;
            }
        }



        if (nm_ifaces_utils_save_config_e (NM_IFACES_BT_ST,
                gtk_entry_get_text (GTK_ENTRY (priv->ip_entry)),
                gtk_entry_get_text (GTK_ENTRY (priv->mask_entry)),
                gtk_entry_get_text (GTK_ENTRY (priv->gw_entry))) == FALSE) {

                    exit (1);
        }
#ifdef FREMANTLE
        if (nm_bt_utils_is_paired(priv->bt_utils, priv->bt_addr)) {
#endif       
            if ((pand = nm_bt_utils_check_pand (priv->bt_utils,
                                                 priv->bt_addr)) == NULL) {

                hildon_banner_show_information (NULL, NULL,
                                                 _("Unable to connect. Check if PAND is running on host!"));
                return FALSE;
            }
            
            if (nm_bt_utils_has_network_connection(priv->bt_utils)) {
                nm_bt_utils_remove_connection (priv->bt_utils);
            }

            if (!nm_bt_utils_connect (priv->bt_utils, priv->bt_addr,
                                       nm_settings_get_value (settings, priv->env, 
                                                               NM_BT_SETTINGS_GROUP,
                                                               NM_SETTINGS_GATEWAY))) {

                hildon_banner_show_information (NULL, NULL,
                                                 _("Could not connect to PC"));
                return FALSE;

            }
            else {
                /* bt connection connected */
                
                /* Enable dhcpd if checkbox active */
                if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(priv->dhcpd_entry))) {
                    
                    /* Make sure that the bnep interface is up before start dhcp */
                    int i;
                    for(i = 0; i < 3; ++i) {
                        if (nm_bt_utils_has_network_connection(priv->bt_utils) == FALSE) {
                            g_debug("[%s] - sleep", __FUNCTION__);
                            sleep(1);
                            
                        }
                        else {
                            break;
                        }
                    }
                    g_debug("[%s] - out of sleep", __FUNCTION__);
                    
                    nm_bt_setup_enable_dhcpd(btsetup, settings, TRUE);
                }
            }

    /*        hildon_banner_show_information(NULL, NULL, "Bluetooth network enabled");*/
#ifdef FREMANTLE
        }
        else {

            if (!nm_bt_utils_pair_device(priv->bt_utils, priv->bt_addr)) {

				hildon_banner_show_information (NULL, NULL,
												 _("Could not pair PC"));
				return FALSE;
			}
        }
#endif
    } else {
        
        if (nm_bt_utils_has_network_connection(priv->bt_utils)) {
            if (!nm_bt_utils_remove_connection (priv->bt_utils)) {
                g_debug("[%s] - Error removing connection", __FUNCTION__);
                return FALSE;
            }
        }
        

    }

    return TRUE;
}

static int
nm_bt_setup_enable_dhcpd(NmBtSetup *self, NmSettings *settings, gboolean enable)
{
    NmBtSetupPrivate *priv = NM_BT_SETUP_GET_PRIVATE (self);
    
    system("sudo /usr/sbin/nm-udhcpd stop bnep0");
    
    if (enable == TRUE) {
        const gchar *ip = gtk_entry_get_text (GTK_ENTRY (priv->ip_entry));
        const gchar *netmask = gtk_entry_get_text (GTK_ENTRY (priv->mask_entry));
        gboolean share_bt = nm_settings_get_boolean (settings, priv->env, NM_INTERNET_SETTINGS_GROUP,
    											NM_INTERNET_IN_INTERFACE_BT);
        gchar command[150];

        sprintf(command, "sudo /usr/sbin/nm-udhcpd start bnep0 %s %s", ip, netmask);
    	/* if sharing internet through bluetooth, set dhcp gateway and dns */
        if (share_bt == TRUE) {
            gchar *dns = nm_settings_get_value (settings, priv->env, NM_INTERNET_SETTINGS_GROUP,
                                                NM_INTERNET_DNS);
        	gchar internet[50];

        	sprintf(internet, " %s %s", ip, dns);
        	strcat(command, internet);
        }
        system(command);
    }
    return 1;

}

gboolean 
nm_bt_setup_activate_environment (gchar *env)
{
    g_debug("[%s] - bt_activate", __FUNCTION__);
    gboolean ret = TRUE;
    
    NmSettings *settings = nm_settings_new ();
    nm_settings_start (settings);
    
    gboolean enabled = nm_settings_get_boolean (settings, env,
                                                     NM_BT_SETTINGS_GROUP,
                                                     NM_SETTINGS_ENABLE);

    GtkWidget *bt = nm_bt_setup_new(env);
    nm_bt_setup_enable(NM_BT_SETUP(bt), enabled);
    ret = nm_bt_setup_apply_changes(NM_CONN_WIDGET(bt));

    
    return ret;
}

void 
nm_bt_setup_save_changes (NmBtSetup *self)
{
    NmBtSetupPrivate *priv = NM_BT_SETUP_GET_PRIVATE (self);
    
    NmSettings *settings = nm_settings_get_context();
    
    nm_settings_set_boolean (settings, priv->env, NM_BT_SETTINGS_GROUP,
                                NM_SETTINGS_ENABLE, nm_bt_setup_is_enabled(self) ); 
    
    if ( nm_bt_setup_is_enabled(self) ) {
        nm_settings_set_boolean (settings, priv->env, NM_BT_SETTINGS_GROUP,
                                 NM_SETTINGS_DHCPD, 
                                 gtk_toggle_button_get_active(priv->dhcpd_entry) ); 
        nm_settings_set_value (settings, priv->env, NM_BT_SETTINGS_GROUP,
                                NM_SETTINGS_IP,
                                gtk_entry_get_text (GTK_ENTRY (priv->ip_entry)));
        nm_settings_set_value (settings, priv->env, NM_BT_SETTINGS_GROUP,
                                NM_SETTINGS_GATEWAY,
                                gtk_entry_get_text (GTK_ENTRY (priv->gw_entry)));
        nm_settings_set_value (settings, priv->env, NM_BT_SETTINGS_GROUP,
                                NM_SETTINGS_MASK,
                                gtk_entry_get_text (GTK_ENTRY (priv->mask_entry)));
        nm_settings_set_value (settings, priv->env, NM_BT_SETTINGS_GROUP,
                                NM_SETTINGS_MAC,
                                priv->bt_addr);
        nm_settings_set_value (settings, priv->env, NM_BT_SETTINGS_GROUP,
                                NM_SETTINGS_HOST,
                                gtk_entry_get_text (GTK_ENTRY (priv->btname_entry)));        
    }

    nm_settings_save(settings);
}

static gboolean
nm_bt_setup_search (GtkButton *self, gpointer data)
{
    GtkWidget *search_dialog;
    NmBtSetupPrivate *priv;
    gint ret;

    priv = NM_BT_SETUP_GET_PRIVATE (data);

    search_dialog = nm_bt_search_dialog_new (priv->bt_utils);

    g_idle_add ((GSourceFunc) nm_bt_search_dialog_start_discovery, search_dialog);

    ret = gtk_dialog_run (GTK_DIALOG (search_dialog));

    nm_bt_search_dialog_stop_discovery (NM_BT_SEARCH_DIALOG (search_dialog));

    if (ret == GTK_RESPONSE_OK) {
        gchar *addr, *name;

        addr = nm_bt_search_dialog_get_selected_addr (NM_BT_SEARCH_DIALOG (search_dialog));
        name = nm_bt_search_dialog_get_selected_name (NM_BT_SEARCH_DIALOG (search_dialog));

        if (name != NULL && addr != NULL) {
            gtk_entry_set_text (GTK_ENTRY(priv->btname_entry), name);

            g_debug ("[%s] User selected device: %s - %s", __FUNCTION__, name, addr);

            priv->bt_addr = addr;
            g_free (name);
        }
    } else {

        gtk_widget_destroy (search_dialog);
        return FALSE;
    }

    gtk_widget_destroy (search_dialog);
    return TRUE;
}

gboolean
bt_setup_check_enabled (NmBtSetup *self)
{
    NmBtSetupPrivate *priv = NM_BT_SETUP_GET_PRIVATE (self);

    if (nm_bt_utils_radio_is_enabled (priv->bt_utils) &&
        nm_bt_utils_has_network_connection (priv->bt_utils)) {

        nm_bt_setup_enable (self, TRUE);
        nm_bt_setup_enable_cb (GTK_TOGGLE_BUTTON (priv->checkbutton), self);
        return TRUE;
    } else {

        nm_bt_setup_enable (self, FALSE);
        nm_bt_setup_enable_cb (GTK_TOGGLE_BUTTON (priv->checkbutton), self);
        return FALSE;
    }
}

void
nm_bt_setup_fill_widget(NmBtSetup *self)
{
    NmBtSetupPrivate *priv = NM_BT_SETUP_GET_PRIVATE (self);
    
    if (strcmp(priv->env, nm_environment_manager_get_active())) {
        g_debug("[%s] - %s is not the active environment", __FUNCTION__, priv->env);
        NmSettings *settings = nm_settings_new ();
        nm_settings_start (settings);
        g_debug("[%s] - bt is enabled: %d", __FUNCTION__, 
                    nm_settings_get_boolean (settings, priv->env, NM_BT_SETTINGS_GROUP, NM_SETTINGS_ENABLE));
                    
        nm_bt_setup_enable(self, nm_settings_get_boolean (settings, priv->env, NM_BT_SETTINGS_GROUP, NM_SETTINGS_ENABLE));
        
    }
    else {
        g_debug("[%s] - %s is the active environment", __FUNCTION__, priv->env);
        nm_bt_setup_check_initial_state (self);
    }
}
