#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <syslog.h>
#include <string.h>
#include <glib.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>

#define NM_BT_CONNECT_ARGC 3

/* Bluez interfaces */
#define DBUS_SERVICE_BLUEZ                  "org.bluez"
#define DBUS_INTERFACE_BLUEZ_NETWORK        "org.bluez.Network"

/* Bluez methods */
#define BLUEZ_CONNECT                       "Connect"

GMainLoop *main_loop = NULL;
DBusGConnection *connection = NULL;

static gboolean connect_device (gchar *device_path, gchar* gw);
static void daemonize (void);
static gboolean ifup (const gchar *gw);
static gboolean ifdown (void);

int main(int argc, char *argv[]) {

    if (argc != NM_BT_CONNECT_ARGC) {

        g_debug ("Usage: nm-bt-connect <device_path> <gateway>\n");
        exit (EXIT_FAILURE);
    }

    daemonize ();
    connect_device(argv[1], argv[2]);

	/* Start main loop */
	main_loop = g_main_loop_new(NULL, FALSE);
	g_main_loop_run(main_loop);

	exit(EXIT_SUCCESS);
}

static void daemonize (void) {

	/* Our process ID and Session ID */
	pid_t pid, sid;

	/* Fork off the parent process */
	pid = fork();
	if (pid < 0) {
			exit(EXIT_FAILURE);
	}
	/* If we got a good PID, then
	   we can exit the parent process. */
	if (pid > 0) {
			exit(EXIT_SUCCESS);
	}

	/* Change the file mode mask */
	umask(0);

	/* Create a new SID for the child process */
	sid = setsid();
	if (sid < 0) {
			/* Log the failure */
			exit(EXIT_FAILURE);
	}

	/* Change the current working directory */
	if ((chdir("/")) < 0) {
			/* Log the failure */
			exit(EXIT_FAILURE);
	}

	/* Close out the standard file descriptors */
	close(STDIN_FILENO);
	close(STDOUT_FILENO);
	close(STDERR_FILENO);
}

static gboolean connect_device (gchar *device_path, gchar* gw) {

    DBusGProxy *p_conn = NULL;
    GError *error = NULL;
    gchar *conn_name = NULL;

    g_type_init();
    connection = dbus_g_bus_get (DBUS_BUS_SYSTEM, NULL);
    if (!connection) {

        g_warning ("D-Bus Connection not created");
        g_return_val_if_fail (connection != NULL, FALSE);
    }

	/* Create proxy and call Connect method, wait for reply */
	p_conn = dbus_g_proxy_new_for_name (connection,
										DBUS_SERVICE_BLUEZ,
										device_path,
										DBUS_INTERFACE_BLUEZ_NETWORK);

    g_debug ("%s", device_path);
	if (!dbus_g_proxy_call (p_conn, BLUEZ_CONNECT, &error, G_TYPE_STRING,
						"NAP", G_TYPE_INVALID, G_TYPE_STRING,
						&conn_name,	G_TYPE_INVALID)) {

		if (error->domain == DBUS_GERROR &&
			error->code == DBUS_GERROR_REMOTE_EXCEPTION) {

			g_warning ("Caught remote method exception %s: %s\n\t%s: %d\n",
						dbus_g_error_get_name (error), error->message,
						__FILE__, __LINE__);
		} else {

			g_warning ("Error: %s\n\t%s: %d\n", error->message, __FILE__,
						__LINE__);
		}

		g_error_free (error);
		g_object_unref (p_conn);

		return FALSE;
	}

	ifdown();
	ifup(gw);
	g_free (conn_name);
	g_object_unref (p_conn);

    return TRUE;
}

static gboolean ifup (const gchar *gw)
{
    // FIXME: replace bnep0 by the file retrieved from dbus
    gint status = system("/sbin/ifup bnep0");

    system (g_strconcat ("/sbin/ip route del default via ", gw,
                          " dev bnep0", NULL));
    system (g_strconcat ("/sbin/ip route append default via ", gw,
                          " dev bnep0", NULL));

    return status == 0;
}

static gboolean ifdown (void)
{
    // FIXME: replace bnep0 by the file retrieved from dbus
    gint status = system("/sbin/ifdown bnep0");
    return status == 0;
}
