/*
 * Copyright (c) 2008, 2009 Nokia Corporation
 * Contact: integration@maemo.org
 *
 * Authors: Marcos Morais <morais@dee.ufcg.edu.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */

/* Encapsulate the persistence mechanism */

#include <stdlib.h>
#include <glib-object.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <sys/stat.h>


#include "nm_settings.h"

typedef struct _NmSettingsPrivate NmSettingsPrivate;

struct _NmSettingsPrivate
{
    GKeyFile*   key_file;
};

#define NM_SETTINGS_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), NM_SETTINGS_TYPE, NmSettingsPrivate))

static void nm_settings_class_init (NmSettingsClass *klass);
static void nm_settings_init       (NmSettings *self);
static void nm_settings_dispose    (GObject *object);
static void nm_settings_finalize   (GObject *object);

G_DEFINE_TYPE (NmSettings, nm_settings, G_TYPE_OBJECT)

static NmSettings *the_nm_settings = NULL;

static void
nm_settings_class_init (NmSettingsClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);

    g_type_class_add_private (klass, sizeof (NmSettingsPrivate));

    object_class->dispose = nm_settings_dispose;
    object_class->finalize = nm_settings_finalize;
}

static void
nm_settings_init (NmSettings *self)
{
    NmSettingsPrivate *priv;

    priv = NM_SETTINGS_GET_PRIVATE (self);

    priv->key_file = NULL;
}

gboolean
nm_settings_start (NmSettings *self)
{
    NmSettingsPrivate *priv;
    GError *error = NULL;
    gchar *filename = g_strconcat (g_getenv ("HOME"), "/.pc-connectivity/", NM_SETTINGS_FILE, NULL);

    mkdir(g_strconcat (g_getenv ("HOME"), "/.pc-connectivity",NULL), S_IREAD | S_IWRITE | S_IEXEC);
    
    priv = NM_SETTINGS_GET_PRIVATE (self);

    priv->key_file = g_key_file_new ();
    if (!g_key_file_load_from_file (priv->key_file,                                        /* key_file */
                                    filename,                                       /* directory+filename */
                                    G_KEY_FILE_KEEP_COMMENTS | G_KEY_FILE_KEEP_TRANSLATIONS,
                                    &error))
    {
        g_free(filename);
        g_debug("[%s] - Error reading key: %s", __FUNCTION__, error->message);
        return nm_settings_set_defaults(self);
    }
    g_free(filename);

    return TRUE;
}

static void
nm_settings_dispose (GObject *object)
{
    NmSettingsPrivate *priv;

    priv = NM_SETTINGS_GET_PRIVATE (object);

    if (priv->key_file) {
        g_key_file_free(priv->key_file);
    }

    G_OBJECT_CLASS (nm_settings_parent_class)->dispose (object);
}

static void
nm_settings_finalize (GObject *object)
{
    G_OBJECT_CLASS (nm_settings_parent_class)->finalize (object);
}

/* singleton */
NmSettings*
nm_settings_new (void)
{
    if (!the_nm_settings)
    {
        the_nm_settings = g_object_new (NM_SETTINGS_TYPE, NULL);
    }
    return the_nm_settings;
}

NmSettings*
nm_settings_get_context(void)
{
    return the_nm_settings;
}

gchar*
nm_settings_get_value (NmSettings *self, const gchar *env, const gchar *group, const gchar *key)
{
    GError *error = NULL;
    NmSettingsPrivate *priv = NM_SETTINGS_GET_PRIVATE (self);
    
    if (env == NULL) {
        return g_key_file_get_value (priv->key_file, group, key, &error);
    }
    else {
        gchar *n_group = g_strconcat(env, "-", group, NULL);
        gchar *ret = g_key_file_get_value (priv->key_file, n_group, key, &error);
        
        if ( (ret == NULL) || (error != NULL && error->code == G_FILE_ERROR_NOENT) ) {
            g_free(n_group);
            g_free(ret);
            error = NULL;
            return g_key_file_get_value (priv->key_file, group, key, &error); 
        }
        
        g_free(n_group);
        return ret;
    }
    
    
}

void
nm_settings_set_value (NmSettings *self, const gchar *env, const gchar *group, const gchar *key, const gchar *value)
{
    NmSettingsPrivate *priv;

    priv = NM_SETTINGS_GET_PRIVATE (self);
    
    if (env == NULL) {
        g_key_file_set_value (priv->key_file, group, key, value);
    }
    else {
        gchar *n_group = g_strconcat(env, "-", group, NULL);
        g_key_file_set_value (priv->key_file, n_group, key, value);
        g_free(n_group);
    }
    
}

gboolean
nm_settings_get_boolean (NmSettings *self, const gchar *env, const gchar *group, const gchar *key)
{
    GError *error = NULL;
    NmSettingsPrivate *priv;

    priv = NM_SETTINGS_GET_PRIVATE (self);
    
    if (env == NULL) {
        return g_key_file_get_boolean (priv->key_file, group, key, &error);
    }
    else {
        gchar *n_group = g_strconcat(env, "-", group, NULL);

        gboolean ret = g_key_file_get_boolean (priv->key_file, n_group, key, &error);
        if (error != NULL && error->code == G_FILE_ERROR_NOENT) {
            g_free(n_group);
            error = NULL;
            return g_key_file_get_boolean (priv->key_file, group, key, &error);
        }
        
        g_free(n_group);
        return ret;

    }
}

void
nm_settings_set_boolean (NmSettings *self, const gchar *env, const gchar *group, const gchar *key, gboolean value)
{
    NmSettingsPrivate *priv;

    priv = NM_SETTINGS_GET_PRIVATE (self);
    
    if (env == NULL) {
        g_key_file_set_boolean (priv->key_file, group, key, value);
    }
    else {
        gchar *n_group = g_strconcat(env, "-", group, NULL);
        g_key_file_set_boolean (priv->key_file, n_group, key, value);
        g_free(n_group);
    }
}

void nm_settings_set_string_list (NmSettings *self, const gchar *env,
                                  const gchar *group, const gchar *key,
                                  const gchar** value, gsize length) 
{
    NmSettingsPrivate *priv;

    priv = NM_SETTINGS_GET_PRIVATE (self);
    
    if (env == NULL) {
        g_key_file_set_string_list(priv->key_file, group, key, value, length);
    }
    else {
        gchar *n_group = g_strconcat(env, "-", group, NULL);
        g_key_file_set_string_list(priv->key_file, n_group, key, value, length);
        g_free(n_group);
    }
    
    
}

gchar** nm_settings_get_string_list (NmSettings *self, const gchar *env, const gchar *group, const gchar *key, gsize *length)
{
    GError *error = NULL;
    NmSettingsPrivate *priv;

    priv = NM_SETTINGS_GET_PRIVATE (self);
    
    g_debug("[%s] - env: %s", __FUNCTION__, env);
    
    if (env == NULL) {
        return g_key_file_get_string_list (priv->key_file, group, key, length, &error);
    }
    else {
        gchar *n_group = g_strconcat(env, "-", group, NULL);
        
        gchar** ret = g_key_file_get_string_list (priv->key_file, n_group, key, length, &error);
        
        g_debug("[%s] - n_group: %s", __FUNCTION__, n_group);
        
        if (error != NULL && error->code == G_FILE_ERROR_NOENT) {
            g_free(n_group);
            error = NULL;
            return g_key_file_get_string_list (priv->key_file, group, key, length, &error);
        }
        
        g_free(n_group);
        return ret;
    }
}

void
nm_settings_save (NmSettings *self)
{
    FILE*   conf_file;
    gchar   *file_content = NULL;
    gsize   length;
    GError *error = NULL;
    NmSettingsPrivate *priv;

    const char *filename = g_strconcat (g_getenv ("HOME"), "/.pc-connectivity/", NM_SETTINGS_FILE, NULL);

    priv = NM_SETTINGS_GET_PRIVATE (self);

    file_content = g_key_file_to_data (priv->key_file, &length, &error);

    /* write the file */
    conf_file = fopen (filename, "w");
    if (conf_file) {
        fprintf(conf_file, "%s", file_content);
        fclose (conf_file);
    }

    g_free (file_content);
}

gboolean
nm_settings_set_defaults (NmSettings *self)
{
    /* Set USB defaults */
    nm_settings_set_boolean (self, NM_SETTINGS_DEFAULT_ENV, NM_USB_SETTINGS_GROUP, NM_SETTINGS_ENABLE, FALSE );
    nm_settings_set_value (self, NM_SETTINGS_DEFAULT_ENV, NM_USB_SETTINGS_GROUP, NM_SETTINGS_IP,      NM_USB_DEFAULT_IP);
    nm_settings_set_value (self, NM_SETTINGS_DEFAULT_ENV, NM_USB_SETTINGS_GROUP, NM_SETTINGS_GATEWAY, NM_USB_DEFAULT_GATEWAY);
    nm_settings_set_value (self, NM_SETTINGS_DEFAULT_ENV, NM_USB_SETTINGS_GROUP, NM_SETTINGS_MASK,    NM_USB_DEFAULT_MASK);

    /* Set BT defaults */
    nm_settings_set_boolean (self, NM_SETTINGS_DEFAULT_ENV, NM_BT_SETTINGS_GROUP, NM_SETTINGS_ENABLE,  FALSE);
    nm_settings_set_value (self, NM_SETTINGS_DEFAULT_ENV, NM_BT_SETTINGS_GROUP, NM_SETTINGS_IP,      NM_BT_DEFAULT_IP);
    nm_settings_set_value (self, NM_SETTINGS_DEFAULT_ENV, NM_BT_SETTINGS_GROUP, NM_SETTINGS_GATEWAY, NM_BT_DEFAULT_GATEWAY);
    nm_settings_set_value (self, NM_SETTINGS_DEFAULT_ENV, NM_BT_SETTINGS_GROUP, NM_SETTINGS_MASK,    NM_BT_DEFAULT_MASK);
    nm_settings_set_value (self, NM_SETTINGS_DEFAULT_ENV, NM_BT_SETTINGS_GROUP, NM_SETTINGS_MAC,     NM_BT_DEFAULT_MAC);

    /* Set WLAN defaults */
    nm_settings_set_boolean (self, NM_SETTINGS_DEFAULT_ENV, NM_WLAN_SETTINGS_GROUP, NM_SETTINGS_ENABLE, FALSE);
    nm_settings_set_value (self, NM_SETTINGS_DEFAULT_ENV, NM_WLAN_SETTINGS_GROUP, NM_SETTINGS_IP,      NM_WLAN_DEFAULT_IP);
    nm_settings_set_value (self, NM_SETTINGS_DEFAULT_ENV, NM_WLAN_SETTINGS_GROUP, NM_SETTINGS_GATEWAY, NM_WLAN_DEFAULT_GATEWAY);
    nm_settings_set_value (self, NM_SETTINGS_DEFAULT_ENV, NM_WLAN_SETTINGS_GROUP, NM_SETTINGS_MASK,    NM_WLAN_DEFAULT_MASK);
    nm_settings_set_value (self, NM_SETTINGS_DEFAULT_ENV, NM_WLAN_SETTINGS_GROUP, NM_WLAN_SETTINGS_ESSID,   NM_WLAN_DEFAULT_ESSID);
    nm_settings_set_value (self, NM_SETTINGS_DEFAULT_ENV, NM_WLAN_SETTINGS_GROUP, NM_WLAN_SETTINGS_NAME,    NM_WLAN_DEFAULT_NAME);
    
    /* SBRSH defaults */
    nm_settings_set_boolean (self, NM_SETTINGS_DEFAULT_ENV, NM_SBRSH_SETTINGS_GROUP, NM_SETTINGS_ENABLE, TRUE);
    nm_settings_set_value (self, NM_SETTINGS_DEFAULT_ENV, NM_SBRSH_SETTINGS_GROUP, NM_SBRSH_PORT, "1202");
    nm_settings_set_boolean (self, NM_SETTINGS_DEFAULT_ENV, NM_SBRSH_SETTINGS_GROUP, NM_SETTINGS_NOSANDBOX, FALSE);
    nm_settings_set_boolean (self, NM_SETTINGS_DEFAULT_ENV, NM_SBRSH_SETTINGS_GROUP, NM_SETTINGS_ALLOWROOT, FALSE);
    nm_settings_set_boolean (self, NM_SETTINGS_DEFAULT_ENV, NM_SBRSH_SETTINGS_GROUP, NM_SETTINGS_DEBUG, FALSE);

    /* UNFS defaults */
    nm_settings_set_boolean (self, NM_SETTINGS_DEFAULT_ENV, NM_UNFS_SETTINGS_GROUP, NM_SETTINGS_ENABLE, TRUE);
    nm_settings_set_value (self, NM_SETTINGS_DEFAULT_ENV, NM_UNFS_SETTINGS_GROUP, NM_UNFS_PORT, "2049");
    nm_settings_set_boolean (self, NM_SETTINGS_DEFAULT_ENV, NM_UNFS_SETTINGS_GROUP, NM_UNFS_SINGLEUSER, FALSE);
    
    /* VNC defaults */
    nm_settings_set_boolean (self, NM_SETTINGS_DEFAULT_ENV, NM_VNC_SETTINGS_GROUP, NM_SETTINGS_ENABLE, FALSE);
    nm_settings_set_value (self, NM_SETTINGS_DEFAULT_ENV, NM_VNC_SETTINGS_GROUP, NM_VNC_PORT, "5900");
    nm_settings_set_boolean (self, NM_SETTINGS_DEFAULT_ENV, NM_VNC_SETTINGS_GROUP, NM_VNC_LOOP, FALSE);
    
    /* Set environment defaults */
    // TODO fix it: create constants
    gchar *tmp[2];
    tmp[0] = strdup("Default");
    nm_settings_set_string_list(self, NULL, NM_SETTINGS_ENVIRONMENT_GROUP, NM_SETTINGS_ENVIRONMENT_LIST, tmp, 1);

    nm_settings_save (self);
    
    return TRUE;
}

