/*
 * Copyright (c) 2008 Nokia Corporation
 * Contact: integration@maemo.org
 *
 * Authors: Hallyson Melo <hallyson.melo@indt.org.br>
 *          Kyller Gorgonio <kyllercg@gmail.com>
 *          Walter Guerra <wguerra@ee.ufcg.edu.br>
 *          Andre Rodrigues <andre.rodrigues@signove.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */

#include <hildon/hildon.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <string.h>
#include <glib.h>


#include "nm_main_dialog.h"
#include "nm_conn_widget.h"
#include "nm_bt_setup.h"
#include "nm_bt_utils.h"
#include "nm_usb_setup.h"
#include "nm_usb_utils.h"
#include "nm_wlan_setup.h"
#include "nm_wlan_utils.h"
#include "nm_settings.h"
#include "nm_environment_manager.h"
#include "nm_sbrsh_setup.h"
#include "nm_unfs_setup.h"
#include "nm_vnc_setup.h"


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#define NM_CONN_TYPE_USB               0
#define NM_CONN_TYPE_BT                1
#define NM_CONN_TYPE_WLAN              2
#define NM_AVAILABLE_CONN_TYPES        3
gulong id_test = 0;

enum
{
   CONNECTION_NAME_COLUMN,
   CONNECTION_WIDGET_COLUMN,
   N_COLUMNS
};

static void nm_main_dialog_class_init (NmMainDialogClass *);
static void nm_main_dialog_init       (NmMainDialog *);
static void nm_main_dialog_dispose    (GObject *);
static void nm_main_dialog_finalize   (GObject *);

static void nm_main_dialog_create_tree_view (NmMainDialog *);
static void nm_main_dialog_add_widget (NmMainDialog *, GtkTreeView *, gchar *, NmConnWidget *,
                                        gboolean);
static void nm_main_dialog_run_apply (NmMainDialog *);
static void nm_main_dialog_run_save (NmMainDialog *);
static void nm_main_dialog_selection_changed (GtkTreeSelection *, gpointer);
static void nm_main_dialog_tool_selection_changed (GtkTreeSelection *, gpointer);
static NmConnWidget* nm_main_dialog_get_selected (NmMainDialog *);
static NmConnWidget* nm_main_dialog_get_tool_selected (NmMainDialog *);
static void nm_main_dialog_settings_changed (gpointer, gpointer);
static void nm_main_dialog_combobox_changed (gpointer, gpointer);
static void nm_main_dialog_environment_dialog (gpointer, gpointer);
static void nm_main_dialog_setup_combobox(NmMainDialog *);
static void nm_main_dialog_setup_basic (NmMainDialog *);
static void nm_main_dialog_setup_advanced (NmMainDialog *);
static void nm_main_dialog_setup_network_manager (NmMainDialog *);
static void nm_main_dialog_setup_tool_manager (NmMainDialog *);
static void nm_main_dialog_switch_to_advanced (NmMainDialog *);
static void nm_main_dialog_switch_to_basic (NmMainDialog *);
static gboolean nm_main_dialog_usb_apply (NmMainDialog *);
static gboolean nm_main_dialog_bt_apply (NmMainDialog *);
static gboolean nm_main_dialog_wlan_apply (NmMainDialog *);
static void nm_main_dialog_request_pin_code_cb(NmBtUtils *, gchar *, gpointer);
static void nm_main_dialog_device_paired_cb(NmBtUtils *, gboolean, gpointer);

typedef struct _NmMainDialogPrivate NmMainDialogPrivate;

struct _NmMainDialogPrivate
{
    gint current_mode;
    NmSettings *settings;
    gint basic_length;
    gint basic_height;
    gint basic_x;
    gint basic_y;
    gint selected_checkbox;
    gint caller;

    GtkWidget *treeview;

    GtkWidget *btsetup;
    GtkWidget *usbsetup;
    GtkWidget *wlansetup;

    GtkWidget *sbrshsetup;
    GtkWidget *unfssetup;
    GtkWidget *vncsetup;

    GtkWidget *ok_button;
    GtkWidget *apply_button;
    GtkWidget *cancel_button;
    GtkWidget *advanced_button;
    GtkWidget *basic_button;
    GtkWidget *reset_button;

    GtkWidget *basic_hbox;

    GtkWidget *advanced_vbox;
    GtkWidget *frame_tree;
    GtkWidget *advanced_combo_box;
    GtkWidget *advanced_combo_box_options_button;
    GtkWidget *advanced_combo_hbox;
    GtkWidget *advanced_notebook;
    GtkWidget *advanced_network_manager_tab;
    GtkWidget *advanced_tool_manager_tab;

    GtkWidget *check_basic_usb;
    GtkWidget *check_basic_bt;
    GtkWidget *check_basic_wlan;

    GtkWidget *tool_frame_tree;
    GtkWidget *tool_treeview;

    gchar *env;
    GHashTable *combobox_name_index;
    NmBtUtils *bt_utils;
};

#define NM_MAIN_DIALOG_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), NM_MAIN_DIALOG_TYPE, NmMainDialogPrivate))

G_DEFINE_TYPE (NmMainDialog, nm_main_dialog, GTK_TYPE_DIALOG)

static void
nm_main_dialog_class_init (NmMainDialogClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);

    g_type_class_add_private (klass, sizeof (NmMainDialogPrivate));

    object_class->dispose = nm_main_dialog_dispose;
    object_class->finalize = nm_main_dialog_finalize;
}

static void
nm_main_dialog_init (NmMainDialog *self)
{
    NmMainDialogPrivate *priv;

    priv = NM_MAIN_DIALOG_GET_PRIVATE (self);

    gtk_window_set_title (GTK_WINDOW(self), "PC-Connectivity Manager" );

    priv->bt_utils = nm_bt_utils_new ();
    g_signal_connect (priv->bt_utils, "request-pin-code",
						G_CALLBACK(nm_main_dialog_request_pin_code_cb), self);
    g_signal_connect (priv->bt_utils, "device-paired",
							G_CALLBACK(nm_main_dialog_device_paired_cb), self);

    priv->settings = nm_settings_new ();
    if (!nm_settings_start (priv->settings)) {

/*        hildon_banner_show_information(GTK_WIDGET (self), NULL,*/
/*                                        "Could not read config file!\n\*/
/*                                         Settings defaults...");*/

/*        g_usleep(1000000);*/
        nm_settings_set_defaults(priv->settings);
    }

    priv->env = nm_environment_manager_get_active();

    /* Put the combobox */
    nm_main_dialog_setup_combobox (self);
    g_debug("[%s] - combobox done", __FUNCTION__);

    /* Setup the modes */
    nm_main_dialog_setup_advanced (self);
    g_debug("[%s] - advanced done", __FUNCTION__);
    nm_main_dialog_setup_basic (self);
    g_debug("[%s] - basic done", __FUNCTION__);

    /* Add the buttons */


    priv->advanced_button = gtk_dialog_add_button (GTK_DIALOG (self),
                                                    _("Advanced"),
                                                    NM_RESPONSE_ADVANCED);

    priv->basic_button = gtk_dialog_add_button (GTK_DIALOG (self),
                                                 _("Basic"),
                                                 NM_RESPONSE_BASIC);


    priv->apply_button = gtk_dialog_add_button (GTK_DIALOG (self),
                                              _("Apply"),
                                              NM_RESPONSE_APPLY);

    priv->ok_button = gtk_dialog_add_button (GTK_DIALOG (self),
                                              _("Save"),
                                              GTK_RESPONSE_OK);




    gtk_button_set_focus_on_click(GTK_BUTTON(priv->advanced_button), FALSE);




/*    priv->cancel_button = gtk_dialog_add_button (GTK_DIALOG (self),*/
/*                                                  _("Close"),*/
/*                                                  GTK_RESPONSE_CLOSE);*/

    gtk_widget_set_sensitive (priv->basic_button, TRUE);

    // Set the default checkbox to 0
    priv->selected_checkbox = 0;

    gtk_widget_hide (GTK_WIDGET (priv->basic_button));
    gtk_widget_hide (GTK_WIDGET (priv->ok_button));

    gtk_window_get_size (GTK_WINDOW(self), &(priv->basic_length),
                          &(priv->basic_height));
    gtk_window_get_position (GTK_WINDOW(self), &(priv->basic_x),
                              &(priv->basic_y));
}

static void
nm_main_dialog_request_pin_code_cb(NmBtUtils *bt_utils, gchar *pin_code, gpointer user_data)
{
	gchar *text = g_strdup_printf("Please enter the following PIN code: %s",
								  pin_code);
	GtkWidget *note = hildon_note_new_information(GTK_WINDOW(user_data), text);

	gtk_widget_show_all(note);
	//gtk_widget_destroy (GTK_WIDGET (note));
	g_free(text);
}

static void
nm_main_dialog_device_paired_cb(NmBtUtils *bt_utils, gboolean paired, gpointer user_data)
{
    NmMainDialogPrivate *priv;
    gchar *btaddr = NULL;
    gchar *gw = NULL;

	priv = NM_MAIN_DIALOG_GET_PRIVATE (user_data);

	if (paired) {

        btaddr = nm_settings_get_value (priv->settings, priv->env, NM_BT_SETTINGS_GROUP,
                                         NM_SETTINGS_MAC);
        gw = nm_settings_get_value (priv->settings, priv->env, NM_BT_SETTINGS_GROUP,
                                     NM_SETTINGS_GATEWAY);

        if (strcmp (btaddr, NM_BT_DEFAULT_MAC) == 0) {

            hildon_banner_show_information (NULL, NULL,
                                             "Please, open PC-Connectivity Manager and\nconfigure bluetooth interface!");
            return;
        } else if (nm_bt_utils_check_pand (priv->bt_utils, btaddr) == NULL) {

            hildon_banner_show_information (NULL, NULL,
                                            "Unable to connect. Check if PAND is running on host!");
            return;
        }

		if (!nm_bt_utils_connect (bt_utils, btaddr, gw)) {

			hildon_banner_show_information (NULL, NULL,
											 "Could not connect to PC");
			return;
		}
		hildon_banner_show_information(NULL, NULL, "Bluetooth network enabled");
	}
	else {

		hildon_banner_show_information(NULL, NULL, "Could not pair PC");
	}

}

static void
nm_main_dialog_dispose (GObject *object)
{
    G_OBJECT_CLASS (nm_main_dialog_parent_class)->dispose (object);
}

static void
nm_main_dialog_finalize (GObject *object)
{
    NmMainDialogPrivate *priv = NM_MAIN_DIALOG_GET_PRIVATE (object);

    g_object_unref (priv->bt_utils);
    G_OBJECT_CLASS (nm_main_dialog_parent_class)->finalize (object);
}

GtkWidget*
nm_main_dialog_new ()
{
    return g_object_new (NM_MAIN_DIALOG_TYPE, NULL);
}

static void
nm_main_dialog_create_tree_view (NmMainDialog *self)
{
    GtkTreeStore *store;
    GtkCellRenderer *renderer;
    GtkTreeSelection *selection;
    GtkTreeViewColumn *column;
    NmMainDialogPrivate *priv;

    priv = NM_MAIN_DIALOG_GET_PRIVATE (self);

    store = gtk_tree_store_new (N_COLUMNS,
                                 G_TYPE_STRING,
                                 NM_CONN_WIDGET_TYPE);

    priv->treeview = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));

    /* Creates the render */
    renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes ("Connection", renderer,
                                                        "text",
                                                        CONNECTION_NAME_COLUMN,
                                                        NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (priv->treeview), column);

    /* Setup the selection handler */
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (priv->treeview));
    gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
    g_signal_connect (G_OBJECT (selection), "changed",
                       G_CALLBACK (nm_main_dialog_selection_changed),
                       self);
}

static void
nm_main_dialog_create_tool_tree_view (NmMainDialog *self)
{
    GtkTreeStore *store;
    GtkCellRenderer *renderer;
    GtkTreeSelection *selection;
    GtkTreeViewColumn *column;
    NmMainDialogPrivate *priv;

    priv = NM_MAIN_DIALOG_GET_PRIVATE (self);

    store = gtk_tree_store_new (N_COLUMNS,
                                 G_TYPE_STRING,
                                 NM_CONN_WIDGET_TYPE);

    priv->tool_treeview = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));

    /* Creates the render */
    renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes ("Connection", renderer,
                                                        "text",
                                                        CONNECTION_NAME_COLUMN,
                                                        NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (priv->tool_treeview), column);

    /* Setup the selection handler */
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (priv->tool_treeview));
    gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
    g_signal_connect (G_OBJECT (selection), "changed",
                       G_CALLBACK (nm_main_dialog_tool_selection_changed),
                       self);
}

static void
nm_main_dialog_add_widget (NmMainDialog *self, GtkTreeView *treeview, gchar *name,
                            NmConnWidget *widget, gboolean is_default)
{
    GtkTreeModel *model;
    NmMainDialogPrivate *priv;
    GtkTreeIter iter;

    priv = NM_MAIN_DIALOG_GET_PRIVATE (self);

    model = gtk_tree_view_get_model (GTK_TREE_VIEW (treeview));

    /* Append the network item */
    gtk_tree_store_append (GTK_TREE_STORE (model), &iter, NULL);
    gtk_tree_store_set (GTK_TREE_STORE (model), &iter,
                         CONNECTION_NAME_COLUMN, name,
                         CONNECTION_WIDGET_COLUMN, widget, -1);

    if (is_default) {

        gtk_tree_selection_select_iter
            (gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview)),
                                           &iter);
    }
}

void
nm_main_dialog_run (NmMainDialog *self, int caller)
{
    NmMainDialogPrivate *priv = NM_MAIN_DIALOG_GET_PRIVATE (self);
    gint ret;
    gboolean done = FALSE;

    /* Do not show the advanced button from the status menu*/
    g_debug("[%s] - caller: %d", __FUNCTION__, caller);
    if (caller == NM_STATUS_MENU) {
        gtk_widget_hide(priv->advanced_button);
    }

    while (!done) {

        ret = gtk_dialog_run (GTK_DIALOG (self));

        switch (ret) {

            case NM_RESPONSE_APPLY:

                nm_main_dialog_run_apply (self);
                break;

            case GTK_RESPONSE_OK:

                nm_main_dialog_run_save (self);
                done = TRUE;
                break;
            case NM_RESPONSE_ADVANCED:

                nm_main_dialog_switch_to_advanced (self);
                break;

            case NM_RESPONSE_BASIC:

                nm_main_dialog_switch_to_basic (self);
                gtk_window_resize (GTK_WINDOW(self), priv->basic_length,
                                    priv->basic_height);
                gtk_window_move (GTK_WINDOW(self), priv->basic_x,
                                  priv->basic_y);
                break;

            default:

                done = TRUE;
                break;
        }
    }
}

static void
nm_main_dialog_run_apply (NmMainDialog *self)
{
    NmMainDialogPrivate *priv = NM_MAIN_DIALOG_GET_PRIVATE (self);

    // Set the current combobox to the active env
    gchar *env = gtk_combo_box_get_active_text (GTK_COMBO_BOX(priv->advanced_combo_box));
    nm_environment_manager_set_active(env);

    nm_main_dialog_run_save(self);

    /* Apply network */
    if (!nm_usb_setup_apply_changes (NM_CONN_WIDGET(priv->usbsetup))) {

        hildon_banner_show_information(GTK_WIDGET (self), NULL,
                                        "Usb settings error!");
    }

    if (!nm_bt_setup_apply_changes (NM_CONN_WIDGET(priv->btsetup))) {

       hildon_banner_show_information(GTK_WIDGET (self), NULL,
                                       "Bluetooth settings error!");
    }

    if (!nm_wlan_setup_apply_changes (NM_CONN_WIDGET(priv->wlansetup))) {

        hildon_banner_show_information(GTK_WIDGET (self), NULL,
                                        "WLAN settings error!");
    }

    /* Apply tools */
    if (!nm_sbrsh_setup_apply_changes (NM_CONN_WIDGET(priv->sbrshsetup))) {

        hildon_banner_show_information(GTK_WIDGET (self), NULL,
                                        "SBRSH settings error!");
    }

    if (!nm_unfs_setup_apply_changes (NM_CONN_WIDGET(priv->unfssetup))) {

        hildon_banner_show_information(GTK_WIDGET (self), NULL,
                                        "UNFS settings error!");
    }

    if (!nm_vnc_setup_apply_changes (NM_CONN_WIDGET(priv->vncsetup))) {

        hildon_banner_show_information(GTK_WIDGET (self), NULL,
                                        "VNC settings error!");
    }

    gchar message[80];
    sprintf(message, "Environment %s applied successfully", env);
    g_debug("message: %s", message);
    hildon_banner_show_information(GTK_WIDGET (self), NULL, message);


    if (priv->current_mode == NM_BASIC_MODE) {

        if (!nm_main_dialog_usb_apply (self)) {

            hildon_banner_show_information(GTK_WIDGET (self), NULL,
                                            "USB settings error!");
        }

        if (!nm_main_dialog_bt_apply (self)) {

           hildon_banner_show_information(GTK_WIDGET (self), NULL,
                                           "Bluetooth settings error!");
        }

        if (!nm_main_dialog_wlan_apply (self)) {

            hildon_banner_show_information(GTK_WIDGET (self), NULL,
                                            "WLAN settings error!");
        }
    }
    g_free(env);
}

static void
nm_main_dialog_run_save (NmMainDialog *self)
{
    g_debug("[%s]", __FUNCTION__);

    NmMainDialogPrivate *priv = NM_MAIN_DIALOG_GET_PRIVATE (self);

    nm_usb_setup_save_changes(NM_USB_SETUP(priv->usbsetup));
    nm_bt_setup_save_changes(NM_BT_SETUP(priv->btsetup));
    nm_wlan_setup_save_changes(NM_WLAN_SETUP(priv->wlansetup));

    nm_sbrsh_setup_save_changes(NM_SBRSH_SETUP(priv->sbrshsetup));
    nm_unfs_setup_save_changes(NM_UNFS_SETUP(priv->unfssetup));
    nm_vnc_setup_save_changes(NM_VNC_SETUP(priv->vncsetup));
}

static void
nm_main_dialog_selection_changed (GtkTreeSelection *new, gpointer data)
{
    NmMainDialogPrivate *priv;
    NmConnWidget *conn_widget;

    priv = NM_MAIN_DIALOG_GET_PRIVATE (data);

    conn_widget = nm_main_dialog_get_selected (NM_MAIN_DIALOG (data));

    // FIXME this shown/hide panels need to be changed for a clever code
    if (conn_widget) {
        if (conn_widget == NM_CONN_WIDGET (priv->usbsetup)) {

            // hide BLUETOOTH and WLAN panels
            gtk_widget_hide (GTK_WIDGET (priv->btsetup));
            gtk_widget_hide (GTK_WIDGET (priv->wlansetup));

            // show USB panel
            gtk_widget_show (GTK_WIDGET (priv->usbsetup));

            // Set usb as the default panel
            priv->selected_checkbox = 0;
        } else if (conn_widget == NM_CONN_WIDGET (priv->btsetup)) {

            // hide USB and WLAN panels
            gtk_widget_hide (GTK_WIDGET (priv->usbsetup));
            gtk_widget_hide (GTK_WIDGET (priv->wlansetup));

            // show BLUETOOTH panel
            gtk_widget_show_all (GTK_WIDGET (priv->btsetup));

            // Set usb as the default panel
            priv->selected_checkbox = 1;
        } else if (conn_widget == NM_CONN_WIDGET (priv->wlansetup)) {

            // hide BLUETOOTH and USB panels
            gtk_widget_hide (GTK_WIDGET (priv->btsetup));
            gtk_widget_hide (GTK_WIDGET (priv->usbsetup));

            // show WLAN panel
            gtk_widget_show_all (GTK_WIDGET (priv->wlansetup));

            // Set wlan as the default panel
            priv->selected_checkbox = 2;
        }
        g_object_unref (conn_widget);
    }
}

static void
nm_main_dialog_tool_selection_changed (GtkTreeSelection *new, gpointer data)
{
    NmMainDialogPrivate *priv;
    NmConnWidget *conn_widget;

    priv = NM_MAIN_DIALOG_GET_PRIVATE (data);

    conn_widget = nm_main_dialog_get_tool_selected (NM_MAIN_DIALOG (data));

    // FIXME this shown/hide panels need to be changed for a clever code
    if (conn_widget) {
        if (conn_widget == NM_CONN_WIDGET (priv->sbrshsetup)) {

            // hide others
            gtk_widget_hide (GTK_WIDGET (priv->unfssetup));
            gtk_widget_hide (GTK_WIDGET (priv->vncsetup));

            // show sbrsh panel
            gtk_widget_show_all (GTK_WIDGET (priv->sbrshsetup));

        } else if (conn_widget == NM_CONN_WIDGET (priv->unfssetup)) {

            // hide others
            gtk_widget_hide (GTK_WIDGET (priv->sbrshsetup));
            gtk_widget_hide (GTK_WIDGET (priv->vncsetup));

            // show unfs panel
            gtk_widget_show_all (GTK_WIDGET (priv->unfssetup));

        } else if (conn_widget == NM_CONN_WIDGET (priv->vncsetup)) {

            // hide others
            gtk_widget_hide (GTK_WIDGET (priv->sbrshsetup));
            gtk_widget_hide (GTK_WIDGET (priv->unfssetup));

            // show vnc panel
            gtk_widget_show_all (GTK_WIDGET (priv->vncsetup));

        }

        g_object_unref (conn_widget);
    }
}

static NmConnWidget*
nm_main_dialog_get_selected (NmMainDialog *self)
{
    GtkTreeIter iter;
    GtkTreeModel *model;
    GtkTreeSelection *selection;
    NmConnWidget *conn_widget = NULL;
    NmMainDialogPrivate *priv;

    priv = NM_MAIN_DIALOG_GET_PRIVATE (self);

    if (priv->current_mode == NM_ADVANCED_MODE) {

        selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (priv->treeview));

        if (gtk_tree_selection_get_selected (selection, &model, &iter)) {

            gtk_tree_model_get (model, &iter, CONNECTION_WIDGET_COLUMN,
                                 &conn_widget, -1);
        }
    }

    return conn_widget;
}

static NmConnWidget*
nm_main_dialog_get_tool_selected (NmMainDialog *self)
{
    GtkTreeIter iter;
    GtkTreeModel *model;
    GtkTreeSelection *selection;
    NmConnWidget *conn_widget = NULL;
    NmMainDialogPrivate *priv;

    priv = NM_MAIN_DIALOG_GET_PRIVATE (self);

    if (priv->current_mode == NM_ADVANCED_MODE) {

        selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (priv->tool_treeview));

        if (gtk_tree_selection_get_selected (selection, &model, &iter)) {

            gtk_tree_model_get (model, &iter, CONNECTION_WIDGET_COLUMN,
                                 &conn_widget, -1);
        }
    }

    return conn_widget;
}

static void
nm_main_dialog_settings_changed (gpointer instance, gpointer data)
{
    NmMainDialogPrivate *priv;

    priv = NM_MAIN_DIALOG_GET_PRIVATE (NM_MAIN_DIALOG (data));

    gtk_widget_set_sensitive (priv->apply_button, TRUE);
}

static void
nm_main_dialog_combobox_changed (gpointer instance, gpointer self)
{
    NmMainDialogPrivate *priv;

    priv = NM_MAIN_DIALOG_GET_PRIVATE (NM_MAIN_DIALOG (self));

    nm_main_dialog_run_save(self);

    gchar *env = gtk_combo_box_get_active_text (instance);
    g_debug("Combobox changed: %s", env);
    g_free(priv->env);
    priv->env = g_strdup(env);
    g_free(env);

    // Remove the widgets
    g_signal_handlers_disconnect_by_func(priv->usbsetup, nm_main_dialog_settings_changed, NULL);
    g_signal_handlers_disconnect_by_func(priv->btsetup, nm_main_dialog_settings_changed, NULL);
    g_signal_handlers_disconnect_by_func(priv->wlansetup, nm_main_dialog_settings_changed, NULL);
    g_signal_handlers_disconnect_by_func(priv->treeview, nm_main_dialog_settings_changed, NULL);

    g_signal_handlers_disconnect_by_func(priv->sbrshsetup, nm_main_dialog_settings_changed, NULL);
    g_signal_handlers_disconnect_by_func(priv->unfssetup, nm_main_dialog_settings_changed, NULL);
    g_signal_handlers_disconnect_by_func(priv->vncsetup, nm_main_dialog_settings_changed, NULL);
    g_signal_handlers_disconnect_by_func(priv->tool_treeview, nm_main_dialog_settings_changed, NULL);

    gtk_container_remove(GTK_CONTAINER(priv->advanced_network_manager_tab), priv->usbsetup);
    gtk_container_remove(GTK_CONTAINER(priv->advanced_network_manager_tab), priv->btsetup);
    gtk_container_remove(GTK_CONTAINER(priv->advanced_network_manager_tab), priv->wlansetup);
    gtk_container_remove(GTK_CONTAINER(priv->frame_tree), priv->treeview);
    gtk_container_remove(GTK_CONTAINER(priv->advanced_network_manager_tab), priv->frame_tree);

    gtk_container_remove(GTK_CONTAINER(priv->advanced_tool_manager_tab), priv->sbrshsetup);
    gtk_container_remove(GTK_CONTAINER(priv->advanced_tool_manager_tab), priv->unfssetup);
    gtk_container_remove(GTK_CONTAINER(priv->advanced_tool_manager_tab), priv->vncsetup);
    gtk_container_remove(GTK_CONTAINER(priv->tool_frame_tree), priv->tool_treeview);
    gtk_container_remove(GTK_CONTAINER(priv->advanced_tool_manager_tab), priv->tool_frame_tree);

    g_debug("[%s] - env: %s", __FUNCTION__, priv->env);

    nm_main_dialog_setup_network_manager(self);
    nm_main_dialog_setup_tool_manager(self);


    gtk_widget_show_all(priv->frame_tree);
    gtk_widget_show(priv->usbsetup);
    gtk_widget_hide(priv->btsetup);
    gtk_widget_hide(priv->wlansetup);

    gtk_widget_show_all(priv->tool_frame_tree);
    gtk_widget_show_all(priv->unfssetup);
    gtk_widget_hide(priv->sbrshsetup);
    gtk_widget_hide(priv->vncsetup);

}

static void
nm_main_dialog_environment_dialog (gpointer instance, gpointer self)
{
    NmMainDialogPrivate *priv;

    priv = NM_MAIN_DIALOG_GET_PRIVATE (NM_MAIN_DIALOG (self));

    GtkWidget* environment_dialog = nm_environment_manager_new ();

    gchar *selected_env = nm_environment_manager_show_dialog(NM_ENVIRONMENT_MANAGER(environment_dialog));

/*    // Update the combobox*/
    g_signal_handler_disconnect(priv->advanced_combo_box, id_test);
    int i;
    gsize length;

    // TODO correct
    for (i = g_hash_table_size(priv->combobox_name_index); i >= 0; i--) {
        g_debug("[%s] - items to remove: %d", __FUNCTION__, i);
        gtk_combo_box_remove_text (GTK_COMBO_BOX(priv->advanced_combo_box), i);
    }

    gchar **list;
    g_hash_table_remove_all(priv->combobox_name_index);

    list = nm_environment_manager_get_list(&length);
    for (i = 0; i < length; i++) {
        g_hash_table_insert(priv->combobox_name_index, list[i], (gpointer) i);
        gtk_combo_box_insert_text (GTK_COMBO_BOX(priv->advanced_combo_box), i, list[i]);
    }

    /* Append the None env */
    g_hash_table_insert(priv->combobox_name_index, NM_ENVIRONMENT_MANAGER_NONE, (gpointer) length);
    gtk_combo_box_append_text(GTK_COMBO_BOX(priv->advanced_combo_box), NM_ENVIRONMENT_MANAGER_NONE );


    if (selected_env == NULL) {
        selected_env = nm_environment_manager_get_active();
    }

    id_test = g_signal_connect (G_OBJECT (priv->advanced_combo_box), "changed",
                      G_CALLBACK (nm_main_dialog_combobox_changed), self);

    g_debug("[%s] - Set the active item on the combobox: %s", __FUNCTION__, selected_env);
    gtk_combo_box_set_active (GTK_COMBO_BOX(priv->advanced_combo_box),
                              (gint) g_hash_table_lookup(priv->combobox_name_index, selected_env));
    // End Populating the combo box

    gtk_widget_destroy( environment_dialog );

}

static void
nm_main_dialog_setup_combobox(NmMainDialog *self)
{
    NmMainDialogPrivate *priv;

    priv = NM_MAIN_DIALOG_GET_PRIVATE (NM_MAIN_DIALOG (self));

    priv->advanced_combo_hbox = gtk_hbox_new (FALSE, 2);
    priv->advanced_combo_box_options_button = gtk_button_new_with_label ("...");
    g_signal_connect (G_OBJECT (priv->advanced_combo_box_options_button),
                       "clicked",
                       G_CALLBACK (nm_main_dialog_environment_dialog), self);


    priv->advanced_combo_box = gtk_combo_box_new_text ();

    // Populate the combo box
    gchar **list;
    int i;
    gsize length = 0;

    priv->combobox_name_index = g_hash_table_new (g_str_hash, g_str_equal);

    list = nm_environment_manager_get_list(&length);
    g_debug("length: %d", length);
    for (i = 0; i < length; i++) {
        g_debug("list[%d]: %s", i, list[i]);

        g_hash_table_insert(priv->combobox_name_index, list[i], (gpointer) i);
        gtk_combo_box_insert_text (GTK_COMBO_BOX(priv->advanced_combo_box), i, list[i]);
    }

    /* Append the None env*/
    g_hash_table_insert(priv->combobox_name_index, NM_ENVIRONMENT_MANAGER_NONE, (gpointer) length);
    gtk_combo_box_append_text(GTK_COMBO_BOX(priv->advanced_combo_box), NM_ENVIRONMENT_MANAGER_NONE );

    // End Populating the combo box

    gchar *active_env = nm_environment_manager_get_active();

    gtk_combo_box_set_active (GTK_COMBO_BOX(priv->advanced_combo_box),
                    GPOINTER_TO_INT(g_hash_table_lookup(priv->combobox_name_index, active_env)));



    id_test = g_signal_connect (G_OBJECT (priv->advanced_combo_box), "changed",
                       G_CALLBACK (nm_main_dialog_combobox_changed), self);

    GtkWidget *env_label = gtk_label_new("Environments: ");
    gtk_box_pack_start(GTK_BOX(priv->advanced_combo_hbox),  env_label,  FALSE, FALSE, 6);
    gtk_box_pack_start(GTK_BOX(priv->advanced_combo_hbox),  priv->advanced_combo_box,  TRUE, TRUE, 6);
    gtk_box_pack_start(GTK_BOX(priv->advanced_combo_hbox),  priv->advanced_combo_box_options_button,  FALSE, FALSE, 6);

    /* Add the advanced_vbox to the vbox */
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (self)->vbox), priv->advanced_combo_hbox,
                         FALSE, TRUE, 6);

    gtk_widget_show_all(priv->advanced_combo_hbox);
    gtk_widget_hide (GTK_WIDGET (priv->advanced_combo_box_options_button));
}

static void
nm_main_dialog_setup_basic (NmMainDialog *self)
{
    NmMainDialogPrivate *priv;

    priv = NM_MAIN_DIALOG_GET_PRIVATE (self);

    priv->basic_hbox = gtk_hbox_new (FALSE, 3);

    priv->check_basic_usb = gtk_check_button_new_with_label ("USB");
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(priv->check_basic_usb),
                                   usb_setup_check_enabled (NM_USB_SETUP(priv->usbsetup)));

    priv->check_basic_bt = gtk_check_button_new_with_label ("Bluetooth");
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(priv->check_basic_bt),
                                   bt_setup_check_enabled (NM_BT_SETUP(priv->btsetup)));

    priv->check_basic_wlan = gtk_check_button_new_with_label ("Wlan");
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(priv->check_basic_wlan),
                                   wlan_setup_check_enabled (NM_WLAN_SETUP(priv->wlansetup)));

    gtk_box_pack_start (GTK_BOX(priv->basic_hbox), priv->check_basic_usb, FALSE, TRUE, 1);
    gtk_box_pack_start (GTK_BOX(priv->basic_hbox), priv->check_basic_bt, FALSE, TRUE, 1);
    gtk_box_pack_start (GTK_BOX(priv->basic_hbox), priv->check_basic_wlan, FALSE, TRUE, 1);

    gtk_widget_show_all (priv->basic_hbox);

    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (self)->vbox), priv->basic_hbox,
                         TRUE, TRUE, 6);

    priv->current_mode = NM_BASIC_MODE;
}

static void
nm_main_dialog_setup_advanced (NmMainDialog *self)
{

    NmMainDialogPrivate *priv;

    priv = NM_MAIN_DIALOG_GET_PRIVATE (self);

    priv->advanced_vbox = gtk_vbox_new (FALSE, 2);


    // Set the notebook
    // TODO Uncomment the notebook to add it
    priv->advanced_notebook = gtk_notebook_new ();
    priv->advanced_network_manager_tab = gtk_hbox_new (FALSE, 3);
    priv->advanced_tool_manager_tab = gtk_hbox_new (FALSE, 3);

    nm_main_dialog_setup_network_manager(self);
    nm_main_dialog_setup_tool_manager(self);
    GtkWidget *network_label = gtk_label_new("Network Manager");
    gtk_notebook_append_page (GTK_NOTEBOOK(priv->advanced_notebook), priv->advanced_network_manager_tab, network_label);
    GtkWidget *tool_label = gtk_label_new("Tool Manager");
    gtk_notebook_append_page (GTK_NOTEBOOK(priv->advanced_notebook), priv->advanced_tool_manager_tab, tool_label);

    gtk_widget_show(priv->advanced_notebook);

    // Put the combobox and the notebook inside the advanced_vbox
    gtk_box_pack_start(GTK_BOX(priv->advanced_vbox),  priv->advanced_notebook,  FALSE, TRUE, 6);

    /* Add the advanced_vbox to the vbox */
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (self)->vbox), priv->advanced_vbox,
                         TRUE, TRUE, 6);
}

static void
nm_main_dialog_setup_network_manager (NmMainDialog *self)
{

    NmMainDialogPrivate *priv;

    priv = NM_MAIN_DIALOG_GET_PRIVATE (self);

    /* Creates the connection type area (frame) */
    priv->frame_tree = gtk_frame_new (" Connection type ");
    gtk_widget_set_size_request(priv->frame_tree, -1, 170);
    nm_main_dialog_create_tree_view (self);

    g_debug("[%s] - env: %s", __FUNCTION__, priv->env);

    priv->usbsetup = nm_usb_setup_new (priv->env);
    nm_main_dialog_add_widget (self, GTK_TREE_VIEW(priv->treeview), "USB",
                                NM_CONN_WIDGET (priv->usbsetup), strcmp(priv->env, NM_ENVIRONMENT_MANAGER_NONE));
    g_signal_connect (G_OBJECT (priv->usbsetup),
                       NM_CONN_WIDGET_SETTINGS_CHANGED,
                       G_CALLBACK (nm_main_dialog_settings_changed), self);

    nm_usb_setup_fill_widget(NM_USB_SETUP(priv->usbsetup));
    g_debug("[%s] - usb done", __FUNCTION__);

    priv->btsetup = nm_bt_setup_new (priv->env);
    nm_main_dialog_add_widget (self, GTK_TREE_VIEW(priv->treeview), "Bluetooth",
                                NM_CONN_WIDGET (priv->btsetup), FALSE);
    g_signal_connect (G_OBJECT (priv->btsetup),
                       NM_CONN_WIDGET_SETTINGS_CHANGED,
                       G_CALLBACK (nm_main_dialog_settings_changed), self);

    nm_bt_setup_fill_widget(NM_BT_SETUP(priv->btsetup));
    g_debug("[%s] - bt done", __FUNCTION__);

    priv->wlansetup = nm_wlan_setup_new (priv->env);
    nm_main_dialog_add_widget (self, GTK_TREE_VIEW(priv->treeview), "Wlan", NM_CONN_WIDGET (priv->wlansetup),
                                FALSE);
    g_signal_connect (G_OBJECT (priv->wlansetup),
                       NM_CONN_WIDGET_SETTINGS_CHANGED,
                       G_CALLBACK (nm_main_dialog_settings_changed), self);

    nm_wlan_setup_fill_widget(NM_WLAN_SETUP(priv->wlansetup));
    g_debug("[%s] - wlan done", __FUNCTION__);

    gtk_container_add (GTK_CONTAINER (priv->frame_tree), priv->treeview);
    gtk_box_pack_start (GTK_BOX (priv->advanced_network_manager_tab), priv->frame_tree,
                         TRUE, TRUE, 0);

    gtk_box_pack_start (GTK_BOX (priv->advanced_network_manager_tab), priv->usbsetup,
                         FALSE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (priv->advanced_network_manager_tab), priv->btsetup,
                         FALSE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (priv->advanced_network_manager_tab), priv->wlansetup,
                         FALSE, TRUE, 0);
    g_debug("[%s] - packing done", __FUNCTION__);

    gtk_widget_show_all(priv->frame_tree);
    gtk_widget_show(priv->advanced_network_manager_tab);
    gtk_widget_hide(priv->btsetup);
    gtk_widget_hide(priv->wlansetup);
    /* If the env is none, set insensitive */
    if (!strcmp(priv->env, NM_ENVIRONMENT_MANAGER_NONE)) {
        gtk_widget_set_sensitive (priv->advanced_network_manager_tab, FALSE);
    }
    else {
        gtk_widget_set_sensitive (priv->advanced_network_manager_tab, TRUE);
    }
    g_debug("[%s] - end", __FUNCTION__);
}

static void
nm_main_dialog_setup_tool_manager (NmMainDialog *self)
{
    NmMainDialogPrivate *priv;

    priv = NM_MAIN_DIALOG_GET_PRIVATE (self);

    /* Creates the connection type area (frame) */
    priv->tool_frame_tree = gtk_frame_new (" Tool ");
    gtk_widget_set_size_request(priv->tool_frame_tree, -1, 170);
    nm_main_dialog_create_tool_tree_view (self);

    priv->unfssetup = nm_unfs_setup_new (priv->env);
    nm_main_dialog_add_widget (self, GTK_TREE_VIEW(priv->tool_treeview), "NFS",
                                NM_CONN_WIDGET (priv->unfssetup), strcmp(priv->env, NM_ENVIRONMENT_MANAGER_NONE));
    g_signal_connect (G_OBJECT (priv->unfssetup),
                       NM_CONN_WIDGET_SETTINGS_CHANGED,
                       G_CALLBACK (nm_main_dialog_settings_changed), self);
    nm_unfs_setup_fill_widget(NM_UNFS_SETUP(priv->unfssetup));


    priv->vncsetup = nm_vnc_setup_new (priv->env);
    nm_main_dialog_add_widget (self, GTK_TREE_VIEW(priv->tool_treeview), "X11VNC",
                                NM_CONN_WIDGET (priv->vncsetup), FALSE);
    g_signal_connect (G_OBJECT (priv->vncsetup),
                       NM_CONN_WIDGET_SETTINGS_CHANGED,
                       G_CALLBACK (nm_main_dialog_settings_changed), self);
    nm_vnc_setup_fill_widget(NM_VNC_SETUP(priv->vncsetup));

    priv->sbrshsetup = nm_sbrsh_setup_new (priv->env);
    nm_main_dialog_add_widget (self, GTK_TREE_VIEW(priv->tool_treeview), "SBRSH",
                                NM_CONN_WIDGET (priv->sbrshsetup), FALSE);
    g_signal_connect (G_OBJECT (priv->sbrshsetup),
                       NM_CONN_WIDGET_SETTINGS_CHANGED,
                       G_CALLBACK (nm_main_dialog_settings_changed), self);
    nm_sbrsh_setup_fill_widget(NM_SBRSH_SETUP(priv->sbrshsetup));

    gtk_container_add (GTK_CONTAINER (priv->tool_frame_tree), priv->tool_treeview);
    gtk_box_pack_start (GTK_BOX (priv->advanced_tool_manager_tab), priv->tool_frame_tree,
                         TRUE, TRUE, 0);

    gtk_box_pack_start (GTK_BOX (priv->advanced_tool_manager_tab), priv->unfssetup,
                         FALSE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (priv->advanced_tool_manager_tab), priv->vncsetup,
                         FALSE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (priv->advanced_tool_manager_tab), priv->sbrshsetup,
                         FALSE, TRUE, 0);

    gtk_widget_show_all(priv->advanced_tool_manager_tab);
    gtk_widget_hide(priv->vncsetup);
    gtk_widget_hide(priv->sbrshsetup);
    /* If the env is none, set insensitive */
    if (!strcmp(priv->env, NM_ENVIRONMENT_MANAGER_NONE)) {
        gtk_widget_set_sensitive (priv->advanced_tool_manager_tab, FALSE);
    }
    else {
        gtk_widget_set_sensitive (priv->advanced_tool_manager_tab, TRUE);
    }

}

static void
nm_main_dialog_switch_to_advanced (NmMainDialog *self)
{
    NmMainDialogPrivate *priv;
    NmConnWidget *conn_widget;
    GtkTreeIter iter;
    GtkTreeModel *model;

    priv = NM_MAIN_DIALOG_GET_PRIVATE (self);

    priv->current_mode = NM_ADVANCED_MODE;

/*    conn_widget = nm_main_dialog_get_selected (self);*/

    // hide/show basic/advanced hboxes
    gtk_widget_show_all (GTK_WIDGET (priv->advanced_combo_box_options_button));
    gtk_widget_hide (GTK_WIDGET (priv->basic_hbox));
    gtk_widget_show (GTK_WIDGET (priv->advanced_vbox));

    // hide/show advanced/basic buttons
    gtk_widget_hide (GTK_WIDGET (priv->advanced_button));
    gtk_widget_show (GTK_WIDGET (priv->basic_button));
    gtk_widget_show (GTK_WIDGET (priv->ok_button));

    // check if has been enabled on basic mode
    if ( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(priv->check_basic_usb)) ) {
        nm_usb_setup_enable(NM_USB_SETUP(priv->usbsetup), NM_USB_NETWORK_LABEL);
    }
    else {
        NmSettings *settings = nm_settings_new ();
        nm_settings_start (settings);


        nm_usb_setup_enable(NM_USB_SETUP(priv->usbsetup),
                            nm_settings_get_value (settings, priv->env,
                                                   NM_USB_SETTINGS_GROUP,
                                                   NM_USB_SETTINGS_MODE));
    }
    if ( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(priv->check_basic_bt)) ) {
        nm_bt_setup_enable(NM_BT_SETUP(priv->btsetup), TRUE);
    }
    else {
        nm_bt_setup_enable(NM_BT_SETUP(priv->btsetup), FALSE);
    }

    if ( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(priv->check_basic_wlan)) ) {
        nm_wlan_setup_enable(NM_WLAN_SETUP(priv->wlansetup), TRUE);
    }
    else {
        nm_wlan_setup_enable(NM_WLAN_SETUP(priv->wlansetup), FALSE);
    }
    // TODO CHECK
//    gtk_widget_hide (GTK_WIDGET (priv->btsetup));
//   gtk_widget_hide (GTK_WIDGET (priv->wlansetup));

    g_debug("[%s]", __FUNCTION__);
//    gtk_widget_hide(GTK_WIDGET(priv->vncsetup));
//    gtk_widget_hide(GTK_WIDGET(priv->sbrshsetup));
}

static void
nm_main_dialog_switch_to_basic (NmMainDialog *self)
{
    NmMainDialogPrivate *priv = NM_MAIN_DIALOG_GET_PRIVATE (self);

    priv->current_mode = NM_BASIC_MODE;

    // hide/show advanced/basic hboxes
    gtk_widget_hide (GTK_WIDGET (priv->advanced_vbox));
    gtk_widget_show_all (GTK_WIDGET (priv->basic_hbox));

    // Set the status of the toggle buttons
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(priv->check_basic_usb),
                                 nm_usb_setup_mode_is_network(NM_USB_SETUP(priv->usbsetup)));

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(priv->check_basic_bt),
                                 nm_bt_setup_is_enabled(NM_BT_SETUP(priv->btsetup)));

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(priv->check_basic_wlan),
                                 nm_wlan_setup_is_enabled(NM_WLAN_SETUP(priv->wlansetup)));

    // Set the focus to the last selected
    if (priv->selected_checkbox == 0) {
        gtk_widget_grab_focus(GTK_WIDGET(priv->check_basic_usb));
    }
    else if (priv->selected_checkbox == 1) {
        gtk_widget_grab_focus(GTK_WIDGET(priv->check_basic_bt));
    }
    else if (priv->selected_checkbox == 2) {
        gtk_widget_grab_focus(GTK_WIDGET(priv->check_basic_wlan));
    }


    // hide/show basic/advanced buttons
    gtk_widget_hide (GTK_WIDGET (priv->basic_button));
    gtk_widget_show (GTK_WIDGET (priv->advanced_button));
    gtk_widget_hide (GTK_WIDGET (priv->ok_button));

}

static gboolean
nm_main_dialog_usb_apply (NmMainDialog *self)
{
    NmMainDialogPrivate *priv = NM_MAIN_DIALOG_GET_PRIVATE (self);

    gboolean is_loaded = nm_usb_utils_is_loaded ();
    gboolean is_toggled = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (priv->check_basic_usb));

    if (!is_loaded && is_toggled) {

        gchar *gw;
        priv->settings = nm_settings_new ();

        if (!nm_settings_start (priv->settings)) {

            hildon_banner_show_information(NULL, NULL,
                                            "Could not read config file!\n\
                                             Please, open PC-Connectivity Manager and cofigure it.");
            return FALSE;
        }

        gw = nm_settings_get_value (priv->settings, priv->env, NM_USB_SETTINGS_GROUP,
                                     NM_SETTINGS_GATEWAY);

        if (nm_usb_utils_load (gw)) {

            hildon_banner_show_information(NULL, NULL, "USB network enabled");
        } else {

            return FALSE;
        }
    } else if (is_loaded && !is_toggled) {

        if (nm_usb_utils_unload ()) {

            hildon_banner_show_information(NULL, NULL,
                                            "USB file storage enabled");
        } else {

            return FALSE;
        }
    }

    return TRUE;
}

static gboolean
nm_main_dialog_bt_apply (NmMainDialog *self)
{
    NmMainDialogPrivate *priv = NM_MAIN_DIALOG_GET_PRIVATE (self);

    gboolean has_net = (nm_bt_utils_has_network_connection (priv->bt_utils));
    gboolean is_toggled = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (priv->check_basic_bt));

    if (!has_net && is_toggled) {

        gchar *btaddr, *gw;
        priv->settings = nm_settings_new ();

        if (!nm_settings_start (priv->settings)) {

            hildon_banner_show_information(NULL, NULL,
                                            "Could not read config file!\n\
                                             Please, open PC-Connectivity Manager and cofigure it.");
            return FALSE;
        }


        if (!nm_bt_utils_radio_is_enabled (priv->bt_utils)) {

            gboolean res = nm_bt_utils_radio_enable (priv->bt_utils, TRUE);
            if (!res) {

                hildon_banner_show_information (NULL, NULL,
                                                 "Error enabling BT radio");
                return FALSE;
            }
        }

        btaddr = nm_settings_get_value (priv->settings, priv->env, NM_BT_SETTINGS_GROUP,
                                         NM_SETTINGS_MAC);
        gw = nm_settings_get_value (priv->settings, priv->env, NM_BT_SETTINGS_GROUP,
                                     NM_SETTINGS_GATEWAY);

        if (nm_bt_utils_is_paired(priv->bt_utils, btaddr)) {

            if (strcmp (btaddr, NM_BT_DEFAULT_MAC) == 0) {

                hildon_banner_show_information (NULL, NULL,
                                                 "Please, open PC-Connectivity Manager and\nconfigure bluetooth interface!");
                return FALSE;
            } else if (nm_bt_utils_check_pand (priv->bt_utils, btaddr) == NULL) {

                hildon_banner_show_information (NULL, NULL,
                                                "Unable to connect. Check if PAND is running on host!");
                return FALSE;
            }

			if (!nm_bt_utils_connect (priv->bt_utils, btaddr, gw)) {

				hildon_banner_show_information (NULL, NULL,
												"Could not connect to PC");
				return FALSE;
			}

			hildon_banner_show_information(NULL, NULL, "Bluetooth network enabled");
        }
        else {

        	if (!nm_bt_utils_pair_device(priv->bt_utils, btaddr)) {

				hildon_banner_show_information (NULL, NULL,
												 "Could not pair PC");
				return FALSE;
        	}
        }


    } else if (has_net && !is_toggled) {

        nm_bt_utils_remove_connection (priv->bt_utils);

        hildon_banner_show_information(NULL, NULL,
                                        "Bluetooth network disabled");
    }

    return TRUE;
}

static gboolean
nm_main_dialog_wlan_apply (NmMainDialog *self)
{
    NmMainDialogPrivate *priv = NM_MAIN_DIALOG_GET_PRIVATE (self);

    gboolean is_loaded =nm_wlan_utils_is_active ();
    gboolean is_toggled = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (priv->check_basic_wlan));

    if (!is_loaded && is_toggled) {

        if (nm_wlan_utils_activate ()) {

            hildon_banner_show_information(NULL, NULL,
                                            "WLAN Ad-Hoc network enabled");
        } else {

            return FALSE;
        }
    } else if (is_loaded && !is_toggled) {

        if (nm_wlan_utils_deactivate ()) {

            hildon_banner_show_information(NULL, NULL,
                                            "WLAN Ad-Hoc network disabled");

        } else {

            return FALSE;
        }
    }

    return TRUE;
}
