#!/bin/sh
#
# Copyright (C) 2007 Nokia Corporation. All rights reserved.
#
# Contact: Jukka Rissanen <jukka.rissanen@nokia.com>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# version 2 as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
# 02110-1301 USA

DEBUG=0
DISABLE=0
IAP=""
USE_CURRENT_IAP=0
MODULES_ONLY=""
AUTOCONF=0
ONLY=0
CLEAR_INTERFACES=""
IAP_BASE=/system/osso/connectivity/IAP
IPV6_KEY=$IAP_BASE/ipv6_enabled
AUTOCONF_KEY=$IAP_BASE/ipv6_autoconf
MODULES="
  xfrm6_mode_transport
  xfrm6_mode_tunnel
  xfrm6_mode_ro
  xfrm6_tunnel
  ip6_tunnel
  tunnel6
  ip6table_filter
  ip6_tables
  ipv6
"

usage()
{
cat <<EOF
Usage: $0 [options]
    where options can be:
    --disable (disable IPv6)
    --d (activate debugging)
    --iap <IAP name to activate IPv6>
    --current (activate IPv6 for the current IAP)
    --modules (unload only modules)
    --interface (clear interfaces setting)
    --autoconf, disable auto-configuring
    --default, set all IPv6 settings to factory defaults
EOF
}

if [ $# -lt 1 ]; then
	# Show the current status
	osso-ipv6-enable.sh
	exit
fi

while
        [ $# -gt 0 ]
do
        case $1 in
	    -di|-disable|--disable) DISABLE=1; shift;;
	    -d|-debug|--debug) DEBUG=1; shift;;
	    -i|-iap|--iap) IAP="$2"; shift 2;;
	    -c|-current|--current) IAP=current; USE_CURRENT_IAP=1; shift;;
	    -m|-modules|--modules) MODULES_ONLY=yes; shift;;
	    -o|-only|--only) ONLY=1; shift;;
	    -f|-interface|--interface|-interfaces|--interfaces) CLEAR_INTERFACES=1; shift;;
	    -a|-auto|-autoconf|--autoconf) AUTOCONF=1; shift;;
	    -default|--default|-defaults|--defaults) osso-ipv6-enable.sh --default; exit;;
            *) usage; exit 1;;
        esac
done


check_module()
{
    MOD="$1"

    /sbin/lsmod | grep "$MOD" > /dev/null
    if [ $? != 0 ]; then
	[ $DEBUG -ne 0 ] && echo "Module $MOD not yet loaded"
	RET=0
    else
	[ $DEBUG -ne 0 ] && echo "Module $MOD already loaded"
	RET=1
    fi

    return $RET
}


unload_module()
{
    MOD="$1"
    FAILED=0

    check_module $MOD
    if [ $? -eq 1 ]; then
	/sbin/rmmod $MODULE_DIR/$MOD.ko
	if [ $? -ne 0 ]; then
	    FAILED=1
	fi
	[ $DEBUG -ne 0 ] && echo "rmmod $MOD returned $RET"
    fi

    return $FAILURE
}


FAILURE=0

if [ "$MODULES_ONLY" = yes ]; then
    UNLOADED=""
    for mod in $MODULES
      do
      unload_module $mod
      if [ $? -eq 1 ]; then
	  echo "Cannot unload $mod"
	  FAILURE=1
      else
	  UNLOADED="$UNLOADED$mod "
      fi
    done

    if [ "$UNLOADED" != "" ]; then
	echo "Unloaded modules: $UNLOADED"
    fi
    if [ $ONLY -eq 1 ]; then
	exit
    fi
fi


# Clear interfaces
if [ "$CLEAR_INTERFACES" = "1" ]; then
    IFACES=`gconftool-2 -g "$IAP_BASE/ipv6_interfaces"`
    IFACES=`echo $IFACES | sed -e 's/\[//g' -e 's/\]//g' -e 's/\,/ /g'`
    [ $DEBUG -ne 0 ] && echo "Clearing interfaces: $IFACES"
    gconftool-2 -u "$IAP_BASE/ipv6_interfaces"
fi



# Disable IPv6 for this IAP
if [ ! -z "$IAP" ]; then
    if [ "$USE_CURRENT_IAP" = 1 ]; then
	CURRENT=`gconftool-2 -g "$IAP_BASE/current"`
	ESCAPED=`gconf_escape "$CURRENT"`
    else
	ESCAPED=`gconf_escape "$IAP"`
    fi

    gconftool-2 -g "$IAP_BASE/$ESCAPED/type" 2>&1 | grep "No value" > /dev/null
    if [ $? -eq 0 ]; then
	# No such escaped IAP found
	echo "Cannot find $IAP ($ESCAPED) in gconf"
	FAILURE=1
    else
	IAP_IPV6_KEY="$IAP_BASE/$ESCAPED/ipv6_enabled"
	gconftool-2 -t int -s "$IAP_IPV6_KEY" 0
	if [ $? -ne 0 ]; then
	    echo "Cannot enable IPv6 for $ESCAPED in gconf"
	    FAILURE=1
	else
	    [ $DEBUG -ne 0 ] && echo "Set $IAP_IPV6_KEY to 0"
	fi
    fi
fi


if [ "$AUTOCONF" = 1 ]; then
    # Turn off ipv6 autoconf globally
    gconftool-2 -t int -s "$AUTOCONF_KEY" 0
    if [ $? -ne 0 ]; then
	echo "Cannot disable IPv6 autoconf in gconf"
	FAILURE=1
    else
	[ $DEBUG -ne 0 ] && echo "Set $AUTOCONF_KEY to 0"
    fi
fi


if [ "$DISABLE" = 1 ]; then
    # Turn off ipv6 globally
    gconftool-2 -t int -s "$IPV6_KEY" 0
    if [ $? -ne 0 ]; then
	echo "Cannot disable IPv6 in gconf"
	FAILURE=1
    else
	[ $DEBUG -ne 0 ] && echo "Set $IPV6_KEY to 0"
    fi
fi


exit $FAILURE


