public class Bookmark: Object
{
    public VerseSet verse_set;
    public string description;

    public Bookmark(VerseSet verse_set, string description)
    {
        this.verse_set = verse_set;
        this.description = description;
    }
}

public class Bookmarks
{
    public List<Bookmark> _bookmarks;
    public List<unowned Bookmark> bookmarks { get { return _bookmarks; } }

    public string filename;


    public Bookmarks(string filename)
    {
        this.filename = filename;

        var key_file = new KeyFile();

        try
        {
            key_file.load_from_file(filename, KeyFileFlags.NONE);

            foreach (var group in key_file.get_groups())
            {
                var description = key_file.get_string(group, "description");
                var book = Books.instance.book_by_code(key_file.get_string(group, "book"));
                var chapter = key_file.get_integer(group, "chapter");

                int[] verses;

                try
                {
                    verses = key_file.get_integer_list(group, "verses");
                }
                catch (KeyFileError e)
                {
                    verses = {};
                }

                _bookmarks.append(new Bookmark(
                    VerseSet(
                        Chapter(
                            book,
                            chapter
                        ),
                        verses
                    ),
                    description
                ));
            }
        }
        catch (KeyFileError e)
        {
            stderr.printf("Bookmarks: KeyFileError: %s\n", e.message);
        }
        catch (FileError e)
        {
            stderr.printf("Bookmarks: FileError: %s\n", e.message);
        }
    }

    ~Bookmarks()
    {
        save();
    }

    public void append(Bookmark bookmark)
    {
        _bookmarks.append(bookmark);
    }

    public void remove(Bookmark bookmark)
    {
        _bookmarks.remove(bookmark);
    }


    public void save()
    {
        var key_file = new KeyFile();

        var number = 1;
        foreach (var bookmark in _bookmarks)
        {
            var group = "%d".printf(number);
            key_file.set_string(group, "book", bookmark.verse_set.chapter.book.code);
            key_file.set_integer(group, "chapter", bookmark.verse_set.chapter.chapter);
            key_file.set_integer_list(group, "verses", bookmark.verse_set.verses);
            key_file.set_string(group, "description", bookmark.description);

            number++;
        }

        try
        {
            FileUtils.set_contents(
                filename,
                key_file.to_data()
            );
        }
        catch (FileError e)
        {
            stderr.printf("Bookmarks: FileError: %s\n", e.message);
        }
    }
}



public class BookmarksWindow: Hildon.StackableWindow
{
    Gtk.ListStore store;

    Hildon.EditToolbar edit_toolbar;

    Gtk.TreeView view;


    Bookmarks bookmarks;


    public signal void bookmark_selected(Bookmark bookmark);


    public BookmarksWindow()
    {
        title = "Bookmarks";

        store = new Gtk.ListStore(3, typeof(string), typeof(string), typeof(Bookmark));

//        view = new Gtk.TreeView.with_model(store);
        view = (Gtk.TreeView)Hildon.gtk_tree_view_new_with_model(Hildon.UIMode.NORMAL, store);

        var location_renderer = new Gtk.CellRendererText();
        // location_renderer.style = Pango.Style.ITALIC;
        location_renderer.weight = 800;
        view.insert_column_with_attributes(
            -1, "Location",
            location_renderer,
            "text", 0
        );
        view.row_activated.connect(on_row_activated);

        var description_renderer = new Gtk.CellRendererText();
        description_renderer.ellipsize = Pango.EllipsizeMode.END;
        view.insert_column_with_attributes(
            -1, "Description",
            description_renderer,
            "text", 1
        );

        var scroll = new Hildon.PannableArea();
        scroll.add(view);

        add(scroll);

        scroll.show_all();


        var app_menu = new Hildon.AppMenu();

        var delete_button = new Hildon.Button.with_text(
            Hildon.SizeType.AUTO,
            Hildon.ButtonArrangement.HORIZONTAL,
            "Delete",
            ""
        );
        delete_button.clicked.connect(on_delete_button_clicked);
        app_menu.append(delete_button);

        app_menu.show_all();
        set_app_menu(app_menu);


        edit_toolbar = new Hildon.EditToolbar.with_text("Select bookmarks to delete", "Delete");
        edit_toolbar.button_clicked.connect(on_edit_toolbar_button_clicked);
        edit_toolbar.arrow_clicked.connect(on_edit_toolbar_arrow_clicked);
        set_edit_toolbar(edit_toolbar);
    }


    public void fill(Bookmarks bookmarks)
    {
        this.bookmarks = bookmarks;

        store.clear();

        Gtk.TreeIter iter;
        foreach (var bookmark in bookmarks.bookmarks)
        {
            store.append(out iter);
            store.set(iter,
                0, bookmark.verse_set.to_string(),
                1, bookmark.description,
                2, bookmark
            );
        }
    }


    private void on_row_activated(Gtk.TreePath path, Gtk.TreeViewColumn column)
    {
        Gtk.TreeIter iter;
        store.get_iter(out iter, path);

        unowned Bookmark bookmark;
        store.@get(iter, 2, out bookmark);

        bookmark_selected(bookmark);
    }


    void on_delete_button_clicked()
    {
        view.row_activated.disconnect(on_row_activated);

        Hildon.gtk_tree_view_set_ui_mode(view, Hildon.UIMode.EDIT);
        view.get_selection().set_mode(Gtk.SelectionMode.MULTIPLE);
        view.get_selection().unselect_all();

        edit_toolbar.show();
        fullscreen();
    }


    void on_edit_toolbar_button_clicked()
    {
        view.get_selection().selected_foreach((model, path, iter) => {
            unowned Bookmark bookmark;
            store.@get(iter, 2, out bookmark);

            bookmarks.remove(bookmark);
        });

        fill(bookmarks);

        on_edit_toolbar_arrow_clicked();
    }

    void on_edit_toolbar_arrow_clicked()
    {
        view.row_activated.connect(on_row_activated);

        Hildon.gtk_tree_view_set_ui_mode(view, Hildon.UIMode.NORMAL);
        view.get_selection().set_mode(Gtk.SelectionMode.SINGLE);

        edit_toolbar.hide();
        unfullscreen();
    }
}
