import sys, logging

log = logging.getLogger("nqaap.Gui")

import gtk, gobject
import hildon

class Gui:
    # Set up:
    def __init__(self, cont = None):
        log.info("Starting GUI")
        # fields
        self.controller = cont
        self.sleep_timer = None
        self.auto_chapter_selected = False # true if we are in the
                                           # midle of an automatic
                                           # chapter change

        # set up gui
        self.setup()

    def start(self):
        gtk.gdk.threads_init()
        gtk.main()


    def setup(self):
 
        gtk.set_application_name("nQa Audiobook Player") # window title
        self.win = hildon.StackableWindow()
   
        #Box that divides the layout in two: cover on the lefta
        #and controls on the right
        self.hbox = gtk.HBox(True,0)
        
        # Cover image
        self.cover = gtk.Image()
        # cover.set_from_image() 

        self.hbox.pack_start(self.cover,True,True,0) 

        # Controls:

        # Label that hold the title of the book,and maybe the chapter
        self.title = gtk.Label("Select a book to start listening")
        self.title.set_justify(gtk.JUSTIFY_CENTER)
            
        # Seekbar 
        self.seek = hildon.GtkHScale()
        self.seek.set_range(0.0,1.0)
        self.seek.set_draw_value(False)
        self.seek.set_update_policy(gtk.UPDATE_DISCONTINUOUS) 
        self.seek.connect('change-value',self.seek_changed) # event
        # self.seek.connect('value-changed',self.seek_changed) # event

        # Pause button
        self.button = hildon.Button(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL)
        self.button.set_text("Play", "Start playing the audiobook")
        self.button.connect('clicked', self.play_pressed) # event

        # Box to hold the controls:
        self.vbox = gtk.VBox(False, 5)

        # Add items to vbox
        self.vbox.pack_start(self.title, True, True, 0)
        self.vbox.pack_start(self.seek, True, True, 0)
        self.vbox.pack_start(self.button, False, False, 0)

        # Add vbox to hbox
        self.hbox.add(self.vbox)

        # Add hbox to the window
        self.win.add(self.hbox)

        #Menu:
        # Create menu
        self.menu = self.create_menu()
        self.win.set_app_menu(self.menu) # Attach menu to the window

        self.win.connect("delete_event", self.quit) # Add shutdown event

        self.win.show_all()

        # Run update timer
        self.setup_timers()
        
    def create_menu(self): 

        menu = hildon.AppMenu()         # create menu

        # Select Book button
        self.book_selector = hildon.TouchSelector(text = True)
        self.book_selector.connect("changed", self.book_selected) # changed event
        # Create a picker button 
        self.book_button = hildon.PickerButton(gtk.HILDON_SIZE_AUTO,
                                          hildon.BUTTON_ARRANGEMENT_VERTICAL)
        self.book_button.set_title("Select an audiobook") # Set a title to the button 
        self.book_button.set_selector(self.book_selector) # Attach the selector to the button
        #self.set_books(self.controller.get_books()) 
        menu.append(self.book_button)        # Add the button to menu
    
        # Select chapter button
        self.chapter_selector = hildon.TouchSelector(text = True)
        self.chapter_selector.connect("changed", self.chapter_selected) # changed event
        # Create a picker button 
        self.chapter_button = hildon.PickerButton(gtk.HILDON_SIZE_AUTO,
                                          hildon.BUTTON_ARRANGEMENT_VERTICAL)
        self.chapter_button.set_title("Select a chapter") # Set a title to the button 
        self.chapter_button.set_selector(self.chapter_selector) # Attach the selector to the button
        menu.append(self.chapter_button)        # Add the button to menu

        
        # Sleep timer button
        self.sleeptime_selector = hildon.TouchSelector(text = True) 
        self.sleeptime_selector.connect("changed", self.sleeptimer_set) # changed event
        for s in ["Cancel","1","10","20","30","60"]:
            self.sleeptime_selector.append_text(s)
        
        # Create a picker button 
        self.sleeptime_button = hildon.PickerButton(gtk.HILDON_SIZE_AUTO,
                                          hildon.BUTTON_ARRANGEMENT_VERTICAL)
        self.sleeptime_button.set_title("Sleeptimer") # Set a title to the button 
        self.sleeptime_button.set_selector(self.sleeptime_selector) # Attach the selector to the button
        menu.append(self.sleeptime_button)        # Add the button to menu
        
        menu.show_all()
        return menu

    def setup_timers(self):
        self.seek_timer = gobject.timeout_add_seconds(3, self.update_seek)

    def update_seek(self):
        #print self.controller.get_percentage()
        if self.controller.is_playing():
            gtk.gdk.threads_enter()
            self.seek.set_value(self.controller.get_percentage())
            gtk.gdk.threads_leave()
        #self.controller.get_percentage() 
        return True                     # run again

    def sleep(self):
        log.info("sleep time timeout")
        hildon.hildon_banner_show_information(self.win, "", "Sleep timer")
        self.controller.stop()
        button.set_text("Resume","Resume playing the audiobook")
        return False                    # do not repeat
        
    # Events:

    def seek_changed(self, seek, scroll ,value ): 
        # print "sok", scroll
        self.controller.seek_percent(seek.get_value())
        
    def play_pressed(self,button):
        #self.win.show_information("Test banner")
        if self.controller.is_playing():
            button.set_text("Resume","Resume playing the audiobook")
            self.controller.stop()
        else:
            button.set_text("Stop","Stop playing the audiobook")
            self.controller.play()
            
    def quit(self, a1, a2):             # what are the arguments?
        self.controller.stop()          # to save the state
        gtk.main_quit()
        log.info("Shutting down")

    def book_selected(self, selector, selected):
        self.controller.set_book(selector.get_current_text())

    def chapter_selected(self, selector, selected):
        log.debug("chapter selected: " + str(selected) +" : "+ str(selector.get_current_text()))
        
        if selector.get_current_text() is None:
            log.debug("chapter selection canceled.")
            #import pdb; pdb.set_trace()     # start debugger
            return True                   # this should end the function and indicate it has been handled
        if self.auto_chapter_selected:
            log.debug("chapter changed (by controller) to: " + selector.get_current_text())
            self.auto_chapter_selected = False
            # do nothing
        else:
            log.debug("chapter selection sendt to controller: " + selector.get_current_text())
            self.controller.set_chapter(selector.get_current_text()) # signal controller

    def sleeptimer_set(self, selector, selected):
        if self.sleep_timer is not None:
            gobject.source_remove(self.sleep_timer)
        value = selector.get_current_text()
        if value != "Cancel":
            time_out = int(value) 
            gobject.timeout_add_seconds(time_out * 60, self.sleep)
        else:
            hildon.hildon_banner_show_information(self.win, "", "Sleep timer canceled") 
        
    # Actions:  

    def set_books(self,books):
        log.debug("new books")
        #self.book_selector.clear()
        for b in books:
            self.book_selector.append_text(b)
        if len(books)==0:
            hildon.hildon_banner_show_information(self.win, "", "No audiobooks found. \nPlease place your audiobooks in the directory ~/MyDocs/Audiobooks")

            
    def set_book(self,book,cover):
        self.button.set_text("Play", "Start playing the audiobook") # reset button
        self.title.set_text(book)
        self.book_button.set_value(book)
        if cover != "":
            self.cover.set_from_file(cover) 

    def set_chapter(self,chapter):
        '''
        Called from controller whenever a new chapter is started

        chapter parameter is supposed to be the index for the chapter, not the name
        '''
        #self.button.set_text("Play", "Start playing the audiobook") # reset button
        #self.chapter_button.set_value(chapter)
        # self.set_selected_value(self.chapter_button,chapter)
        self.auto_chapter_selected = True 
        self.chapter_button.set_active(chapter)
        
    
    def set_chapters(self, chapters):
        m = self.chapter_selector.get_model(0)
        m.clear()
        #m.set_column_types(gobject.TYPE_STRING)
        log.debug("setting chapters" )
        for c in chapters:
            m.append([c])

    def set_sleep_timer(self, mins):
        pass

    # Utils
    def set_selected_value(self,button, value):
        i = button.get_selector().get_model(0).index[value] # get index of value from list
        button.set_active(i)                                # set active index to that index
   
if __name__ == "__main__":
    g = Gui(None)
