/*
 *  Copyright (C) 2011  Neverball authors
 *
 *  This  program is  free software;  you can  redistribute  it and/or
 *  modify it  under the  terms of the  GNU General Public  License as
 *  published by the Free Software Foundation; either version 2 of the
 *  License, or (at your option) any later version.
 *
 *  This program  is distributed in the  hope that it  will be useful,
 *  but  WITHOUT ANY WARRANTY;  without even  the implied  warranty of
 *  MERCHANTABILITY or FITNESS FOR  A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a  copy of the GNU General Public License
 *  along  with this  program;  if  not, write  to  the Free  Software
 *  Foundation,  Inc.,   59  Temple  Place,  Suite   330,  Boston,  MA
 *  02111-1307 USA
 */

#include "maemo.h"
#include <SDL/SDL_timer.h>
#include <libosso.h>

static osso_context_t *osso_context = NULL;
static SDL_TimerID screen_timer_id = 0;

static Uint32 disable_display_blank(Uint32 interval, void *param)
{
    int ret;
    ret = osso_display_blanking_pause(osso_context);
    return interval;
}

/* Fast FPU mode is a bit inaccurate (non IEEE), but definitely faster */
static void fpu_set_fast_mode(int fast_mode)
{
#ifdef __arm__
  if (fast_mode)
  {
      int tmp;
      __asm__ volatile(
        "fmrx       %[tmp], fpscr\n"
        "orr        %[tmp], %[tmp], #(1 << 24)\n" /* flush-to-zero */
        "orr        %[tmp], %[tmp], #(1 << 25)\n" /* default NaN */
        "bic        %[tmp], %[tmp], #((1 << 15) | (1 << 12) | (1 << 11) | (1 << 10) | (1 << 9) | (1 << 8))\n" /* clear exception bits */
        "fmxr       fpscr, %[tmp]\n"
        : [tmp] "=r" (tmp)
      );
  }
  else
  {
      int tmp;
      __asm__ volatile(
        "fmrx       %[tmp], fpscr\n"
        "bic        %[tmp], %[tmp], #(1 << 24)\n" /* flush-to-zero */
        "bic        %[tmp], %[tmp], #(1 << 25)\n" /* default NaN */
        "fmxr       fpscr, %[tmp]\n"
        : [tmp] "=r" (tmp)
      );
  }
#endif /* if __arm__ */
}

int maemo_init(const char *program)
{
    osso_context = osso_initialize(program, "1.0", 0, NULL);
    if (!osso_context) {
        fprintf(stderr, "osso_initialize failed!\n");
        return 0;
    }

    osso_display_blanking_pause(osso_context);
    screen_timer_id = SDL_AddTimer(30000, disable_display_blank, NULL);

    fpu_set_fast_mode(1);
    return 1;
}

void maemo_quit(void)
{
    if (osso_context)
        osso_deinitialize(osso_context);

    SDL_RemoveTimer(screen_timer_id);
}

