/*
 * Copyright (c) 2006, Jakub Pavelek <jpavelek@welho.com>
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 *   notice and this list of conditions
 * * Neither the name of Jakub Pavelek nor the names of contributors 
 *   may be used to endorse or promote products derived from this 
 *   software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <SDL/SDL.h>
#include <SDL/SDL_image.h>
#include <glib.h>
#include <glib/gstdio.h>

#define NAKO_DATADIR "/usr/games/nako/"
#define DEFAULT_TILESET "2004.tileset"
#define DEFAULT_UI "default.ui"

#define TILES_EASY 15
#define TILES_NORMAL 18
#define TILES_CHALLENGING 21
#define TILES_MAX TILES_CHALLENGING


/* Locations of graphic UI elements in the UI file */
#define GRAB_CLOSE_BUTTON_X 730
#define GRAB_CLOSE_BUTTON_Y 11
#define GRAB_CLOSE_BUTTON_X_DOWN 0
#define GRAB_CLOSE_BUTTON_Y_DOWN 480

#define GRAB_SHUFFLE_BUTTON_X 730
#define GRAB_SHUFFLE_BUTTON_Y 82
#define GRAB_SHUFFLE_BUTTON_X_DOWN 60
#define GRAB_SHUFFLE_BUTTON_Y_DOWN 480

#define GRAB_UP_X 269
#define GRAB_UP_Y 530
#define GRAB_UP_X_DOWN 405
#define GRAB_UP_Y_DOWN 530
#define GRAB_UP_X_DIMMED 541
#define GRAB_UP_Y_DIMMED 530

#define GRAB_DOWN_X 337
#define GRAB_DOWN_Y 530
#define GRAB_DOWN_X_DOWN 473
#define GRAB_DOWN_Y_DOWN 530
#define GRAB_DOWN_X_DIMMED 609
#define GRAB_DOWN_Y_DIMMED 530

#define GRAB_NUMBERS_X 266
#define GRAB_NUMBERS_Y 485
#define GRAB_TILEBACK_X 161
#define GRAB_TILEBACK_Y 480

#define GRAB_DIF_W 61
#define GRAB_DIF_H 46
#define GRAB_DIF_Y 480
#define GRAB_DIF_CHALLENGING_X 422
#define GRAB_DIF_CHALLENGING_X_DIMMED 483
#define GRAB_DIF_NORMAL_X 544
#define GRAB_DIF_NORMAL_X_DIMMED 605
#define GRAB_DIF_EASY_X 666
#define GRAB_DIF_EASY_X_DIMMED 727

/* Blitting locations */

#define TILE_GFX_HEIGHT 70
#define TILE_GFX_WIDTH 100

#define COUNTER_X_START 737
#define COUNTER_Y_START 430
#define COUNTER_TEXT_HEIGHT 27
#define COUNTER_TEXT_WIDTH 15
#define UPDOWN_BUTTON_X 730
#define UPDOWN_BUTTON_W 60
#define UPDOWN_BUTTON_H 25
#define UP_BUTTON_Y 320
#define DOWN_BUTTON_Y 384

#define THUMBNAIL_X 735
#define THUMBNAIL_Y 347
#define DIFFICULTY_X 729
#define DIFFICULTY_CHALLENGING_Y 168
#define DIFFICULTY_NORMAL_Y 214
#define DIFFICULTY_EASY_Y 260

#define BIGBUTTON_X 729
#define BIGBUTTON_W 60
#define BIGBUTTON_H 70
#define CLOSE_BUTTON_Y 11
#define SHUFFLE_BUTTON_Y 82

#define THUMB_MID 400
#define THUMB_SHIFT 25
#define THUMB_Y 230
#define THUMB_W 50
#define THUMB_H 35


#define AUTOHIDE_TIMEOUT 1000


/* Types definition */
typedef struct t_MapElement
{
	unsigned char tile_code;
	SDL_bool uncovered;
} MapElement;

typedef struct t_RandList
{
	unsigned char tile_code;
	SDL_bool taken;
	struct t_RandList *next;
} RandList;

typedef struct t_TileSet 
{
	gchar* filename;
	SDL_Surface *thumbnail;
} TileSet;

typedef enum { BNone, BClose, BShuffle, BChallenging, BNormal, BEasy, BUp, BDown } PressedButton;
typedef enum { SNone, SDown, SUp } ButtonStates;

int kill_the_state_file(int bohoo);
int restartgame(int bohoo);
int stoppause(int bohoo);
int deinitandsave(int bohoo);
int quitgamedontsave(int bohoo);
int topthewindow(int bohoo);


/* Global variables */
static int tilecount;
static int tiles_x, tiles_y;
static int top_margin, left_margin;
static int first_tile, second_tile;
static MapElement map[TILES_MAX * 2];
static int counter, matched;
static Uint32 uncover_time, now_time;
static SDL_Rect src, dest;
static SDL_Surface *screen;
static SDL_Surface *tile;
static SDL_Surface *ui;
static PressedButton bp;
static int bootthumbnails;

int exit_state = 0;
GList *tilesets;
GList *current_tileset;


/*
 * Draw the proper move-counter
 */
void
UpdateCounter (int counter)
{
	int n, nn, nnn;
	SDL_Rect src, dest;
	
	if (counter >999) return;
	
	nnn = (int) counter/100;
	nn = (int) (counter - nnn*100) / 10;
	n = (int) (counter - nnn*100 - nn*10);
	
	/* Blit the saved background first */
	dest.x = COUNTER_X_START;
	dest.y = COUNTER_Y_START;
	dest.w = 3*COUNTER_TEXT_WIDTH;
	dest.h = COUNTER_TEXT_HEIGHT;
	SDL_BlitSurface (ui, &dest, screen, &dest);
	SDL_UpdateRect (screen, COUNTER_X_START, COUNTER_Y_START, 3*COUNTER_TEXT_WIDTH, COUNTER_TEXT_HEIGHT);
	
	/* Blit the counter on top of it */
	src.y = GRAB_NUMBERS_Y;
	src.h = COUNTER_TEXT_HEIGHT;
	src.w = COUNTER_TEXT_WIDTH;
	src.x = GRAB_NUMBERS_X + nnn*COUNTER_TEXT_WIDTH;
	SDL_BlitSurface (ui, &src, screen, &dest);
	dest.x = dest.x + COUNTER_TEXT_WIDTH;
	src.x = GRAB_NUMBERS_X + nn*COUNTER_TEXT_WIDTH;
	SDL_BlitSurface (ui, &src, screen, &dest);
	dest.x = dest.x + COUNTER_TEXT_WIDTH;
	src.x = GRAB_NUMBERS_X + n*COUNTER_TEXT_WIDTH;
	SDL_BlitSurface (ui, &src, screen, &dest);
	SDL_UpdateRect (screen, COUNTER_X_START, COUNTER_Y_START, 3*COUNTER_TEXT_WIDTH, COUNTER_TEXT_HEIGHT);
}
	


/*
 * Generated a game map of a size proper for the tilecount
 */
static void
generate_map (void)
{
	RandList *first, *last, *curr;
	int i, remaining, required, position;

	/* First create the list for random pickups */
	first = last = curr = NULL;
	i = 0;
	while (i < (tilecount * 2))
	{
		curr = (RandList *) malloc (sizeof (RandList));
		if (first == NULL)
		{
			first = curr;
		}
		else
		{
			last->next = curr;
		}
		curr->next = NULL;
		last = curr;
		curr->taken = SDL_FALSE;
		curr->tile_code = i % tilecount;
		i++;
	}
	remaining = tilecount * 2;
	for (i = 0; i < (tilecount * 2); i++)
	{
		required = rand () % remaining;
		curr = first;
		position = 0;
		while (curr)
		{
			while (curr->taken)
			{	/* Skip taken */
				curr = curr->next;
			}
			if (position == required)
			{
				curr->taken = SDL_TRUE;
				map[i].tile_code = curr->tile_code;
				map[i].uncovered = SDL_FALSE;
				remaining--;
				break;
			}
			curr = curr->next;
			position++;
		}
	}
	/* Now free the list */
	curr = first;
	while (curr)
	{
		first = curr;
		curr = curr->next;
		free (first);
	}
}

/*
 * Init Runtime variables (at start of each game)
 */
static void
InitRuntime (void)
{
	switch (tilecount)
	{
	case TILES_EASY:
		tiles_x = 6;
		tiles_y = 5;
		top_margin = 65;
		left_margin = 65;
		break;
	case TILES_NORMAL:
		tiles_x = 6;
		tiles_y = 6;
		top_margin = 30;
		left_margin = 65;
		break;
	case TILES_CHALLENGING:
		tiles_x = 7;
		tiles_y = 6;
		top_margin = 30;
		left_margin = 10;
		break;
	default:
		printf("ERROR: no such tilecount!\n");
		break;
	}
	first_tile = second_tile = -1;
	counter = matched = 0;
}

void
BlitDifficulty(int tilecount)
{
	SDL_Rect bdest, bsrc;

	/* First restore all difficulty buttons, bit optimized for size */
	bsrc.x = 720;
	bsrc.y = 160;
	bsrc.w = 80;
	bsrc.h = 160;
	SDL_BlitSurface (ui, &bsrc, screen, &bsrc);
	SDL_UpdateRect (screen, bsrc.x, bsrc.y, bsrc.w, bsrc.h);

	/* Now highlight the proper one */

	bsrc.y = GRAB_DIF_Y;
	bdest.w = bsrc.w = GRAB_DIF_W;
	bdest.h = bsrc.h = GRAB_DIF_H;
	bdest.x = DIFFICULTY_X;
	switch (tilecount) {
		case TILES_CHALLENGING:
			bsrc.x = GRAB_DIF_CHALLENGING_X;
			bdest.y = DIFFICULTY_CHALLENGING_Y;
			break;
		case TILES_NORMAL:
			bsrc.x = GRAB_DIF_NORMAL_X;
			bdest.y = DIFFICULTY_NORMAL_Y;
			break;
		case TILES_EASY:
		default:
			bsrc.x = GRAB_DIF_EASY_X;
			bdest.y = DIFFICULTY_EASY_Y;
			break;
	}
	SDL_BlitSurface (ui, &bsrc, screen, &bdest);
	SDL_UpdateRect (screen, bdest.x, bdest.y, bdest.w, bdest.h);
}

/*
 * Blit the arrows depending on the state of current tileset
 */
static void
BlitArrows (void)
{
	SDL_Rect s, d;
	
	/* First the UP gfx */
	s.y = GRAB_UP_Y;
	s.w = UPDOWN_BUTTON_W;
	s.h = UPDOWN_BUTTON_H;
	if (current_tileset->next)
	{
		s.x = GRAB_UP_X;
	} else {
		s.x = GRAB_UP_X_DIMMED;
	}
	d.x = UPDOWN_BUTTON_X;
	d.y = UP_BUTTON_Y;
	d.w = UPDOWN_BUTTON_W;
	d.h = UPDOWN_BUTTON_H;
	SDL_BlitSurface (ui, &s, screen, &d);
	SDL_UpdateRect (screen, d.x, d.y, d.w, d.h);
	/* Next the DOWN gfx */
	s.y = GRAB_DOWN_Y;
	s.w = UPDOWN_BUTTON_W;
	s.h = UPDOWN_BUTTON_H;
	if (current_tileset->prev)
	{
		s.x = GRAB_DOWN_X;
	} else {
		s.x = GRAB_DOWN_X_DIMMED;
	}
	d.x = UPDOWN_BUTTON_X;
	d.y = DOWN_BUTTON_Y;
	d.w = UPDOWN_BUTTON_W;
	d.h = UPDOWN_BUTTON_H;
	SDL_BlitSurface (ui, &s, screen, &d);
	SDL_UpdateRect (screen, d.x, d.y, d.w, d.h);	
}

static void
BlitTileSetThumbnail()
{
	SDL_Rect dest;
	TileSet *tsp;
	SDL_Surface *thumbnail;
	
	dest.x = THUMBNAIL_X;
	dest.y = THUMBNAIL_Y;
	dest.w = 50;
	dest.h = 35;
	tsp = (TileSet*)current_tileset->data;
	thumbnail = tsp->thumbnail;
	
	SDL_BlitSurface(thumbnail, NULL, screen, &dest);
	SDL_UpdateRect (screen, dest.x, dest.y, dest.w, dest.h);
}

/*
 * Create map and draw turned tiles
 */
static void
InitAndDraw (PressedButton b)
{
	int i, j;
	
	if (b == BNone) 
	{
		/* No button, full blit of UI */
		src.w = 800; 
		src.h = 480;
		src.y = src.x = 0;
		SDL_BlitSurface (ui, &src, screen, &src);
		SDL_UpdateRect (screen, 0, 0, 0, 0);
	} else {
		/* Button, blit only playfield surroundings, rest will be covered by tiles anyway */
		src.x = 0;
		src.w = left_margin;
		src.y = 0;
		src.h = 480;
		SDL_BlitSurface (ui, &src, screen, &src);
		SDL_UpdateRect (screen, src.x, src.y, src.w, src.h);
		src.x = 720 - left_margin;
		src.w = left_margin;
		src.y = 0;
		src.h = 480;
		SDL_BlitSurface (ui, &src, screen, &src);
		SDL_UpdateRect (screen, src.x, src.y, src.w, src.h);
		src.x = left_margin;
		src.w = 720 - 2*left_margin;
		src.y = 0;
		src.h = top_margin;
		SDL_BlitSurface (ui, &src, screen, &src);
		SDL_UpdateRect (screen, src.x, src.y, src.w, src.h);
		src.y = 480 - top_margin;
		src.h = top_margin;
		SDL_BlitSurface (ui, &src, screen, &src);
		SDL_UpdateRect (screen, src.x, src.y, src.w, src.h);
	}

	/* Generate game map */
	srand (SDL_GetTicks ());
	generate_map ();
	
	/* Draw the covered board */
	src.x = GRAB_TILEBACK_X;
	src.y = GRAB_TILEBACK_Y;
	src.w = dest.w = TILE_GFX_WIDTH;
	src.h = dest.h = TILE_GFX_HEIGHT;
	for (i = 0; i < tiles_y; i++)
	{
		for (j = 0; j < tiles_x; j++)
		{
			dest.x = left_margin + j * TILE_GFX_WIDTH;
			dest.y = top_margin + i * TILE_GFX_HEIGHT;
			SDL_BlitSurface (ui, &src, screen, &dest);
		}
	}
	SDL_UpdateRect (screen, 0, 0, 0, 0);
	
	BlitDifficulty (tilecount);
	BlitArrows ();
	UpdateCounter (counter);
	BlitTileSetThumbnail ();
}



SDL_Surface*
CreateShrankThumbnail(SDL_Surface* sin)
{
	SDL_Surface *sout;
	guint x,y,bpp;
	
	bpp = screen->format->BytesPerPixel;

	sout = SDL_CreateRGBSurface(SDL_SWSURFACE, 
		50, 
		35,
		screen->format->BitsPerPixel,
		screen->format->Rmask, screen->format->Gmask,
		screen->format->Bmask, screen->format->Amask);
	if (!sout)
	{
		fprintf(stderr, "ERROR: CreateRGBSurface failed for shrank thumbnail: %s\n", SDL_GetError());
		exit(-2);
	}
	SDL_LockSurface (sin);
	for (y=0; y < 35; y++)
	{
		for (x=0; x<50; x++)
		{
			memcpy(sout->pixels + (bpp*x + sout->w*bpp*y), (sin->pixels + 2*(bpp*x + sin->w*bpp*y)), 2);
		}
	}
	SDL_UnlockSurface (sin);
	
	return sout;
}

SDL_Surface*
CreateTileSetThumbnail(gchar *tilesetname, SDL_Surface *surface)
{
	SDL_Surface *localtile;
	SDL_Surface *thumbnail;
	SDL_Surface *hthumbnail;
	SDL_Rect src;

	if (surface == NULL)
	{
		localtile = IMG_Load (tilesetname);
	}
	else
	{
		localtile = surface;
	}
	
	if (!localtile) return NULL;
		
	thumbnail = SDL_CreateRGBSurface(SDL_SWSURFACE, 
		100, 
		70,
		screen->format->BitsPerPixel,
		screen->format->Rmask, screen->format->Gmask,
		screen->format->Bmask, screen->format->Amask);
	if (!thumbnail)
	{
		fprintf(stderr, "\tERROR:CreateRGBSurface failed: %s\n", SDL_GetError());
		exit (-1);
	}
	/* Get the first frame for the thumbnail */
	src.x = 0;
	src.y = 0;
	src.w = 100;
	src.h = 70;
	SDL_BlitSurface(localtile, &src, thumbnail, &src);
	SDL_UpdateRect(thumbnail, 0, 0, 100, 70);
	
	hthumbnail = CreateShrankThumbnail(thumbnail);
	SDL_FreeSurface(thumbnail);
	if (!surface) SDL_FreeSurface(localtile);
	
	return hthumbnail;
}

static void
BlitBootThumbnail (SDL_Surface *s)
{
	SDL_Rect ls, ld;
	
	if (bootthumbnails == 0)
	{
		ld.x = THUMB_MID - THUMB_SHIFT;
		ld.y = THUMB_Y;
		ld.w = THUMB_W;
		ld.h = THUMB_H;
		SDL_BlitSurface (s, NULL, screen, &ld);
		SDL_UpdateRect (screen, ld.x, ld.y, ld.w, ld.h);
	} else {
		ls.x = THUMB_MID - bootthumbnails*THUMB_SHIFT;
		if (ls.x<0) ls.x = 0;
		ls.y = ld.y = THUMB_Y;
		ls.w = ld.w = (THUMB_MID + bootthumbnails*THUMB_SHIFT) - ls.x;
		ls.h = ld.h = THUMB_H;
		ld.x = ls.x - THUMB_SHIFT;
		if (ld.x < 0) ld.x = 0;
		SDL_BlitSurface (screen, &ls, screen, &ld);
		SDL_UpdateRect (screen, ld.x, ld.y, ld.w, ld.h);
		ld.x = (THUMB_MID - THUMB_SHIFT) + bootthumbnails*THUMB_SHIFT;
		ld.w = THUMB_W;
		SDL_BlitSurface (s, NULL, screen, &ld);
		SDL_UpdateRect (screen, ld.x, ld.y, ld.w, ld.h);
	}
	bootthumbnails++;
}


void LoadFiles()
{
	GDir *gdir;
	GError *gerror;
	const char *home;
	gchar *path;
	const gchar *fname;
	TileSet *ttileset;
	guint ntilesets;
	
	tilesets = NULL;
	ntilesets = 0;
	bootthumbnails = 0;
	/* Load from game dir */
	gerror = NULL;
	gdir = g_dir_open(NAKO_DATADIR, 0, &gerror);
	if (!gdir)
	{
		fprintf(stderr, "\tERROR: Can't open %s for reading\n", NAKO_DATADIR);
		return;
	}
	while ((fname = g_dir_read_name(gdir)) != NULL)
	{
		if (g_strrstr(fname, ".tileset"))
		{
			ttileset = g_malloc0 (sizeof (TileSet));
			ttileset->filename = g_strjoin (NULL, NAKO_DATADIR, fname, NULL);
			ttileset->thumbnail = CreateTileSetThumbnail (ttileset->filename, NULL);
			BlitBootThumbnail (ttileset->thumbnail);
			tilesets = g_list_append (tilesets, ttileset);
			current_tileset = tilesets;
			ntilesets++;
			printf("\tAdding %s\n", fname);
		}
	}
	g_dir_close(gdir);

	/* Load from Home dir */
	home = g_getenv("HOME");
	path = g_strdup_printf("%s/MyDocs/.images", home);
	gdir = g_dir_open(path, 0, &gerror);
	if (!gdir)
	{
		fprintf(stderr, "\tERROR: Can't open user's Images folder for reading\n");
	} else {
		while ((fname = g_dir_read_name(gdir)) != NULL)
		{
			if (g_strrstr(fname, ".tileset"))
			{
				ttileset = g_malloc0 (sizeof (TileSet));
				ttileset->filename = g_strdup_printf ("%s/MyDocs/.images/%s", home, fname);
				ttileset->thumbnail = CreateTileSetThumbnail (ttileset->filename, NULL);
				BlitBootThumbnail (ttileset->thumbnail);
				tilesets = g_list_append (tilesets, ttileset);
				ntilesets++;
				printf("\tAdding %s\n", fname);
			}
		}
		g_dir_close(gdir);
	}
	g_free(path);
	
	/* Select random tileset at start */
	if (ntilesets > 1)
	{
		int nth;
		
		srand (SDL_GetTicks ());
		nth = rand () % ntilesets;
		current_tileset = g_list_nth (tilesets, nth);
	} else
		current_tileset = tilesets;
		
	tile = IMG_Load (((TileSet*)(current_tileset->data))->filename);
		
}

static void LoadGui()
{
	ui = IMG_Load (NAKO_DATADIR DEFAULT_UI);	
}


#define DEFAULTSCREEN screen->format->BitsPerPixel, screen->format->Rmask, screen->format->Gmask, screen->format->Bmask, screen->format->Amask

/*
 * Game initialization (global variables, loading images, etc.
 */
static void
InitGame (void)
{

	/* Init global variables for normal game */
	/* TODO - state saving stuff should go here */
	tilecount = TILES_CHALLENGING;
	InitRuntime ();
	LoadFiles();
	LoadGui();
	InitAndDraw(BNone);
}



/*
 * Load previous tileset
 */
static SDL_bool
SelectPreviousTileset(void)
{
	SDL_bool retval = SDL_FALSE;
	
	if (current_tileset->next)
	{
		current_tileset = current_tileset->next;
		if (tile) SDL_FreeSurface (tile);
		tile = IMG_Load ( ((TileSet*)(current_tileset->data))->filename);
		//BlitTileSetThumbnail ();
		//BlitArrows ();
		retval = SDL_TRUE;
	}	
	return retval;
}

/*
 * Load next tileset
 */
static SDL_bool
SelectNextTileset(void)
{
	SDL_bool retval = SDL_FALSE;
	
	if (current_tileset->prev)
	{
		current_tileset = current_tileset->prev;
		if (tile) SDL_FreeSurface (tile);
		tile = IMG_Load ( ((TileSet*)(current_tileset->data))->filename);
		//BlitTileSetThumbnail ();
		//BlitArrows ();
		retval = SDL_TRUE;
	}	
	return retval;
}


/*
 * Show tile
 */
static void
ShowTile(const int t)
{
	if (t >= 0)
	{
		dest.x = left_margin + (t%tiles_x) * TILE_GFX_WIDTH;
		dest.y = top_margin + (t/tiles_x) * TILE_GFX_HEIGHT;
		src.x = map[t].tile_code*TILE_GFX_WIDTH;
		src.y = 0;
		src.w = 100;
		src.h = 70;
		SDL_BlitSurface (tile, &src, screen, &dest);
		SDL_UpdateRect (screen, dest.x, dest.y, TILE_GFX_WIDTH, TILE_GFX_HEIGHT);
	}
}


/*
 * Hide tile
 */
static void
HideTile(const int t)
{
	if (t >= 0)
	{
		dest.x = left_margin + (t%tiles_x)*TILE_GFX_WIDTH;
		dest.y = top_margin + (t/tiles_x)*TILE_GFX_HEIGHT;
		src.x = GRAB_TILEBACK_X;
		src.y = GRAB_TILEBACK_Y;
		src.w = TILE_GFX_WIDTH;
		src.h = TILE_GFX_HEIGHT;
		SDL_BlitSurface (ui, &src, screen, &dest);
		SDL_UpdateRect (screen, dest.x, dest.y, TILE_GFX_WIDTH, TILE_GFX_HEIGHT);
	}
}


/*
 * Blit button states
 */
static void
BlitButton (PressedButton b, ButtonStates s)
{
	SDL_Rect ls, ld;
	
	switch (b) {
		case BClose:
			ld.x = BIGBUTTON_X;
			ld.y = CLOSE_BUTTON_Y;
			ld.w = ls.w = BIGBUTTON_W;
			ld.h = ls.h = BIGBUTTON_H;
			if (s == SDown) {
				ls.x = GRAB_CLOSE_BUTTON_X_DOWN;
				ls.y = GRAB_CLOSE_BUTTON_Y_DOWN;
			} else {
				ls.x = GRAB_CLOSE_BUTTON_X;
				ls.y = GRAB_CLOSE_BUTTON_Y;
			}
			SDL_BlitSurface (ui, &ls, screen, &ld);
			SDL_UpdateRect (screen, ld.x, ld.y, ld.w, ld.h);
			break;
		case BShuffle:
			ld.x = BIGBUTTON_X;
			ld.y = SHUFFLE_BUTTON_Y;
			ld.w = ls.w = BIGBUTTON_W;
			ld.h = ls.h = BIGBUTTON_H;
			if (s == SDown) {
				ls.x = GRAB_SHUFFLE_BUTTON_X_DOWN;
				ls.y = GRAB_SHUFFLE_BUTTON_Y_DOWN;
			} else {
				ls.x = GRAB_SHUFFLE_BUTTON_X;
				ls.y = GRAB_SHUFFLE_BUTTON_Y;
			}
			SDL_BlitSurface (ui, &ls, screen, &ld);
			SDL_UpdateRect (screen, ld.x, ld.y, ld.w, ld.h);		
			break;
		case BEasy:	
			ld.x = DIFFICULTY_X;
			ld.y = DIFFICULTY_EASY_Y;
			ld.w = ls.w = GRAB_DIF_W;
			ld.h = ls.h = GRAB_DIF_H;
			if (s == SDown) {
				ls.x = GRAB_DIF_EASY_X_DIMMED;
				ls.y = GRAB_DIF_Y;
			} else {
				ls.x = GRAB_DIF_EASY_X;
				ls.y = GRAB_DIF_Y;
			}
			SDL_BlitSurface (ui, &ls, screen, &ld);
			SDL_UpdateRect (screen, ld.x, ld.y, ld.w, ld.h);		
			break;
		case BNormal:
			ld.x = DIFFICULTY_X;
			ld.y = DIFFICULTY_NORMAL_Y;
			ld.w = ls.w = GRAB_DIF_W;
			ld.h = ls.h = GRAB_DIF_H;
			if (s == SDown) {
				ls.x = GRAB_DIF_NORMAL_X_DIMMED;
				ls.y = GRAB_DIF_Y;
			} else {
				ls.x = GRAB_DIF_NORMAL_X;
				ls.y = GRAB_DIF_Y;
			}
			SDL_BlitSurface (ui, &ls, screen, &ld);
			SDL_UpdateRect (screen, ld.x, ld.y, ld.w, ld.h);
			break;
		case BChallenging:
			ld.x = DIFFICULTY_X;
			ld.y = DIFFICULTY_CHALLENGING_Y;
			ld.w = ls.w = GRAB_DIF_W;
			ld.h = ls.h = GRAB_DIF_H;
			if (s == SDown) {
				ls.x = GRAB_DIF_CHALLENGING_X_DIMMED;
				ls.y = GRAB_DIF_Y;
			} else {
				ls.x = GRAB_DIF_CHALLENGING_X;
				ls.y = GRAB_DIF_Y;
			}
			SDL_BlitSurface (ui, &ls, screen, &ld);
			SDL_UpdateRect (screen, ld.x, ld.y, ld.w, ld.h);
			break;
		case BUp:
			ld.x = UPDOWN_BUTTON_X;
			ld.y = UP_BUTTON_Y;
			ld.w = ls.w = UPDOWN_BUTTON_W;
			ld.h = ls.h = UPDOWN_BUTTON_H;
			if (s == SDown) {
				if (!current_tileset->next) break;
				ls.x = GRAB_UP_X_DOWN;
				ls.y = GRAB_UP_Y_DOWN;
			} else {
				if (current_tileset->next) {
					ls.x = GRAB_UP_X;
					ls.y = GRAB_UP_Y;
				} else {
					ls.x = GRAB_UP_X_DIMMED;
					ls.y = GRAB_UP_Y_DIMMED;
				}
			}
			SDL_BlitSurface (ui, &ls, screen, &ld);
			SDL_UpdateRect (screen, ld.x, ld.y, ld.w, ld.h);
			break;
		case BDown:
			ld.x = UPDOWN_BUTTON_X;
			ld.y = DOWN_BUTTON_Y;
			ld.w = ls.w = UPDOWN_BUTTON_W;
			ld.h = ls.h = UPDOWN_BUTTON_H;
			if (s == SDown) {
				if (!current_tileset->prev) break;
				ls.x = GRAB_DOWN_X_DOWN;
				ls.y = GRAB_DOWN_Y_DOWN;
			} else {
				if (current_tileset->prev) {
					ls.x = GRAB_DOWN_X;
					ls.y = GRAB_DOWN_Y;
				} else {
					ls.x = GRAB_DOWN_X_DIMMED;
					ls.y = GRAB_DOWN_Y_DIMMED;
				}
			}
			SDL_BlitSurface (ui, &ls, screen, &ld);
			SDL_UpdateRect (screen, ld.x, ld.y, ld.w, ld.h);
			break;
		default:
			break;
	}
}

/*
 * Main game loop
 *
 * Waits for events, and acts on them
 *
 */
void
GameLoop (void)
{
	SDL_Event event;
	int tile;
	SDL_bool shouldquit;

	shouldquit = SDL_FALSE;
start:
	while (SDL_PollEvent(&event))
	{
		switch (event.type)
		{
		case SDL_QUIT:
			shouldquit = SDL_TRUE;
			SDL_Quit ();
			break;
		case SDL_KEYDOWN:
			if (event.key.keysym.scancode == 9) /* FIXME!!! */
			{
				shouldquit = SDL_TRUE;
				SDL_Quit ();
			}
			break;
		case SDL_MOUSEBUTTONDOWN:
			bp = BNone;
			if ((event.button.x > UPDOWN_BUTTON_X) && (event.button.x < UPDOWN_BUTTON_X + UPDOWN_BUTTON_W))
			{ 
				if ((event.button.y > UP_BUTTON_Y) && (event.button.y < UP_BUTTON_Y + UPDOWN_BUTTON_H))
				{
					bp = BUp;
				} else if ((event.button.y > DOWN_BUTTON_Y) && (event.button.y < DOWN_BUTTON_Y + UPDOWN_BUTTON_H)) {
					bp = BDown;
				}
			}
			if ((event.button.x > DIFFICULTY_X) && (event.button.x < DIFFICULTY_X + GRAB_DIF_W))  
			{ 
				if ((event.button.y > DIFFICULTY_CHALLENGING_Y) && (event.button.y < DIFFICULTY_CHALLENGING_Y + GRAB_DIF_H))
				{
					bp = BChallenging;
				} else if ((event.button.y > DIFFICULTY_NORMAL_Y) && (event.button.y < DIFFICULTY_NORMAL_Y + GRAB_DIF_H)) {
					bp = BNormal;
				} else if ((event.button.y > DIFFICULTY_EASY_Y) && (event.button.y < DIFFICULTY_EASY_Y + GRAB_DIF_H)) {
					bp = BEasy;
				}
			}
			if ((event.button.x > BIGBUTTON_X) && (event.button.x < BIGBUTTON_X + BIGBUTTON_W))
			{	
				if ((event.button.y > CLOSE_BUTTON_Y) && (event.button.y < CLOSE_BUTTON_Y + BIGBUTTON_H))
				{
					bp = BClose;
				} else if ((event.button.y > SHUFFLE_BUTTON_Y) && (event.button.y < SHUFFLE_BUTTON_Y + BIGBUTTON_H)) {
					bp = BShuffle;
				}
			}
			switch (bp) {
				case BClose:
					BlitButton (BClose, SDown);
					break;
				case BShuffle:
					BlitButton (BShuffle, SDown);
					break;
				case BEasy:
					BlitButton (BEasy, SDown);
					break;
				case BNormal:
					BlitButton (BNormal, SDown);
					break;
				case BChallenging:
					BlitButton (BChallenging, SDown);
					break;
				case BUp:
					BlitButton (BUp, SDown);
					break;
				case BDown:
					BlitButton (BDown, SDown);
					break;
				case BNone:
				default:
					break;
			}			
			break;
			
		case SDL_MOUSEBUTTONUP:
			switch (bp) {
				case BClose:
					shouldquit = SDL_TRUE;
					BlitButton (BClose, SUp);
					SDL_Quit ();
					break;
				case BShuffle:
					BlitButton (BShuffle, SUp);
					InitRuntime();
					InitAndDraw(BShuffle);
					break;
				case BEasy:
					BlitButton (BEasy, SUp);
					if (tilecount != TILES_EASY) 
					{
						tilecount = TILES_EASY;
						InitRuntime();
						InitAndDraw(BEasy);
					}
					break;
				case BNormal:
					BlitButton (BNormal, SUp);
					if (tilecount != TILES_NORMAL)
					{
						tilecount = TILES_NORMAL;
						InitRuntime();
						InitAndDraw(BNormal);
					}
					break;
				case BChallenging:
					BlitButton (BChallenging, SUp);
					if (tilecount != TILES_CHALLENGING)
					{
						tilecount = TILES_CHALLENGING;
						InitRuntime();
						InitAndDraw(BChallenging);
					}
					break;
				case BUp:
					BlitButton (BUp, SUp);
					if (SelectPreviousTileset())
					{
						InitRuntime();
						InitAndDraw(BUp);
					}
					break;
				case BDown:
					BlitButton (BDown, SUp);
					if (SelectNextTileset())
					{
						InitRuntime();
						InitAndDraw(BDown);
					}
					break;
				default:
					break;
			}				
			
			if (bp != BNone) break;
			if ((event.button.x < left_margin) || (event.button.x > left_margin + tiles_x*TILE_GFX_WIDTH) || 
				(event.button.y < top_margin) || (event.button.y > top_margin + tiles_y*TILE_GFX_HEIGHT)) 
				break;

			tile = (event.button.x - left_margin) / TILE_GFX_WIDTH + tiles_x * ((event.button.y - top_margin) / TILE_GFX_HEIGHT);
			if (map[tile].uncovered == SDL_TRUE)
				break;
			
			uncover_time = SDL_GetTicks();
			if (first_tile == -1)
			{	
				/* Nothing is turned, turn first */
				first_tile = tile;
				ShowTile(first_tile);
			}
			else
			{
				if (first_tile == tile)
				{
					/* The same tile clicked again */
					break;
				}
				/* First one is shown already */
				if (second_tile != -1)
				{	/* Both already turned. Cover both and uncover the new first. Remove the hiding timeout */
					HideTile(first_tile);
					first_tile = -1;
					HideTile(second_tile);
					second_tile = -1;
					/* Turn the new first */
					first_tile = tile;
					ShowTile(first_tile);
				}
				else
				{	/* Second not turned yet - turn it */
					second_tile = tile;
					ShowTile(second_tile);
					counter++;
					UpdateCounter(counter);
					if (map[first_tile].tile_code == map[second_tile].tile_code)
					{
						/* This is match! */
						map[first_tile].uncovered = map[second_tile].uncovered = SDL_TRUE;
						first_tile = second_tile = -1;
						matched++;
						if (matched == tilecount)
						{
							/* End of game, maybe we should do something about it, like play a "hurray" sound ;-)
							*/
						}
					}
					/* Start the auto-cover timer */
					uncover_time = SDL_GetTicks();
				}
			}
			break;
		}
	}
	usleep(1000);
	if (second_tile != -1)
	{
		now_time = SDL_GetTicks();
		if ((now_time - uncover_time) > AUTOHIDE_TIMEOUT)
		{
			/* Timeout - hide the shown tiles */
			HideTile(first_tile);
			HideTile(second_tile);
			first_tile = second_tile = -1;
		}
	}
	if (shouldquit == SDL_FALSE) goto start;
}

int
main (int argc, char *argv[])
{
	putenv("SDL_VIDEO_X11_WMCLASS=nako");
	
	if (SDL_Init (SDL_INIT_VIDEO) != 0)
	{
		printf ("Unable to init SDL: %s\n", SDL_GetError ());
		return (1);
	}

	atexit (SDL_Quit);
	screen = SDL_SetVideoMode (800, 480, 16, SDL_FULLSCREEN);
	if (screen == NULL)
	{
		printf ("Unable to set video mode: %s\n", SDL_GetError ());
		return (1);
	}
	SDL_WM_SetCaption ("nako", NULL);
	SDL_ShowCursor (SDL_DISABLE);
	InitGame ();
	GameLoop ();

	return 0;
}
