/*
 * ui_callback.c - UI of MSA program.
 * This file is part of MSA program.
 *
 * Copyright (C) 2009 - Andrey Bogachev
 * 
 * MSA program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MSA program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with MSA program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */
 
#include "ui_general.h"
#include <hildon/hildon.h>

/** save profile request **/
const char* ui_profile_req = "\
<TransitData>\
<TargetID>vkontakte</TargetID>\
<SourceID>ui</SourceID>\
<Content>\
<Request class=\"profile\" function = \"sendMyProfile\">\
<Params/>\
</Request>\
</Content>\
</TransitData>";

/**
 * @brief fullscreen callback
 * @param
 * @return
**/
void callback_full(GtkAction* action, gpointer data)
{
    gtk_window_fullscreen(GTK_WINDOW(ui_data->window));
}

/**
 * @brief go to page callback
 * @param
 * @return
**/
void callback_goto(GtkWidget* widget, int i)
{
    gtk_window_set_title(GTK_WINDOW (ui_data->window), ui_data->str_page[i]);
    gtk_notebook_set_current_page(GTK_NOTEBOOK(ui_data->tasks), i);
    ui_data->prev = ui_data->page;
    ui_data->page = i;
    if (ui_data->page == 2 && !ui_data->empty_profile) {
        ui_data->empty_profile = FALSE;
        callback_button_profile_update(NULL, NULL);
    }
}

/**
 * @brief settings window callback
 * @param
 * @return
**/
void callback_sets(GtkAction* action, gpointer data)
{
    gtk_notebook_set_current_page(GTK_NOTEBOOK(ui_data->tasks), 3);
}

/**
 * @brief about window callback
 * @param
 * @return
**/
void callback_bout(GtkAction* action, gpointer data)
{
    ui_init_bout();
}

/**
 * @brief help window callback
 * @param
 * @return
**/
void callback_help(GtkAction* action, gpointer data)
{
    ui_init_help();
}

/**
 * @brief edit detail field callback
 * @param
 * @return
**/
void callback_detail_edit(GtkWidget* button, gpointer value)
{
    gchar* str = hildon_button_get_value(button);
    GtkWindow* window;
    GtkDialog* dialog;
    GtkWidget* caption;    
    GtkWidget* entry;
    int result;      

    dialog = GTK_DIALOG(gtk_dialog_new());
    if(!dialog) {
        return;
    }

    //gtk_window_set_title(GTK_WINDOW(dialog), g_strconcat("Enter ", gtk_widget_get_name(button), NULL));
    gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);
    window = GTK_WINDOW(dialog);

    /* buttons */    
    gtk_dialog_add_button (GTK_DIALOG (dialog), STR_CNCL, GTK_RESPONSE_NONE);
    gtk_dialog_add_button (GTK_DIALOG (dialog), STR_SAVE, GTK_RESPONSE_ACCEPT);

    entry = hildon_entry_new(HILDON_SIZE_AUTO);
    hildon_entry_set_text(entry, str);
    gtk_container_add(GTK_CONTAINER(dialog->vbox), entry);
    gtk_widget_show_all(GTK_WIDGET(dialog));

    /* Run dialog cycle */
    result = gtk_dialog_run(dialog);

    if(result == GTK_RESPONSE_ACCEPT) {
        hildon_button_set_value(button, hildon_entry_get_text(entry)); 
    }
    
    gtk_widget_destroy(GTK_WIDGET(dialog));
    
    g_free(str);
    //ui_data->prev = ui_data->page;
    //callback_goto(NULL, 3);    
}

/**
 * @brief on combobox changed value callback
 * @param
 * @return
**/
void callback_combo_changed(GtkComboBox* widget, gpointer user_data)
{
    //g_print("%s: ", (gchar*)user_data);
    //g_print("%s\n", gtk_combo_box_get_active_text(GTK_COMBO_BOX(widget)));
    
    gchar *obj_path = g_strconcat("//TransitData/Content/Response/Params/select[@name='", (gchar*)user_data,"']", NULL);
    xmlXPathObject* obj = xpath(obj_path, ui_data->ui_request);
    g_free(obj_path);
    //g_print("\n\n\n!!!!! %s !!!!!\n\n\n%d\n\n\n", hildon_button_get_value (widget), hildon_picker_button_get_active(widget));
    if (obj->nodesetval != NULL) {
    
        //xmlNodePtr node = xmlNewDocNode(ui_data->ui_request, NULL, BAD_CAST TYPE_STRNG, gtk_combo_box_get_active_text(GTK_COMBO_BOX(widget)));
        xmlNodePtr node = xmlNewDocNode(ui_data->ui_request, NULL, BAD_CAST TYPE_STRNG, hildon_button_get_value (widget));
        xmlSetProp(node, "selected", "1");
    
        xmlUnlinkNode(obj->nodesetval->nodeTab[0]->children);
        xmlFreeNodeList(obj->nodesetval->nodeTab[0]->children);
        obj->nodesetval->nodeTab[0]->children = NULL;
    
        xmlAddChild(obj->nodesetval->nodeTab[0], node);
    }
    else {
        g_debug("callback_date error obj is null");    
    }
}

/**
 * @brief edit date field callback
 * @param
 * @return
**/
void callback_date_edit(GtkWidget* button, gpointer value)
{
    gint y, m, d;
    hildon_date_button_get_date(button, &y, &m, &d);
    struct tm date;
    date.tm_mday = d;
    date.tm_mon = m;
    date.tm_year = y - 1900;
    gchar* str_date = g_malloc(sizeof(gchar) * 11);
    strftime(str_date, 11, "%d.%m.%Y\0", &date);
    hildon_button_set_value(button, str_date);
    //xmlXPathObject* obj = xpath("//TransitData/Content/Response/Params/date[@name='Birthday']", ui_data->ui_request);
    //g_print("\n\n\n\t!!! %s !!!\n\n\n", str_date);
    //if (obj != NULL && obj->nodesetval->nodeNr != 0) {
    //    g_debug("!!!!");
    //    xmlNodeSetContent(obj->nodesetval->nodeTab[0], str_date);
    //}
    //g_print("\n\ndate is set\n\n");
    g_free(str_date);
}

/**
 * @brief edit date field callback
 * @param
 * @return
**/
void callback_date_changed(GtkWidget* widget, gpointer user_data)
{
    gint y, m, d;
    hildon_date_selector_get_date(widget, &y, &m, &d);
    struct tm date;
    date.tm_mday = d;
    date.tm_mon = m;
    date.tm_year = y - 1900;
    gchar* str_date = g_malloc(sizeof(gchar) * 11);
    strftime(str_date, 11, "%d.%m.%Y\0", &date);
    g_print("\n\n!!! %s !!!\n\n", str_date);
}

/**
 * @brief get profile callback
 * @param
 * @return
**/
void callback_button_profile_get(GtkWidget* button, gpointer data)
{
    xmlDocPtr resp = NULL;
    //hildon_gtk_window_set_progress_indicator(ui_data->window, 1);
    kernel->send(xmlParseDoc("<?xml version=\"1.0\"?><TransitData id=\"1\" \
    type=\"data\"><SourceID>ui</SourceID>\
    <TargetID>db</TargetID>\
    <Content><Request class=\"profile\" function = \"getProfile\"><Params></Params></Request></Content></TransitData>"), &resp, NULL);
    xmlFreeDoc(resp);
}

/**
 * @brief get friends list callback
 * @param
 * @return
**/
void callback_button_friends_get(GtkWidget* button, gpointer data)
{
    //hildon_gtk_window_set_progress_indicator(ui_data->window, 1);
    int from = FRIENDS_PER_PAGE*ui_data->page_frds;
    int to = FRIENDS_PER_PAGE*(ui_data->page_frds + 1);
    gchar* str_sort;
    if(ui_data->ascending) {
        str_sort = g_strdup("asc");
    } else {
        str_sort = g_strdup("desc");
    }
    /* correct number of friends */
    if (to >  ui_data->num_frds) to =  ui_data->num_frds;
    //gtk_label_set_text(GTK_LABEL(ui_data->label_friends_page), g_strdup_printf("%d - %d / %d\n", from + 1, to, ui_data->num_frds));
    if (to <  FRIENDS_PER_PAGE) to =  FRIENDS_PER_PAGE;
    //g_print(g_strdup_printf("%d",from)));
    //g_print("callback_button_friends_get KERNEL->SEND\n");
    
    gchar *sfrom = g_strdup_printf("%d", from);
    gchar *sto = g_strdup_printf("%d", to);
    gchar *request = g_strconcat("<?xml version=\"1.0\"?><TransitData id=\"1\" \
    type=\"data\"><SourceID>ui</SourceID>\
    <TargetID>db</TargetID>\
    <Content><Request class=\"friends\" function = \"getListFriends\"><Params From=\"", sfrom,
    "\" To=\"", sto, "\" Sort=\"", str_sort, "\"/></Request></Content></TransitData>", NULL);
    xmlDocPtr resp = NULL;
    kernel->send(xmlParseDoc(request), &resp, NULL);
    xmlFreeDoc(resp);
    g_free(request);
    g_free(sfrom);
    g_free(str_sort);
    g_free(sto);
    //g_print("GETTING FRIENDS FROM %d to %d\n", from + 1, to + 1);
}

/**
 * @brief sort friends lsit
 * @param
 * @return
**/
void callback_button_friends_sort(GtkWidget* button, gpointer data)
{
    ui_data->ascending = !ui_data->ascending;
    callback_button_friends_get(NULL, NULL);
}

/**
 * @brief filter
 * @param
 * @return
**/
void callback_friends_filter(gchar* str_filter)
{
    if (str_filter != NULL) {
        if (strlen(str_filter) > 0) {
            ui_data->friends_filter = TRUE;
        } else {
            ui_data->friends_filter = FALSE;
        }
    } else {
        ui_data->friends_filter = FALSE;
    }
    
    if (ui_data->friends_filter) {
        gtk_widget_hide(ui_data->box_friends_navi);
        gtk_label_set_text(GTK_LABEL(ui_data->label_friends_filter), g_strconcat(STR_RSLT, " \"", str_filter, "\"", NULL));
        gtk_widget_show_all(ui_data->box_friends_result);
    } else {
        gtk_widget_show_all(ui_data->box_friends_navi);
        gtk_widget_hide(ui_data->box_friends_result);
    }
}

/**
 * @brief update friends list in database callback
 * @param
 * @return
**/
void callback_button_friends_result(GtkWidget* button_cb, gpointer data)
{
    callback_friends_filter(NULL);
    callback_button_friends_get(NULL, NULL);
}

/**
 * @brief update friends list in database callback
 * @param
 * @return
**/
void callback_button_friends_update(GtkWidget* button, gpointer data)
{
    xmlDocPtr resp = NULL;
    //hildon_gtk_window_set_progress_indicator(ui_data->window, 1);
    kernel->send(xmlParseDoc("<?xml version=\"1.0\"?><TransitData id=\"1\" \
    type=\"data\"><SourceID>ui</SourceID>\
    <TargetID>db</TargetID>\
    <Content><Request class=\"friends\" function = \"updateListFriends\"><Params></Params></Request></Content></TransitData>"), &resp, NULL);
    xmlFreeDoc(resp);
}

/**
 * @brief prev page of a friends list
 * @param
 * @return
**/
void callback_button_friends_page_prev(GtkWidget* button, gpointer data)
{
    if (ui_data->page_frds > 0) {
        ui_data->page_frds--;
    }
    int from = ui_data->page_frds * FRIENDS_PER_PAGE;
    int to = (ui_data->page_frds + 1) * FRIENDS_PER_PAGE;
    if (to >  ui_data->num_frds) to =  ui_data->num_frds;
    //gtk_label_set_text(GTK_LABEL(ui_data->label_friends_page), g_strdup_printf("%d - %d / %d\n", from + 1, to, ui_data->num_frds));
    callback_button_friends_get(NULL, NULL);
}

/**
 * @brief next page of a friends list
 * @param
 * @return
**/
void callback_button_friends_page_next(GtkWidget* button, gpointer data)
{
    if (ui_data->page_frds < ui_data->num_frds / FRIENDS_PER_PAGE) {
        ui_data->page_frds++;
    }
    int from = ui_data->page_frds * FRIENDS_PER_PAGE;
    int to = (ui_data->page_frds + 1) * FRIENDS_PER_PAGE;
    if (to >  ui_data->num_frds) to =  ui_data->num_frds;
    //gtk_label_set_text(GTK_LABEL(ui_data->label_friends_page), g_strdup_printf("%d - %d / %d\n", from + 1, to, ui_data->num_frds));
    callback_button_friends_get(NULL, NULL);
}

/**
 * @brief the 1st page of a friends list
 * @param
 * @return
**/
void callback_button_friends_page_frst(GtkWidget* button, gpointer data)
{
    ui_data->page_frds = 0;
    callback_button_friends_get(NULL, NULL);
}

/**
 * @brief the last page of a friends list
 * @param
 * @return
**/
void callback_button_friends_page_last(GtkWidget* button, gpointer data)
{
    ui_data->page_frds = ui_data->num_frds / FRIENDS_PER_PAGE;
    callback_button_friends_get(NULL, NULL);
}

/**
 * @brief get messages list callback
 * @param
 * @return
**/
void callback_button_messages_get(GtkWidget* button, gpointer data)
{
    //UIdata* ui_data = NULL;
    
    //ui_data = ( UIdata* )data;
    //g_assert(ui_data != NULL);
    gchar* str_msgs_box;
    if (ui_data->msgs_box == INBOX) {
        str_msgs_box = g_strdup("Inbox");
    } else {
        str_msgs_box = g_strdup("Outbox");
    }

    GList* children = gtk_container_get_children (GTK_CONTAINER(ui_data->box_messages_list));
    for (; children != NULL; children = children->next) {
        gtk_widget_destroy(children->data); 
    }
    gtk_label_set_text(GTK_LABEL(ui_data->label_messages_page), STR_EMPT);
    
    /* correct number of messages */
    int from = MESSAGES_PER_PAGE*ui_data->page_msgs;
    int to = MESSAGES_PER_PAGE*(ui_data->page_msgs + 1);
    if (to >  ui_data->num_msgs) to =  ui_data->num_msgs;
    //gtk_label_set_text(GTK_LABEL(ui_data->label_messages_page), g_strdup_printf("%d - %d / %d\n", from + 1, to, ui_data->num_msgs));
    if (to <  MESSAGES_PER_PAGE) to =  MESSAGES_PER_PAGE;

    gchar *sfrom = g_strdup_printf("%d", from);
    gchar *sto = g_strdup_printf("%d", to);
    gchar *request = g_strconcat("<?xml version=\"1.0\"?><TransitData id=\"1\" \
    type=\"data\"><SourceID>ui</SourceID>\
    <TargetID>db</TargetID>\
    <Content><Request class=\"messages\" function = \"get",str_msgs_box,"Messages\"><Params From=\"",sfrom,
    "\" To=\"",sto,"\"></Params></Request></Content></TransitData>",NULL);
    //hildon_gtk_window_set_progress_indicator(ui_data->window, 1);
    kernel->send(xmlParseDoc(request), NULL, NULL);

    g_free(request);
    g_free(str_msgs_box);
    g_free(sfrom);
    g_free(sto);
}

/**
 * @brief reply on a message callback
 * @param
 * @return
**/
void callback_message_reply(GtkWidget* container, gpointer data)
{
    Message* msg = data;
    GtkWidget* window_msg = hildon_stackable_window_new();
    gtk_window_set_title(GTK_WINDOW(window_msg), STR_PROG);
    gtk_container_add(GTK_CONTAINER((GtkWidget*)window_msg), ui_init_message_reply(data));
    gtk_widget_show(GTK_WIDGET(window_msg));
    return;
}

/**
 * @brief show reply entry
 * @param
 * @return
**/
void callback_message_show_reply_entry(GtkWidget* button, gpointer data)
{
    gtk_widget_show_all(gtk_widget_get_parent(GTK_WIDGET(button)));
    gtk_widget_grab_focus(GTK_WIDGET(data));
    gtk_widget_hide(GTK_WIDGET(button));
}

/**
 * @brief send reply
 * @param
 * @return
**/
void callback_message_send_reply(GtkWidget* button, gpointer data)
{
    gtk_widget_show_all(gtk_widget_get_parent(GTK_WIDGET(button)));
    gtk_widget_hide(GTK_WIDGET(data));
    gtk_widget_hide(GTK_WIDGET(button));
    GtkTextIter start, end;
    GtkTextBuffer* text_buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(data));
    gtk_text_buffer_get_start_iter(text_buffer, &start);
    gtk_text_buffer_get_end_iter(text_buffer, &end);
    gchar* str_msg = gtk_text_buffer_get_text(text_buffer, &start, &end, FALSE);
    //g_object_unref(G_OBJECT(text_buffer));
    //g_print("\n\n\n\tSENDING MSG TO: %s\n\t%s\n\n\n", gtk_widget_get_name(GTK_WIDGET(data)), str_msg);
    gtk_text_view_set_buffer(GTK_TEXT_VIEW(data), gtk_text_buffer_new(gtk_text_tag_table_new()));
    CURL* curl = curl_easy_init();
    gchar* str_to_send = curl_easy_escape(curl, str_msg, 0);
    curl_easy_cleanup(curl);
    g_free(str_msg);
    gchar *request = g_strconcat("<?xml version=\"1.0\"?><TransitData id=\"1\" \
        type=\"data\"><SourceID>ui</SourceID><TargetID>vkontakte</TargetID><Content><Request class=\"messages\" function = \"sendMessage\">\
        <Params id=\"", gtk_widget_get_name(GTK_WIDGET(data)), "\"><string name=\"text\">", str_to_send , "</string></Params></Request></Content></TransitData>", NULL);
    //hildon_gtk_window_set_progress_indicator(ui_data->window, 1);
    kernel->send(xmlParseDoc(request), NULL, NULL);
    g_free(request);
    g_free(str_to_send);
}

/**
 * @brief prev page of a messages list
 * @param
 * @return
**/
void callback_button_msgs_page_prev(GtkWidget* button, gpointer data)
{
    if (ui_data->page_msgs > 0) {
        ui_data->page_msgs--;
    }
// kulakov: unused variables???
//    int from = ui_data->page_msgs * MESSAGES_PER_PAGE;
//    int to = (ui_data->page_msgs + 1) * MESSAGES_PER_PAGE;
//    if (to >  ui_data->num_msgs) to =  ui_data->num_msgs;
    callback_button_messages_get(NULL, NULL);
}

/**
 * @brief next page of a messages list
 * @param
 * @return
**/
void callback_button_msgs_page_next(GtkWidget* button, gpointer data)
{
    if (ui_data->page_msgs < ui_data->num_msgs / MESSAGES_PER_PAGE) {
        ui_data->page_msgs++;
    }

// kulakov: unused variables???
//    int from = ui_data->page_msgs * MESSAGES_PER_PAGE;
//    int to = (ui_data->page_msgs + 1) * MESSAGES_PER_PAGE;
//    if (to >  ui_data->num_msgs) to =  ui_data->num_msgs;
    callback_button_messages_get(NULL, NULL);
}

/**
 * @brief switch inbox / sent messages
 * @param
 * @return
**/
void callback_button_msgs(GtkWidget* widget, gpointer user_data)
{
    ui_data->page_msgs = 0;
    if ((int)user_data == 0 && ui_data->msgs_box != INBOX) {
        ui_data->msgs_box = INBOX;
        callback_button_messages_get(NULL, NULL);
    }
    if ((int)user_data == 1 && ui_data->msgs_box == INBOX) {
        ui_data->msgs_box = SENT;
        callback_button_messages_get(NULL, NULL);
    }
}

/**
 * @brief view friend's profile from message form
 * @param
 * @return
**/
void callback_message_profile(GtkWidget* container, gpointer user_data)
{
    Friend* friend = user_data;
    callback_friend(NULL, (gpointer)friend->str_id);    
}


/**
 * @brief click on a friend callback
 * @param
 * @return
**/
void callback_friend(GtkWidget* container, gpointer value)
{
    gchar *request = g_strconcat("<?xml version=\"1.0\"?><TransitData id=\"1\" \
    type=\"data\"><SourceID>ui</SourceID>\
    <TargetID>db</TargetID>\
    <Content><Request class=\"profile\" function = \"getProfile\"><Params id=\"", (gchar*)value, "\"></Params></Request></Content></TransitData>", NULL);
    //hildon_gtk_window_set_progress_indicator(ui_data->window, 1);
    kernel->send(xmlParseDoc(request), NULL, NULL);
    g_free(request);
    ui_init_friend_profile(container, value);
}

/**
 * @brief update messages in database
 * @param
 * @return
**/
void callback_button_messages_update(GtkWidget* button, gpointer data)
{
    gchar* str_msgs_box = NULL;
    if (ui_data->msgs_box == INBOX) {
        str_msgs_box = g_strdup("Inbox");
    } else {
        str_msgs_box = g_strdup("Outbox");
    }
    
    gchar *request = g_strconcat("<?xml version=\"1.0\"?><TransitData id=\"1\" \
    type=\"data\"><SourceID>ui</SourceID>\
    <TargetID>db</TargetID>\
    <Content><Request class=\"messages\" function = \"update", str_msgs_box, "Messages\"><Params></Params></Request></Content></TransitData>", NULL);
    //hildon_gtk_window_set_progress_indicator(ui_data->window, 1);
    kernel->send(xmlParseDoc(request), NULL, NULL);
    g_free(request);
    g_free(str_msgs_box);
}

/**
 * @brief cancel reply on a message callback
 * @param
 * @return
**/
void callback_message_reply_cancel(GtkWidget* button, gpointer data)
{
    gtk_widget_show_all(GTK_WIDGET(ui_data->box_messages));
    gtk_widget_hide(GTK_WIDGET(ui_data->box_message_reply));
}

/**
 * @brief reply on a message callback
 * @param
 * @return
**/
void callback_message_reply_entry(GtkWidget* button_cb, gpointer ptr_msg)
{
    Message* msg = ptr_msg;
    GtkWidget* entry;    
    GtkWindow* window;
    GtkDialog* dialog;

    GtkWidget* vbox = gtk_vbox_new(FALSE, 0); 
    GtkWidget* caption;    
    GtkWidget* label;
    GtkWidget* button;

    int result;      

    dialog = GTK_DIALOG(gtk_dialog_new());
    if(!dialog) {
        return;
    }

    gtk_window_set_title(GTK_WINDOW(dialog), STR_NMSG);
    gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);
    window = GTK_WINDOW(dialog);

    /* buttons */        
    gtk_dialog_add_button(GTK_DIALOG (dialog), STR_CNCL, GTK_RESPONSE_NONE);
    gtk_dialog_add_button(GTK_DIALOG (dialog), STR_OK__, GTK_RESPONSE_ACCEPT);
    entry = hildon_text_view_new ();
    gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (entry), TRUE);
    gtk_text_view_set_editable (GTK_TEXT_VIEW (entry), TRUE);
    //gtk_widget_set_usize(entry, MIN_WIDTH * 2, MIN_HEIGHT * 2);

    gtk_container_add( GTK_CONTAINER( dialog->vbox ), entry);
    gtk_widget_show_all( GTK_WIDGET( dialog ) );

    /* Run dialog cycle */
    result = gtk_dialog_run(dialog);

    if(result == GTK_RESPONSE_ACCEPT ) {
        GtkTextIter start;
        GtkTextIter end;
        GtkTextBuffer* text_buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(entry));
        gtk_text_buffer_get_start_iter(text_buffer, &start);
        gtk_text_buffer_get_end_iter(text_buffer, &end);
        gchar* str_msg = gtk_text_buffer_get_text(text_buffer, &start, &end, FALSE);
        g_object_unref(text_buffer);
        
        CURL* curl = curl_easy_init();
        gchar* str_to_send = curl_easy_escape(curl, str_msg, 0);
        curl_easy_cleanup(curl);
        g_free(str_msg);
        g_print("\n%s\n", str_to_send);
        
        gchar *request = g_strconcat("<?xml version=\"1.0\"?><TransitData id=\"1\" \
        type=\"data\"><SourceID>ui</SourceID><TargetID>vkontakte</TargetID><Content><Request class=\"messages\" function = \"sendMessage\">\
        <Params id=\"", msg->str_sender_id, "\"><string name=\"text\">", str_to_send , "</string></Params></Request></Content></TransitData>", NULL);
        //hildon_gtk_window_set_progress_indicator(ui_data->window, 1);
        kernel->send(xmlParseDoc(request), NULL, NULL);
        g_free(request);
        g_free(str_to_send);
    }
    
    gtk_widget_destroy(GTK_WIDGET(dialog));
}

/**
 * @brief send a message callback
 * @param
 * @return
**/
void callback_message_reply_send(GtkWidget* button, gpointer ptr_msg)
{

}

/**
 * @brief generate xpath request to xmlDocPtr 
 * @param req - xpath req
 * @param doc - xmlDocPtr  
 * @return xmlXPathObject*  
**/
void callback_button_friends_profile_update(GtkWidget* button, gpointer data)
{
    gchar *request = g_strconcat("<?xml version=\"1.0\"?><TransitData id=\"1\" \
    type=\"data\"><SourceID>ui</SourceID>\
    <TargetID>db</TargetID>\
    <Content><Request class=\"profile\" function = \"updateProfile\"><Params id=\"",(gchar*)data,"\"></Params></Request></Content></TransitData>",NULL);
    //hildon_gtk_window_set_progress_indicator(ui_data->window, 1);
    kernel->send(xmlParseDoc(request), NULL, NULL);
    g_free(request);
}

/**
 * @brief join one profile with another
 * @param req - xpath req
 * @param doc - xmlDocPtr  
 * @return xmlXPathObject*  
**/
void callback_button_friends_profile_join(GtkWidget* button, gpointer data)
{
    gchar *str = g_strconcat("\tJOIN PROFILE ", (gchar*)data, " WITH...", NULL);
    g_print(str);
    g_free(str);
    //FIXME
    /*kernel->send(xmlParseDoc(g_strdup(g_strconcat("<?xml version=\"1.0\"?><TransitData id=\"1\" \
    type=\"data\"><SourceID>ui</SourceID>\
    <TargetID>db</TargetID>\
    <Content><Request class=\"profile\" function = \"getProfiles\"><Params/></Request></Content></TransitData>", NULL))), NULL, NULL);*/
    callback_friends_join(NULL, NULL);
}

/**
 * @brief join profiles dialog
 * @param req - xpath req
 * @param doc - xmlDocPtr  
 * @return xmlXPathObject*  
**/
void callback_friends_join(GtkWidget* button_cb, gpointer data)
{    
    GtkWindow* window;
    GtkDialog* dialog;
    GtkWidget* scrollable = gtk_scrolled_window_new(NULL, NULL);
    
    GtkWidget* hbox = gtk_hbox_new(FALSE, 0); 
    GtkWidget* vbox_chck = gtk_vbox_new(TRUE, PADDING);
    GtkWidget* vbox_name = gtk_vbox_new(TRUE, PADDING);
    GtkWidget* vbox_icon = gtk_vbox_new(TRUE, PADDING);
    GtkWidget* caption;    
    GtkWidget* label;
    GtkWidget* button;
    GtkWidget* check;
    GtkWidget* icon;
    GtkWidget* alignment_label;
    GdkPixbufLoader* pbloader;
    
    int result;
    int i;
    
    gchar* str_serv = NULL;
    guchar* str_icon = NULL;

    dialog = GTK_DIALOG(gtk_dialog_new());
    if(!dialog) {
        return;
    }

    gtk_window_set_title(GTK_WINDOW(dialog), g_strconcat(STR_SLCT, " ", STR_PR2J, NULL));
    gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);
    //window = GTK_WINDOW(dialog);

    /* buttons */        
    //button = gtk_button_new_with_label(STR_CNCL);
    //gtk_dialog_add_action_widget( dialog, button, GTK_RESPONSE_CANCEL);
    //button = gtk_button_new_with_label(STR_OK__);
    //gtk_dialog_add_action_widget( dialog, button, GTK_RESPONSE_ACCEPT );
    gtk_dialog_add_button (GTK_DIALOG (dialog), STR_CNCL, GTK_RESPONSE_NONE);
    gtk_dialog_add_button (GTK_DIALOG (dialog), STR_OK__, GTK_RESPONSE_ACCEPT);
    //gtk_container_add( GTK_CONTAINER(dialog), vbox);
    gtk_scrolled_window_add_with_viewport(GTK_WINDOW(scrollable), hbox);
    gtk_scrolled_window_set_policy(scrollable, GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(hbox), vbox_chck, FALSE, TRUE, PADDING);
    gtk_box_pack_start(GTK_BOX(hbox), vbox_name, TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), vbox_icon, FALSE, TRUE, PADDING);
    
    //FIXME
    
    GList* node;
    msa_module* info;
    int size;
    for(node = ui_data->drivers; node != NULL; node = node->next) {
        info = (msa_module*)node->data;
        str_serv = g_strdup(info->name); 
        if (info->pic != NULL){
            str_icon = g_base64_decode((gchar*)info->pic, &size);
        }
    }    
    
    for (i = 0; i < 100; i++) {
        check = gtk_check_button_new();
        gtk_box_pack_start (GTK_BOX (vbox_chck), check, FALSE, FALSE, 0);
        label = gtk_label_new(g_strdup_printf("%s %d", STR_PRFL, i));
        alignment_label = gtk_alignment_new(0, 0.5, 0, 0); 
        gtk_container_add(GTK_CONTAINER(alignment_label), label);
        gtk_box_pack_start (GTK_BOX (vbox_name), alignment_label, FALSE, FALSE, 0);

        pbloader = gdk_pixbuf_loader_new_with_type ("png", NULL);
        gdk_pixbuf_loader_write(pbloader, str_icon, size, NULL);
	GdkPixbuf *pix_orig = gdk_pixbuf_loader_get_pixbuf(pbloader);
        gdk_pixbuf_loader_close (pbloader, NULL);

        int width = gdk_pixbuf_get_width(pix_orig) / 2;
        GdkPixbuf* pixbuf = gdk_pixbuf_scale_simple (pix_orig, width, width, GDK_INTERP_BILINEAR);
        icon = gtk_image_new_from_pixbuf(pixbuf);
        g_object_unref(pix_orig);
        g_object_unref(pixbuf);
        gtk_box_pack_start (GTK_BOX (vbox_icon), icon, FALSE, FALSE, 0);
    }
    
    g_free(str_serv);
    g_free(str_icon);
    
    gtk_widget_set_usize(scrollable, 400, 300);    
    //gtk_widget_set_usize (entry, -1, MIN_HEIGHT);
    gtk_container_add( GTK_CONTAINER( dialog->vbox ), scrollable);
    gtk_widget_show_all( GTK_WIDGET( dialog ) );
    result = gtk_dialog_run(dialog);
    if( result == GTK_RESPONSE_ACCEPT ) {
        //gchar* str = gtk_entry_get_text(GTK_ENTRY(entry));
        //kernel->send(xmlParseDoc(g_strdup(g_strconcat("<?xml version=\"1.0\"?><TransitData id=\"1\" \
        //type=\"data\"><SourceID>ui</SourceID>\
        //<TargetID>db</TargetID>\
        //<Content><Request class=\"friends\" function = \"getListFriends\"><Params Filter=\"",str,"\"/></Request></Content></TransitData>", NULL))), NULL, NULL);
        
    }
    
    gtk_widget_destroy( GTK_WIDGET( dialog ) );
}

/**
 * @brief find friends
 * @param req - xpath req
 * @param doc - xmlDocPtr  
 * @return xmlXPathObject*  
**/
void callback_button_friends_find(GtkWidget* button_cb, gpointer data)
{
    GtkDialog* dialog;
    GtkWidget* vbox = gtk_vbox_new(FALSE, 0); 
    GtkWidget* caption;    
    GtkWidget* label;
    GtkWidget* button;
    GtkWidget* entry;
    int result;      
    dialog = GTK_DIALOG(gtk_dialog_new());
    if(!dialog) {
        return;
    }
    gtk_window_set_title(GTK_WINDOW(dialog), STR_FIND);
    gtk_dialog_add_button(GTK_DIALOG(dialog), STR_CNCL, GTK_RESPONSE_NONE);
    gtk_dialog_add_button(GTK_DIALOG(dialog), STR_FIND, GTK_RESPONSE_ACCEPT);
    entry = hildon_entry_new(HILDON_SIZE_AUTO);
    gtk_container_add(GTK_CONTAINER(dialog->vbox), entry);
    gtk_widget_show_all(GTK_WIDGET(dialog));
    result = gtk_dialog_run(dialog);
    if(result == GTK_RESPONSE_ACCEPT) {
        gchar* str = gtk_entry_get_text(GTK_ENTRY(entry));
        gchar *request = g_strconcat("<?xml version=\"1.0\"?><TransitData id=\"1\" \
        type=\"data\"><SourceID>ui</SourceID>\
        <TargetID>db</TargetID>\
        <Content><Request class=\"friends\" function = \"getListFriends\"><Params Filter=\"",str,"\"/></Request></Content></TransitData>", NULL);
        //hildon_gtk_window_set_progress_indicator(ui_data->window, 1);
        kernel->send(xmlParseDoc(request), NULL, NULL);
        g_free(request);
    }    
    gtk_widget_destroy(GTK_WIDGET(dialog));
}

/**
 * @brief send a message to a friend
 * @param req - xpath req
 * @param doc - xmlDocPtr  
 * @return xmlXPathObject*  
**/
void callback_friend_send_message(GtkWidget* button_cb, gpointer ptr_str_id)
{
    GtkWidget* entry;
    GtkWindow* window;
    GtkDialog* dialog;

    GtkWidget* vbox = gtk_vbox_new(FALSE, 0); 
    GtkWidget* caption;    
    GtkWidget* label;
    GtkWidget* button = ui_init_button_service();

    int result;      

    dialog = GTK_DIALOG(gtk_dialog_new());
    if(!dialog) {
        return;
    }

    gtk_window_set_title(GTK_WINDOW(dialog), STR_NMSG);
    gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);
    window = GTK_WINDOW(dialog);    
    entry = hildon_text_view_new();
    gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(entry), TRUE);
    gtk_text_view_set_editable(GTK_TEXT_VIEW(entry), TRUE);
    //gtk_widget_set_usize(entry, MIN_WIDTH * 2, MIN_HEIGHT * 2);

    gtk_container_add(GTK_CONTAINER(dialog->vbox), entry);
    gtk_container_add(GTK_CONTAINER(dialog->action_area), button);
    gtk_dialog_add_button(GTK_DIALOG(dialog), STR_CNCL, GTK_RESPONSE_NONE);
    gtk_dialog_add_button(GTK_DIALOG(dialog), STR_SEND, GTK_RESPONSE_ACCEPT);
    gtk_widget_show_all(GTK_WIDGET(dialog));

    /* Run dialog cycle */
    result = gtk_dialog_run(dialog);

    if (result == GTK_RESPONSE_ACCEPT) {
        GtkTextIter start, end;
        GtkTextBuffer* text_buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(entry));
        gtk_text_buffer_get_start_iter(text_buffer, &start);
        gtk_text_buffer_get_end_iter(text_buffer, &end);
        gchar* str_msg = gtk_text_buffer_get_text(text_buffer, &start, &end, FALSE);
        g_object_unref(text_buffer);
        CURL* curl = curl_easy_init();
        gchar* str_to_send = curl_easy_escape(curl, str_msg, 0);
        curl_free(curl);
        gchar *request = g_strconcat("<?xml version=\"1.0\"?><TransitData id=\"1\" \
        type=\"data\"><SourceID>ui</SourceID><TargetID>vkontakte</TargetID><Content><Request class=\"messages\" function = \"sendMessage\">\
        <Params id=\"",(gchar*)ptr_str_id, "\"><string name=\"text\">", str_to_send , "</string></Params></Request></Content></TransitData>", NULL);
        //hildon_gtk_window_set_progress_indicator(ui_data->window, 1);
        kernel->send(xmlParseDoc(request), NULL, NULL);
        g_free(request);
        g_free(str_to_send);
        g_free(str_msg);
    }
    gtk_widget_destroy(GTK_WIDGET(dialog));
}

/**
 * @brief generate xpath request to xmlDocPtr 
 * @param req - xpath req
 * @param doc - xmlDocPtr  
 * @return xmlXPathObject*  
**/
void callback_button_friends(GtkWidget* button, gpointer data)
{
    gtk_widget_hide(GTK_WIDGET(ui_data->box_friend));
    gtk_widget_show_all(GTK_WIDGET(ui_data->box_friends));
    if (ui_data->friends_filter) {
        gtk_widget_hide(GTK_WIDGET(ui_data->box_friends_navi));
    } else {
        gtk_widget_hide(GTK_WIDGET(ui_data->box_friends_result));
    }
}

/**
 * @brief get friend's profile from database 
 * @param req - xpath req
 * @param doc - xmlDocPtr  
 * @return xmlXPathObject*  
**/
void callback_button_friend_get(GtkWidget* button, gpointer data)
{
    gchar *request = g_strconcat("<?xml version=\"1.0\"?><TransitData id=\"1\" \
        type=\"data\"><SourceID>ui</SourceID>\
        <TargetID>db</TargetID>\
        <Content><Request class=\"profile\" function = \"getProfile\"><Params id=\"", (gchar*)data, "\"></Params></Request></Content></TransitData>", NULL);
    //hildon_gtk_window_set_progress_indicator(ui_data->window, 1);
    kernel->send(xmlParseDoc(request), NULL, NULL);
    g_free(request);
}

/**
 * @brief update profile
 * @param req - xpath req
 * @param doc - xmlDocPtr  
 * @return xmlXPathObject*  
**/
void callback_button_profile_update(GtkWidget* button, gpointer data)
{
    //hildon_gtk_window_set_progress_indicator(ui_data->window, 1);
    kernel->send(xmlParseDoc("<?xml version=\"1.0\"?><TransitData id=\"1\" \
    type=\"data\"><SourceID>ui</SourceID>\
    <TargetID>db</TargetID>\
    <Content><Request class=\"profile\" function = \"updateProfile\"><Params></Params></Request></Content></TransitData>"), NULL, NULL);
}

/**
 * @brief generate xpath request to xmlDocPtr 
 * @param req - xpath req
 * @param doc - xmlDocPtr  
 * @return xmlXPathObject*  
**/
void callback_button_profile_save(GtkWidget* button, gpointer data)
{
    xmlNodePtr node;
    /** treansit request to driver **/
    xmlDocPtr req_drv = xmlCopyDoc(ui_data->ui_request,1);
    /** change name node, class, function **/
    xmlXPathObject* obj = xpath("//TransitData/Content/Response", req_drv);
    if (obj->nodesetval->nodeNr != 0) {
        node = obj->nodesetval->nodeTab[0];
        xmlNodeSetName(node, "Request");
        xmlSetProp(node, "class", PROFILE);
        xmlSetProp(node, "function", SET_PROFILE);
    }
    else {
        g_debug("callback_button_profile_save: save_request error response obj is null");    
    }
// !!!    
   	xmlXPathFreeObject(obj);	
    obj = NULL;
    
    /** change source id **/
    obj = xpath(SOURCE_TAG_XPATH, req_drv);
    if (obj->nodesetval->nodeNr != 0) {
        node = obj->nodesetval->nodeTab[0];
        xmlNodeSetContent(node, UI_ID);
    }
    else {
        g_debug("callback_button_profile_save: save_request error source obj is null");    
    }
    
// !!!    
   	xmlXPathFreeObject(obj);	
    obj = NULL;
    
    
    /** create target id **/
    obj = xpath(TARGET_TAG_XPATH, req_drv);
    if (obj->nodesetval->nodeNr != 0) {
        node = obj->nodesetval->nodeTab[0];
        xmlNodeSetContent(node, "vkontakte");
    }
    else {
        g_debug("callback_button_profile_save: save_request error source obj is null");
        //FIXME
        node = xmlNewDocNode(req_drv, NULL, TARGET_NODE, "vkontakte");
        xmlNodePtr root = xmlDocGetRootElement(req_drv);
        xmlAddChild(root, node);    
    }
    
// !!!    
   	xmlXPathFreeObject(obj);	
    obj = NULL;

    
    kernel->send(req_drv, NULL, NULL);
    if (ui_data->ui_request != NULL) {
        xmlFreeDoc(ui_data->ui_request);
        ui_data->ui_request = NULL;
    }
}

/**
 * @brief save profile detail
 * @param container - container widget  
**/
void callback_button_profile_save_detail(GtkWidget* container)
{
    GList* children = gtk_container_get_children(container);
    for (; children != NULL; children = children->next){
        gtk_widget_realize(children->data);
        gtk_widget_show_now(children->data);
        gtk_widget_show_all(children->data);
        GList* subchildren;
        if (GTK_IS_CONTAINER(children->data)) {
            subchildren = gtk_container_get_children (GTK_CONTAINER(children->data));
            if(subchildren != NULL)
                callback_button_profile_save_detail(children->data);
        } else {
            gchar* label = gtk_widget_get_name(GTK_WIDGET(children->data));
            gchar* value;
            if (GTK_IS_LABEL(children->data))
                value = gtk_label_get_text(GTK_WIDGET(children->data));
        }
    }
}

/**
 * @brief callback on setup service button pressed
 * @param button - pressed button
 * @param id_service _ service id    
**/
void callback_button_setup_service(GtkWidget* button, gpointer id_service)
{
    ui_data->srvc = (int)id_service;
    ui_init_driver_settings(NULL);
    GList* node;
    int i = 0;
    msa_module* info;
    for(node = ui_data->drivers; node != NULL; node = node->next) {
        if (i == (int)id_service){
            info = (msa_module*)node->data;
            gchar *request = g_strconcat("<?xml version=\"1.0\"?><TransitData id=\"1\" \
    type=\"data\"><SourceID>ui</SourceID>\
    <TargetID>",info->id,"</TargetID>\
    <Content><Request class=\"settings\" function = \"getSettings\"><Params></Params></Request></Content></TransitData>", NULL);
            //hildon_gtk_window_set_progress_indicator(ui_data->window, 1);
            kernel->send(xmlParseDoc(request), NULL, NULL);
            g_free(request);
            break;
        }
        i++;
    }
    g_print("setting up service #%d\n", id_service);
}

/**
 * @brief driver settings dialog
**/
void callback_driver_settings(gpointer data)
{
    GtkDialog* dialog;
    GtkWidget* label;
    GtkWidget* button;
    int result;      
    dialog = GTK_DIALOG(gtk_dialog_new());
    if(!dialog) {
        return;
    }
    gtk_window_set_title(GTK_WINDOW(dialog), STR_SETS);
    gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);
    gtk_dialog_add_button (GTK_DIALOG (dialog), STR_CNCL, GTK_RESPONSE_NONE);
    gtk_dialog_add_button (GTK_DIALOG (dialog), STR_SAVE, GTK_RESPONSE_ACCEPT);
    gtk_container_add(GTK_CONTAINER(dialog->vbox), ui_data->box_setting);
    gtk_widget_show_all(GTK_WIDGET(dialog));
    /* Run dialog cycle */
    result = gtk_dialog_run(dialog);
    if( result == GTK_RESPONSE_ACCEPT ) {
        callback_button_save_setup_service(NULL, ui_data->srvc);
    }
    gtk_widget_destroy(GTK_WIDGET(dialog));
}

/**
 * @brief cancel edit
 * @param button - button widget
 * @param data - pointer to data
**/
void callback_button_cancel_edit(GtkWidget* button, gpointer data)
{
    callback_goto(NULL, ui_data->prev);
}

/**
 * @brief cancel setuo service 
 * @param button - button widget
 * @param id_service - number of service id
 * @return xmlXPathObject*  
**/
void callback_button_cancel_setup_service(GtkWidget* button, int id_service)
{
    callback_goto(NULL, ui_data->prev);
}

/**
 * @brief generate xpath request to xmlDocPtr 
 * @param req - xpath req
 * @param doc - xmlDocPtr  
 * @return xmlXPathObject*  
**/
void callback_button_save_setup_service(GtkWidget* button, gpointer id)
{
    GList* children = gtk_container_get_children (ui_data->box_profile);
    for (; children != NULL; children = children->next){
        gtk_widget_destroy(children->data); 
    }
    ui_data->empty_profile = TRUE;
    children = gtk_container_get_children (ui_data->box_friends_list);
    for (; children != NULL; children = children->next){
        gtk_widget_destroy(children->data); 
    }
    ui_data->empty_friends = TRUE;
    
    children = gtk_container_get_children (ui_data->box_messages_list);
    for (; children != NULL; children = children->next){
        gtk_widget_destroy(children->data); 
    }
    ui_data->empty_messages = TRUE;
    GList* node;
    int i = 0;
    msa_module* info;
    for(node = ui_data->drivers; node != NULL; node = node->next) {
        if (i == ui_data->srvc){
            info = (msa_module*)node->data;
            char* actv;
            actv = g_strdup("1");
            info->status = ON;
            info->state = CONF;
            //hildon_gtk_window_set_progress_indicator(ui_data->window, 1); 
            kernel->msa_settings(info->id, xmlParseDoc(g_strconcat("<?xml version=\"1.0\"?><TransitData><SourceID>ui</SourceID><TargetID>",info->id,"</TargetID><Content><Request class=\"settings\" function=\"setSettings\"><Params><string name=\"login\">",gtk_entry_get_text(GTK_LABEL(ui_data->entry_logn)),"</string><string name=\"password\">",gtk_entry_get_text(GTK_LABEL(ui_data->entry_pass)),"</string><number name=\"autoupdate\">",g_strdup_printf("%d",(hildon_number_editor_get_value(ui_data->entry_auto))),"</number><boolean name=\"active\">",actv,"</boolean></Params></Request></Content></TransitData>",NULL)));
            break;
        }
        i++;
    }
}

/**
 * @brief generate xpath request to xmlDocPtr 
 * @param req - xpath req
 * @param doc - xmlDocPtr  
 * @return xmlXPathObject*  
**/
void callback_label_profile_edit(GtkWidget* label, gpointer data)
{
    gchar *text = gtk_label_get_text(GTK_LABEL(label));
    g_free(text);
}

/**
 * @brief generate xpath request to xmlDocPtr 
 * @param req - xpath req
 * @param doc - xmlDocPtr  
 * @return xmlXPathObject*  
**/
void callback_empty(GtkWidget* widget, gpointer data)
{
    Message* msg = data;
    hildon_banner_show_information(NULL, NULL, (gchar*)msg->str_text ); 
}

/**
 * @brief requests and titles on switching tasks 
 * @param notebook - tabs
 * @param page - current page
 * @param page_num - current page number 
**/
void callback_task_switch(GtkWidget* notebook, GtkNotebookPage* page, guint page_num, gpointer user_data)
{
    switch(page_num) {
    case 0:
        callback_button_profile_get(NULL, NULL);
        gtk_window_set_title (GTK_WINDOW (ui_data->window), g_strconcat(STR_PROG, STR_TTLS, STR_PRFL, NULL));
        break;
    case 1:
        callback_button_friends_get(NULL, NULL);
        gtk_window_set_title (GTK_WINDOW (ui_data->window), g_strconcat(STR_PROG, STR_TTLS, STR_FRDS, NULL));
        break;
    case 2:
        callback_button_messages_get(NULL, NULL);
        gtk_window_set_title (GTK_WINDOW (ui_data->window), g_strconcat(STR_PROG, STR_TTLS, STR_MSGS, NULL));
        break;
    case 3:
        gtk_window_set_title (GTK_WINDOW (ui_data->window), g_strconcat(STR_PROG, STR_TTLS, STR_SETS, NULL));
        break;
    }
}

/**
 * @brief generate xpath request to xmlDocPtr 
 * @param req - xpath req
 * @param doc - xmlDocPtr  
 * @return xmlXPathObject*  
**/
void callback_check_service(GtkWidget* widget, gpointer data)
{
    g_debug("checked\n");
}

/**
 * @brief generate xpath request to xmlDocPtr 
 * @param req - xpath req
 * @param doc - xmlDocPtr  
 * @return xmlXPathObject*  
**/
void callback_exit(GtkAction* action, gpointer data)
{
    kernel->shutdown();
    gtk_main_quit();
}