/*
 * repository.c - store for kernel transit data
 * This file is part of MSA program
 *
 * Copyright (C) 2009 - Alexander A. Lomov
 *
 * MSA program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MSA program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with MSA program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, 
 * Boston, MA  02110-1301  USA
 */
 
 // $id$

#include "repository.h"

/* Doubly-Linked Lists for data, and define a Lock for thread-safe */
static GList* repository = NULL;
G_LOCK_DEFINE_STATIC(repository);

/* Repositore data element */
typedef struct repository_element
{
    time_t time;
    transit_data* data;
} repository_element;


static void free_repository_element(repository_element* element);


/**************************************************/
/*************** External functions ***************/

/**
 * Gets the number of elements in a repository.
 *
 * @return number of elements in the repository.
 */
guint repository_get_length()
{
    G_LOCK(repository);
    guint length = g_list_length(repository);
    G_UNLOCK(repository);

    return length;
}


/** 
 * Add data to repository. 
 *
 * @param data transit data for save.
 *
 * @return 0 on success and not 0 otherwise.
 */    
gint repository_add_data(transit_data* data)
{
    g_debug("repository_add_data\n");
    repository_element* element = g_try_new(repository_element, 1);
    
    if(element == NULL)
    {
	    return ERROR_ALLOCATION_OF_MEMORY;
    }
    g_debug("repository_add_data - create alement\n");

    element->time = time(NULL);
    g_debug("repository_add_data - create time\n");
    element->data = data;

    G_LOCK(repository);
    repository = g_list_append(repository, element);
    G_UNLOCK(repository);

    return SUCCESS;
}
    

/**
 * Gets last added data, and if need remove it from list.
 *
 * @param need_remove remove flag.
 *
 * @return transit data from repository if has it or NULL otherwise.
 */
transit_data* repository_get_newest_transit_data(gboolean need_remove)
{
    G_LOCK(repository);
    GList* node = g_list_last(repository);

    if(node == NULL)
    {
        G_UNLOCK(repository);
        return NULL;
    }

    repository_element* element = node->data;
    transit_data* tr_data = element->data;

    if(need_remove == TRUE)
    {
        free_repository_element(element);

        node->data = NULL;
        repository = g_list_delete_link(repository, node);
        g_list_free_1(node);
    }
    G_UNLOCK(repository);
    return tr_data;
}

/**
 * Gets data by index, and if need remove it from list.
 *
 * @param index index of data, elemets start from 0.
 * @param need_remove remove flag.
 *
 * @return transit data from repository if has it or NULL otherwise.
 */
transit_data* repository_get_transit_data_by_index(const guint index, 
                                                   const gboolean need_remove)
{
    G_LOCK(repository);
    
    GList* node = g_list_nth(repository, (guint)(index));
    
    if(node == NULL)
    {
        G_UNLOCK(repository);
        return NULL;
    }
    
    repository_element* element = node->data;
    transit_data* tr_data = element->data;
 
    if(need_remove == TRUE)
    {
        free_repository_element(element);

        node->data = NULL;
        repository = g_list_remove_link(repository, node);
        g_list_free_1(node);
    }
    G_UNLOCK(repository);
    
    return tr_data; 
}


/**
 * Gets oldest added data, and if needed remove it from list.
 *
 * @param need_remove remove flag.
 *
 * @return transit data from repository if has it or NULL otherwise.
 */
transit_data* repository_get_oldest_transit_data(gboolean need_remove)
{
    
    G_LOCK(repository);
    GList* node = g_list_first(repository);

    if(node == NULL)
    {
        G_UNLOCK(repository);
        return NULL;
    }

    repository_element* element = node->data;
    transit_data* tr_data = element->data;

    if(need_remove == TRUE)
    {
        free_repository_element(element);
    
        node->data = NULL;
        repository = g_list_delete_link(repository, node);
    }
    G_UNLOCK(repository);
    return tr_data;
}


/**
 * Free resources allocated for repository.
 *
 */
void repository_free()
{
    g_debug("repository_free START");
    
    GList* list_walker = repository;
    
    while(list_walker != NULL)
    {
        free_repository_element((repository_element*)list_walker->data);
        list_walker->data = NULL;
        list_walker = g_list_next(list_walker);
    }
        
    g_list_free(repository);
    repository = NULL;
    g_debug("repository_free END");    
}


/**************************************************/
/***************** Satic functions ****************/

/**
 * Free resources allocated for repository data element.
 *
 * @param repository data element.
 */
static void free_repository_element(repository_element* element)
{
    element->data = NULL;
    g_free(element);    
}


/*****************************************************/
/*** Not used function (maybe will be used later) ****/

/**
 * Checks the equality of the target_id in transit_data and given id
 *
 * @param repository_element element from repository
 * @param id given identifier for compare
 * @return compare result: -1 - not equal, 0 - equal
 */
/*
static gint compare_function_by_id(repository_element* element, gchar* id)
{
    return (g_strrstr(element->data->target_id,  id) == NULL) ? -1 : 0; 
    return -1;
}
*/



