/*
 * handler.c - manage signals
 * This file is part of MSA program
 *
 * Copyright (C) 2009 - Alexander A. Lomov
 *
 * MSA program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MSA program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with MSA program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, 
 * Boston, MA  02110-1301  USA
 */
 
 // $id$

#include "handler.h"
#include "kernel.h"

/* Store for signals */
static GSList* signal_store = NULL;
G_LOCK_DEFINE_STATIC(signal_store);

/* Define: signals processing is in process */
static gboolean is_process_proceed = FALSE;
G_LOCK_DEFINE_STATIC(is_process_proceed);

/* Define signal */
typedef struct signal
{
    xmlChar* id;
    xmlChar* source_id;
    GTimeVal time;
    glong start_time;
    glong repeat_time;
    xmlDocPtr data;
} signal_data;

/* Actions for signal */
typedef enum actions
{
    NOT_SET,
    SET_SIGNAL,
    REMOVE_SIGNAL
} action_type;


/* Functions for work with xml */
static gboolean check_node_name(xmlNodePtr node, xmlChar* name);
static xmlDocPtr create_xml_doc_with_root(xmlNodePtr root_node);
static xmlDocPtr get_inner_content(const xmlDocPtr doc, xmlChar* node_name);
static xmlNodePtr find_sibling_node_by_name(const xmlNodePtr start_node, 
                                            const xmlChar* find_name);
static gint convert_node_content_to_int(const xmlNodePtr node);

/* Functions for work with store */
static void free_signal_data(signal_data* signal);
static void remove_from_signal_store_by_id_and_source(const xmlChar* id, 
                                                      const xmlChar* source_id);
static gint add_to_signal_store(signal_data* signal);
static gint sorting_by_time(signal_data* a, signal_data* b);

/* Function for work with signals */
static void processing();
static void set_time(signal_data* signal);
static void clear_signal_store();
static action_type check_action_type(xmlChar* type);
static signal_data* convert_to_handler_format(xmlDocPtr doc, 
                                              action_type* action);


//#define INCLUDE_HANDLER_TEST_FUNC 1
//#ifdef INCLUDE_HANDLER_TEST_FUNC
//#include "../../test/kernel_test/handler_test_func.c"
//#endif

/**************************************************/
/*************** External functions ***************/

/**
 * Initialization of handler.
 *
 * @return 0 on success and not 0 value otherwise.
 */
gint handler_initialization()
{
    if(!g_thread_supported()) 
	{
	    g_thread_init(NULL);
	}

    return SUCCESS;   
}


/**
 * Shutdown handler.
 *
 */
void handler_shutdown()
{
    g_debug("Handler shutdown START");
    G_LOCK(is_process_proceed);
    is_process_proceed = FALSE;
    G_UNLOCK(is_process_proceed);
    clear_signal_store();
    g_debug("Handler shutdown END");
}


/**
 * Receives the data describing the signal.
 *
 * @param data xml text describing the data.
 *
 * @return 0 on success and not 0 value otherwise.
 */
gint handler_put_data(xmlChar* data)
{
    g_debug("handler_put_data START");
    xmlDocPtr doc = xmlParseDoc(data);    
    xmlFree(data);
    data = NULL;
     
    if(doc == NULL)
    {
        return ERROR_CANT_PARSE_DATA;
    }    
    //xmlDocDump(stdout, doc);    
   
    action_type action = NOT_SET;   
    signal_data* signal = convert_to_handler_format(doc, &action);
    

    xmlFreeDoc(doc);

    if (signal == NULL) {
        return FAILURE;
    }
    
    set_time(signal);
    
    //xmlDocDump(stdout, signal->data);
    
    if (signal->id != NULL && signal->source_id != NULL) {
        remove_from_signal_store_by_id_and_source(signal->id, 
                                                  signal->source_id);
    }           
    
    if(action == REMOVE_SIGNAL || signal == NOT_SET)
    {
        free_signal_data(signal);
        return SUCCESS;
    }   
                                           
    add_to_signal_store(signal);

    G_LOCK(is_process_proceed);
    if(is_process_proceed == FALSE)
    {
        is_process_proceed = TRUE;
        g_thread_create((GThreadFunc)processing, NULL, TRUE, NULL);        
    }
    G_UNLOCK(is_process_proceed);

    g_debug("handler_put_data END");

    return SUCCESS;
}

/**************************************************/
/***************** Satic functions ****************/

/**
 * Check node name. 
 *
 * @param node xml node.
 * @param name name for check.
 *
 * @return TRUE if node name and given name are equals or FALSE otherwise.
 */
static gboolean check_node_name(xmlNodePtr node, xmlChar* name)
{
    if(node == NULL || name == NULL)
    {
        return FALSE;
    }
    
    if(xmlStrcmp(node->name, name) != 0)
    {
        return FALSE;
    }
    
    return TRUE;
}


/**
 * Create new xml document and set given node as root. 
 *
 * @param root_node root node for new xml document.
 *
 * @return new xml document on success or NULL.
 */
static xmlDocPtr create_xml_doc_with_root(xmlNodePtr root_node)
{
    g_debug("create_xml_doc_with_root START");
    xmlDocPtr content_doc = xmlNewDoc(BAD_CAST XML_VERSION);

    if(content_doc == NULL)
    {
        g_debug("No content doc");
        return NULL;
    }
   
    xmlNodePtr content = xmlCopyNodeList(root_node);
    
    if(content == NULL)
    {
        g_debug("No content");
        xmlFreeDoc(content_doc);
        return NULL;
    }
    xmlDocSetRootElement(content_doc, content);       
    
    g_debug("create_xml_doc_with_root END");
    return content_doc;
}


/**
 * Get inner context and try to make new xml document from it. 
 *
 * @param doc xml document.
 * @param node_name name of node, from which take the contents.
 *
 * @return new xml document on success or NULL.
 */
static xmlDocPtr get_inner_content(const xmlDocPtr doc, xmlChar* node_name)
{
    g_debug("get_inner_content START");
    xmlNodePtr root_node = xmlDocGetRootElement(doc);
    
    if(root_node == NULL)
    {
        g_debug("get_inner_content: no root");
        return NULL;
    }
    
   xmlNodePtr node_walker = root_node->children;
    
    while(node_walker != NULL)
    {
        g_debug("name %s = %s", node_walker->name, node_name);

        if(xmlStrcmp(node_walker->name, node_name) == 0)
        {
            xmlDocPtr new_doc = create_xml_doc_with_root(node_walker->children);
            //xmlDocDump(stdout, new_doc);
            return new_doc;
        }
        node_walker = node_walker->next;
    }
    g_debug("get_inner_content END");
    return NULL;
}


// Remove from hanler later!
/**
 * Passes through sibling nodes and looks for node with given name.
 *
 * @param start_node node for start search.
 * @param find_name name of node.
 *
 * @return first node with given name or NULL if not found.
 */
static xmlNodePtr find_sibling_node_by_name(const xmlNodePtr start_node, 
                                            const xmlChar* find_name)
{
    g_message("find_sibling_node_by_name starts");

    xmlNodePtr node_walker = start_node;

    while(node_walker)
    {
        if(!xmlStrcmp(node_walker->name, find_name))
        {
            g_message("find_sibling_node_by_name; node db_get_table_walker:%s", 
                      node_walker->name);
            return node_walker;
        }
        node_walker = node_walker->next;
    }
    
    node_walker = start_node->prev;
    
    while(node_walker)
    {
        if(!xmlStrcmp(node_walker->name, find_name))
        {
             g_message("find_sibling_node_by_name; node_walker:%s", 
                       node_walker->name);
            return node_walker;
        }
  
        node_walker = node_walker->prev;
    }
    
    g_debug("find_sibling_node_by_name END");
    return NULL;
}


/**
 * Gets a node content and try to convert it to integer value. 
 *
 * @param node xml node.
 *
 * @return integer value on success or -1.
 */
static gint convert_node_content_to_int(const xmlNodePtr node)
{
    g_debug("convert_node_context_to_int START");
    xmlChar* context = xmlNodeGetContent(node);
    
    g_debug("Node %s context %s", node->name    , context);
    glong result = (context != NULL) ? (glong)strtod((char*)context, NULL) : -1;
    
    xmlFree(context);
    
    g_debug("convert_node_context_to_int END");
    return result;
} 


/**
 * Remove all elemets from store and free memory.
 *
 */
static void clear_signal_store()
{
    GSList* list_walker = signal_store;

    G_LOCK(signal_store);
    while(list_walker != NULL)
    {
        signal_data* signal = (signal_data*)list_walker->data;        
        free_signal_data(signal);
        signal = NULL;
        list_walker = (GSList*)g_slist_next(list_walker);
    }
    g_slist_free(signal_store);
    signal_store = NULL;
        
    G_UNLOCK(signal_store);    
}


/**
 * Remove signal from store with given id and source id. 
 *
 * @param id signal's id.
 * @param source id of source. 
 */
static void remove_from_signal_store_by_id_and_source(const xmlChar* id, 
                                                      const xmlChar* source_id)
{
    GSList* list_walker = signal_store;
    G_LOCK(signal_store);
    while(list_walker != NULL)
    {
        signal_data* signal = (signal_data*)list_walker->data;        
        
        if((xmlStrcmp(signal->id, id) == 0) 
           && (xmlStrcmp(signal->source_id, source_id) == 0))
        {
            free_signal_data(signal);
            signal = NULL;
            signal_store = g_slist_remove_link(signal_store, list_walker);
            g_slist_free_1(list_walker);
            break;
        }
        
        list_walker = (GSList*)g_slist_next(list_walker);
    }
    G_UNLOCK(signal_store);
}


/**
 * Add signal to store. 
 *
 * @param signal signal structure.
 *
 * @return 0 on success or not 0 otherwise.
 */
static gint add_to_signal_store(signal_data* signal)
{
    if(signal == NULL)
    {
        return FAILURE;
    }   

    G_LOCK(signal_store);
    
    signal_store = g_slist_insert_sorted(signal_store, (gpointer)signal, 
                                         (GCompareFunc)sorting_by_time);
	G_UNLOCK(signal_store);
	
	return SUCCESS;
}


/**
 * Function using for signal's list sort. Using time for sort. 
 *
 * @param a first signal for compare.
 * @param b second signal for compare.
 *
 * @return -1 when the first signal will be called before second, 
 *         1 - when the second signal will be called before first,
 *         0 - call time is equal
 */
static gint sorting_by_time(signal_data* a, signal_data* b)
{
    if(a->time.tv_sec < b->time.tv_sec)
    {
        return -1;
    }       
    else if(a->time.tv_sec > b->time.tv_sec)
    {
        return 1;
    }
    else 
    {
        return 0;
    }
}


/**
 * Free resources allocated for signas structre.
 *
 * @param signal signal structure.
 */
static void free_signal_data(signal_data* signal)
{
    if(signal == NULL)
    {
        return;
    }

    xmlFree(signal->id);
    xmlFree(signal->source_id);
    xmlFreeDoc(signal->data);

    signal->id = NULL;
    signal->source_id = NULL;
    signal->data = NULL;
    
    g_free(signal);
}


/**
 * Check action type for signal. 
 *
 * @param type 	text describing action.
 *
 * @return action_type.
 */
static action_type check_action_type(xmlChar* type)
{
    action_type action = NOT_SET;    
    
    if(xmlStrcmp(type, BAD_CAST REMOVE_SIGNAL_TEXT) == 0)
    {
        action = REMOVE_SIGNAL;
    }
    else if(xmlStrcmp(type, BAD_CAST SET_SIGNAL_TEXT) == 0)
    {
        action = SET_SIGNAL;
    }
    
    return action;
}


/**
 * Convert xml signal data into handler format. 
 *
 * @param doc xml document.
 * @param action function sets action type. Used outside. 
 *
 * @return new signal on success or NULL otherwise.
 */
static signal_data* convert_to_handler_format(xmlDocPtr doc, 
                                              action_type* action)
{
    g_debug("convert_to_handler_format START");
    xmlNodePtr root_node = xmlDocGetRootElement(doc);
    xmlDocPtr inner_content = get_inner_content(doc, BAD_CAST CONTENT_TAG_NAME);
    xmlNodePtr inner_root_node = xmlDocGetRootElement(inner_content);
    
    signal_data* signal = g_try_new(signal_data, 1);
    
    if(root_node == NULL || signal == NULL || inner_content == NULL 
       || inner_root_node == NULL)
    {
        g_free(signal);
        xmlFreeDoc(inner_content);
        return NULL;
    }

    if(check_node_name(inner_root_node, BAD_CAST SIGNAL_TAG_NAME) == FALSE)
    {
        g_free(signal);
        xmlFreeDoc(inner_content);
        return NULL;
    }
        
    xmlNodePtr tmp_node = find_sibling_node_by_name(root_node->children, 
                                                    BAD_CAST SOURCE_TAG_NAME);
    signal->source_id = xmlNodeGetContent(tmp_node);	
    
    tmp_node = find_sibling_node_by_name(inner_root_node->children, 
                                         BAD_CAST CONTENT_TAG_NAME);
    if (tmp_node != NULL) {
        signal->data = create_xml_doc_with_root(tmp_node->children);
    } else {
        signal->data = NULL;
    }    

    tmp_node = find_sibling_node_by_name(inner_root_node->children, 
                                         BAD_CAST START_TIME_TAG_NAME);
    signal->start_time = convert_node_content_to_int(tmp_node);
    
    tmp_node = find_sibling_node_by_name(inner_root_node->children, 
                                          BAD_CAST REPEAT_TIME_TAG_NAME);
    signal->repeat_time = convert_node_content_to_int(tmp_node);

    tmp_node = find_sibling_node_by_name(inner_root_node->children, 
                                         BAD_CAST ACTION_TAG_NAME);
    (*action) = check_action_type(xmlNodeGetContent(tmp_node));	
    
    signal->id = xmlGetProp(inner_root_node, BAD_CAST SIGNAL_ATTR_ID);
    
    xmlFreeDoc(inner_content);
    
    if(signal->data == NULL && (*action) == SET_SIGNAL)
    {   
        g_debug("convert_to_handler_format; NO SIGNAL DATA");
        free_signal_data(signal);
        return NULL;
    }
    g_debug("convert_to_handler_format END");
    return signal;
}

 
/**
 * Sets new time for call for signal.
 *
 * @param signal signal structure
 */
static void set_time(signal_data* signal)
{
    GTimeVal time;
    g_get_current_time(&time);
    
    if(signal->start_time >= 0)
    {
        time.tv_sec += signal->start_time ;
//        g_time_val_add(&time, signal->start_time * G_USEC_PER_SEC);
        signal->start_time = -1;  
        signal->time = time;
    }
    else if(signal->repeat_time > 0)
    {
        time.tv_sec += signal->repeat_time ;
//        g_time_val_add(&time, signal->repeat_time);// * G_USEC_PER_SEC);
        signal->time = time;
    }
    else
    {
        signal->time = time;
    }
}


/**
 * Get singnal from sotore and manage it. 
 *
 */
static void processing()
{
    GTimeVal time;    
    gint interval = 0;  
    
    while(g_slist_length(signal_store) > 0 && is_process_proceed == TRUE)
    {
        g_debug("signal_store len = %i", g_slist_length(signal_store));

        G_LOCK(signal_store);
        GSList* elem = g_slist_nth(signal_store, 0);         

        g_get_current_time(&time);

        signal_data* signal = (signal_data*)elem->data;

        interval = signal->time.tv_sec - time.tv_sec;
        G_UNLOCK(signal_store);
        if (interval > 0 ) {
            g_debug("Interval %i ", interval);
            g_usleep(G_USEC_PER_SEC * interval);        
	    continue;
        }
        
        if(is_process_proceed == FALSE)
        {
            break;    
        }
        
        G_LOCK(signal_store);
        kernel_put_data(xmlCopyDoc(signal->data, 1));
        set_time(signal);
        
        
        signal_store = g_slist_sort(signal_store,
                                   (GCompareFunc)sorting_by_time);
    
        if(signal->start_time <= -1 && signal->repeat_time <= 0)
        {
            g_message("\nSIGNAL_NAME_R = %s\n", signal->source_id);
            free_signal_data(signal);
            signal = NULL;
            signal_store = g_slist_remove_link (signal_store, elem);
            g_slist_free_1(elem);
        }
        G_UNLOCK(signal_store);
    }

    G_LOCK(is_process_proceed);
    is_process_proceed = FALSE;
    G_UNLOCK(is_process_proceed);
}
