/*
 * Copyright (C) 2015 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "plugintrack.h"
#include "pluginmanager.h"
#include "resources.h"

PluginTrack::PluginTrack(QObject *parent) :
    MKTrack(parent),
    m_request(0)
{
}

PluginTrack::PluginTrack(const QString &service, const QString &id, QObject *parent) :
    MKTrack(parent),
    m_request(0)
{
    loadTrack(service, id);
}

PluginTrack::PluginTrack(const QString &service, const QVariantMap &track, QObject *parent) :
    MKTrack(parent),
    m_request(0)
{
    loadTrack(service, track);
}

PluginTrack::PluginTrack(const PluginTrack *track, QObject *parent) :
    MKTrack(track, parent),
    m_request(0),
    m_commentsId(track->commentsId()),
    m_relatedTracksId(track->relatedTracksId())
{
}

QString PluginTrack::commentsId() const {
    return m_commentsId;
}

void PluginTrack::setCommentsId(const QString &i) {
    if (i != commentsId()) {
        m_commentsId = i;
        emit changed();
        emit commentsIdChanged();
    }
}

QString PluginTrack::relatedTracksId() const {
    return m_relatedTracksId;
}

void PluginTrack::setRelatedTracksId(const QString &i) {
    if (i != relatedTracksId()) {
        m_relatedTracksId = i;
        emit changed();
        emit relatedTracksIdChanged();
    }
}

QString PluginTrack::errorString() const {
    return m_request ? m_request->errorString() : QString();
}

ResourcesRequest::Status PluginTrack::status() const {
    return m_request ? m_request->status() : ResourcesRequest::Null;
}

void PluginTrack::loadTrack(const QString &service, const QString &id) {
    setService(service);
    setId(id);
    
    if (status() == ResourcesRequest::Loading) {
        return;
    }
    
    if (ResourcesRequest *r = request()) {
        r->get(Resources::TRACK, id);
        emit changed();
        emit statusChanged(status());
    }
}

void PluginTrack::loadTrack(const QString &service, const QVariantMap &track) {
    setService(service);
    setArtist(track.value("artist").toString());
    setArtistId(track.value("artistId").toString());
    setCommentsId(track.value("commentsId").toString());
    setDate(track.value("date").toString());
    setDescription(track.value("description").toString());
    setDownloadable(track.value("downloadable", true).toBool());
    setFormat(track.value("format").toString());
    setGenre(track.value("genre").toString());
    setId(track.value("id").toString());
    setLargeThumbnailUrl(track.value("largeThumbnailUrl").toString());
    setPlayCount(track.value("playCount").toLongLong());
    setRelatedTracksId(track.value("relatedTracksId").toString());
    setStreamUrl(track.value("streamUrl").toString());
    setThumbnailUrl(track.value("thumbnailUrl").toString());
    setTitle(track.value("title").toString());
    setUrl(track.value("url").toString());
        
    if (track.value("duration").type() == QVariant::String) {
        setDurationString(track.value("duration").toString());
    }
    else {
        setDuration(track.value("duration").toLongLong());
    }
    
    if (track.value("size").type() == QVariant::String) {
        setSizeString(track.value("size").toString());
    }
    else {
        setSize(track.value("size").toLongLong());
    }
}

void PluginTrack::loadTrack(PluginTrack *track) {
    MKTrack::loadTrack(track);
    setCommentsId(track->commentsId());
    setRelatedTracksId(track->relatedTracksId());
}

ResourcesRequest* PluginTrack::request() {
    if (!m_request) {
        m_request = PluginManager::instance()->createRequestForService(service(), this);

        if (m_request) {
            connect(m_request, SIGNAL(finished()), this, SLOT(onRequestFinished()));
        }
    }

    return m_request;
}

void PluginTrack::onRequestFinished() {
    if (m_request->status() == ResourcesRequest::Ready) {
        loadTrack(service(), m_request->result().toMap());
    }
    
    emit changed();
    emit statusChanged(status());
}
