/*
 * Copyright (C) 2015 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "plugincomment.h"
#include "pluginmanager.h"
#include "resources.h"

PluginComment::PluginComment(QObject *parent) :
    MKComment(parent),
    m_request(0)
{
}

PluginComment::PluginComment(const QString &service, const QString &id, QObject *parent) :
    MKComment(parent),
    m_request(0)
{
    loadComment(service, id);
}

PluginComment::PluginComment(const QString &service, const QVariantMap &comment, QObject *parent) :
    MKComment(parent),
    m_request(0)
{
    loadComment(service, comment);
}

PluginComment::PluginComment(const PluginComment *comment, QObject *parent) :
    MKComment(comment, parent),
    m_request(0)
{
}

QString PluginComment::errorString() const {
    return m_request ? m_request->errorString() : QString();
}

ResourcesRequest::Status PluginComment::status() const {
    return m_request ? m_request->status() : ResourcesRequest::Null;
}

void PluginComment::loadComment(const QString &service, const QString &id) {
    setService(service);
    setId(id);
    
    if (status() == ResourcesRequest::Loading) {
        return;
    }

    if (ResourcesRequest *r = request()) {
        r->get(Resources::COMMENT, id);
        emit changed();
        emit statusChanged(status());
    }
}

void PluginComment::loadComment(const QString &service, const QVariantMap &comment) {
    setService(service);
    setArtist(comment.value("artist").toString());
    setArtistId(comment.value("artistId").toString());
    setBody(comment.value("body").toString());
    setDate(comment.value("date").toString());
    setId(comment.value("id").toString());
    setThumbnailUrl(comment.value("thumbnailUrl").toString());
    setTrackId(comment.value("trackId").toString());
}

void PluginComment::loadComment(PluginComment *comment) {
    MKComment::loadComment(comment);
}

ResourcesRequest* PluginComment::request() {
    if (!m_request) {
        m_request = PluginManager::instance()->createRequestForService(service(), this);

        if (m_request) {
            connect(m_request, SIGNAL(finished()), this, SLOT(onRequestFinished()));
        }
    }

    return m_request;
}

void PluginComment::onRequestFinished() {
    if (m_request->status() == ResourcesRequest::Ready) {
        loadComment(service(), m_request->result().toMap());
    }
    
    emit changed();
    emit statusChanged(status());
}
