#include "events.h"
#include "soundcloud.h"
#include "json.h"
#include "feedurls.h"
#include <QUrl>
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QDBusInterface>
#include <QStringList>
#include <QCoreApplication>
#include <QTimer>
#include <QFileInfo>
#include <QFile>
#include <QDir>
#include <QDateTime>
#include <QSettings>
#include <QSqlDatabase>
#include <QSqlQuery>
#include <QSqlRecord>
#if (defined (MEEGO_EDITION_HARMATTAN)) || (defined (Q_WS_SAILFISH))
#include <QNetworkConfigurationManager>
#include <QNetworkSession>
#ifdef MEEGO_EDITION_HARMATTAN
#include <meegotouchevents/meventfeed.h>
#else
#include <meegotouchevents-qt5/meventfeed.h>
#endif
#endif
#ifdef Q_WS_MAEMO_5
#include "alarm.h"
#endif
#if QT_VERSION >= 0x050000
const QString HOME_DIR(QStandardPaths::writableLocation(QStandardPaths::HomeLocation));
#else
#include <QDesktopServices>
const QString HOME_DIR(QDesktopServices::storageLocation(QDesktopServices::HomeLocation));
#endif

using namespace QtJson;

Events::Events(QObject *parent) :
    QObject(parent),
    m_nam(new QNetworkAccessManager(this)),
    m_soundcloud(new SoundCloud(this)),
    m_index(0),
    m_initialized(false)
{
}

void Events::init() {
    m_soundcloud->setNetworkAccessManager(m_nam);
#if (defined (MEEGO_EDITION_HARMATTAN)) || (defined (Q_WS_SAILFISH))
    QNetworkConfigurationManager configManager;
    QNetworkSession *session = new QNetworkSession(configManager.defaultConfiguration(), this);
    session->setSessionProperty("ConnectInBackground", true);
    this->connect(session, SIGNAL(error(QNetworkSession::SessionError)), this, SLOT(syncFailed()));
    this->connect(this, SIGNAL(destroyed()), session, SLOT(close()));
    session->open();
#endif
    QSqlDatabase database(QSqlDatabase::addDatabase("QSQLITE"));
    database.setDatabaseName(HOME_DIR + "/MusiKloud/MusiKloud.sqlite");
    database.open();

    QSqlQuery query;
    QSqlRecord record;
    query.exec("SELECT token FROM soundCloudAccount");
    record = query.record();

    if (record.count() > 0) {
        while (query.next()) {
            m_soundcloud->setAccessToken(query.value(0).toString());
        }
    }

    database.close();

    m_initialized = true;
}

bool Events::enableSync() {
    QSettings settings("MusiKloud", "MusiKloud");
#if (defined (MEEGO_EDITION_HARMATTAN)) || (defined (Q_WS_SAILFISH))
    settings.setValue("Events/eventFeedEnabled", true);
    bool success = true;
#else
    qint64 cookie = add_alarm();
    bool success = cookie > 0;

    if (success) {
        settings.setValue("Events/eventsAlarmCookie", cookie);
        settings.setValue("Events/eventFeedEnabled", true);
        QDBusInterface iface("com.maemo.eventFeed", "/", "com.maemo.eventFeed");
        iface.call(QDBus::NoBlock, "addRefreshAction", "musikloud", "dbus:session com.maemo.MusiKloudEvents / com.maemo.MusiKloudEvents startSync");
    }
#endif
    if (success) {
        this->startSync();
    }

    return success;
}

bool Events::disableSync() {
    QSettings settings("MusiKloud", "MusiKloud");
#if (defined (MEEGO_EDITION_HARMATTAN)) || (defined (Q_WS_SAILFISH))
    settings.remove("Events/latestEvent");
    settings.setValue("Events/eventFeedEnabled", false);
    MEventFeed::instance()->removeItemsBySourceName("musikloud");
    this->clearThumbnails();
    bool success = true;
#else
    qint64 cookie = settings.value("Events/eventsAlarmCookie", 0).toLongLong();
    bool success = cookie > 0;

    if (success) {
        success = delete_alarm(cookie);

        if (success) {
            settings.remove("Events/eventsAlarmCookie");
            settings.remove("Events/latestEvent");
            settings.setValue("Events/eventFeedEnabled", false);
            QDBusInterface iface("com.maemo.eventFeed", "/", "com.maemo.eventFeed");
            iface.call(QDBus::NoBlock, "removeEventsBySourceName", "musikloud");
            iface.call(QDBus::NoBlock, "removeRefreshAction", "musikloud");
        }
    }
#endif
    QTimer::singleShot(1000, QCoreApplication::instance(), SLOT(quit()));

    return success;
}

void Events::startSync() {
    if (!m_initialized) {
        this->init();
    }

    this->getEvents();
}

void Events::abortSync() {
    QCoreApplication::instance()->quit();
}

void Events::syncSuccess() {
    QCoreApplication::instance()->quit();
}

void Events::syncFailed() {
    QCoreApplication::instance()->quit();
}

void Events::getEvents() {
    if (m_soundcloud->userSignedIn()) {
        QNetworkReply *reply = m_soundcloud->createReply(ALL_ACTIVITIES_FEED);
        this->connect(reply, SIGNAL(finished()), this, SLOT(processEvents()));
    }
    else {
        this->syncFailed();
    }
}

void Events::processEvents() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->syncFailed();
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    switch (statusCode) {
    case 200:
    case 201: {
        QString response(reply->readAll());
        bool ok;
        QVariantMap res = Json::parse(response, ok).toMap();

        if (ok) {
            QVariantList entries = res.value("collection").toList();
            qint64 latest = QSettings("MusiKloud", "MusiKloud").value("Events/latestEvent", 0).toLongLong();

            for (int i = 0; i < entries.size(); i++) {
                QVariantMap entry = entries.at(i).toMap();
                qint64 date = QDateTime::fromString(entry.value("created_at").toString().section(' ', 0, 1), Qt::ISODate).toLocalTime().toMSecsSinceEpoch();

                if (date > latest) {
                    Event event(entry);
                    event.setDate(date);
                    m_events.append(event);
                }
            }

            if (!m_events.isEmpty()) {
                QSettings("MusiKloud", "MusiKloud").setValue("Events/latestEvent", m_events.first().date());
            }
#if (defined (MEEGO_EDITION_HARMATTAN)) || (defined (Q_WS_SAILFISH))
            this->downloadThumbnails();
#else
            this->addEvents();
#endif
        }
    }
        break;
    default:
        reply->deleteLater();
        this->syncFailed();
        return;
    }

    reply->deleteLater();
}

void Events::addEvents() {
#if (defined (MEEGO_EDITION_HARMATTAN)) || (defined (Q_WS_SAILFISH))
    while (!m_events.isEmpty()) {
        QVariantList args;
        QVariantMap itemArgs;
        Event event = m_events.takeFirst();
        itemArgs.insert("title", event.title());
        itemArgs.insert("icon", "/usr/share/icons/hicolor/64x64/apps/musikloud64.png");
        itemArgs.insert("body", event.body());
        itemArgs.insert("imageList", QStringList(event.thumbnailUrl().toLocalFile()));
        itemArgs.insert("timestamp", QDateTime::fromMSecsSinceEpoch(event.date()).toString("yyyy-MM-dd hh:mm:ss"));
        itemArgs.insert("video", false);
        itemArgs.insert("footer", event.artist());
        itemArgs.insert("sourceName", QString("musikloud"));
        itemArgs.insert("sourceDisplayName", QString("MusiKloud"));

#ifdef MEEGO_EDITION_HARMATTAN
        switch (event.activityType()) {
        case Activities::Playlist:
            itemArgs.insert("action", "com.maemo.MusiKloud / com.maemo.MusiKloud displayPlaylist " + this->base64SerializedVariant(event.id()));
            break;
        default:
            itemArgs.insert("action", "com.maemo.MusiKloud / com.maemo.MusiKloud displayTrack " + this->base64SerializedVariant(event.id()));
            break;
        }

        args.append(itemArgs);

        QDBusInterface("com.nokia.home.EventFeed", "/eventfeed", "com.nokia.home.EventFeed").callWithArgumentList(QDBus::Block, "addItem", args);
#else
        itemArgs.insert("url", event.url().toString());
        args.append(itemArgs);

        QDBusInterface("org.nemomobile.events.EventFeed", "/org/nemomobile/events/EventFeed", "org.nemomobile.events.EventFeed").callWithArgumentList(QDBus::Block, "addItem", args);
#endif
    }
#else
    while (!m_events.isEmpty()) {
        QVariantList args;
        Event event = m_events.takeFirst();
        args.append("musikloud");
        args.append("MusiKloud");
        args.append("/usr/share/icons/hicolor/64x64/apps/musikloud64.png");
        args.append(event.title());
        args.append(event.body());
        args.append(QStringList(event.thumbnailUrl().toString()));
        args.append(event.artist());
        args.append(QString::number(event.date()));

        switch (event.activityType()) {
        case Activities::Playlist:
            args.append("dbus:session com.maemo.MusiKloud / com.maemo.MusiKloud displayPlaylist " + event.id());
            break;
        default:
            args.append("dbus:session com.maemo.MusiKloud / com.maemo.MusiKloud displayTrack " + event.id());
            break;
        }

        QDBusInterface("com.maemo.eventFeed", "/", "com.maemo.eventFeed").callWithArgumentList(QDBus::Block, "addEvent", args);
    }
#endif
    this->syncSuccess();
}

#if (defined (MEEGO_EDITION_HARMATTAN)) || (defined (Q_WS_SAILFISH))
QString Events::base64SerializedVariant(const QVariant &value) const
{
    QByteArray ba;
    QDataStream stream(&ba, QIODevice::WriteOnly);
    stream << value;
    return ba.toBase64();
}

void Events::downloadThumbnails() {
    if ((m_index >= 0) && (m_index < m_events.size())) {
        QFileInfo info(QString("%1/MusiKloud/.feedthumbnails/%2.jpg").arg(HOME_DIR).arg(m_events[m_index].id()));

        if (info.exists()) {
            m_events[m_index].setThumbnailUrl(QUrl::fromLocalFile(info.absoluteFilePath()));
            m_index++;
            this->downloadThumbnails();
        }
        else {
            QNetworkRequest request(m_events[m_index].thumbnailUrl());
            QNetworkReply *reply = m_nam->get(request);
            this->connect(reply, SIGNAL(finished()), this, SLOT(onThumbnailDownloaded()));
        }
    }
    else {
        this->addEvents();
    }
}

void Events::onThumbnailDownloaded() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if ((!reply) || (m_index < 0) || (m_index >= m_events.size())) {
        this->syncFailed();
        return;
    }

    QDir dir(HOME_DIR + "/MusiKloud/.feedthumbnails/");
    dir.mkpath(dir.path());
    QFile file(QString("%1/%2.jpg").arg(dir.path()).arg(m_events[m_index].id()));

    if (file.open(QIODevice::WriteOnly)) {
        file.write(reply->readAll());
        file.close();
        m_events[m_index].setThumbnailUrl(QUrl::fromLocalFile(file.fileName()));
    }

    reply->deleteLater();
    m_index++;
    this->downloadThumbnails();
}

void Events::clearThumbnails() {
    QDir dir(HOME_DIR + "/MusiKloud/.feedthumbnails/");

    foreach (QString thumbnail, dir.entryList(QStringList("*.jpg"))) {
        dir.remove(thumbnail);
    }
}
#endif
