#ifndef MEDIAPLAYER_H
#define MEDIAPLAYER_H

#include "../../base/trackitem.h"
#include <QSharedPointer>
#include <qplatformdefs.h>
#ifdef MEEGO_EDITION_HARMATTAN
#include <resource/qt4/policy/resource-set.h>
#endif
#include <gst/gstelement.h>
#include <gst/gstbus.h>
#include <gst/gstmessage.h>

class QTimer;

class MediaPlayer : public QObject
{
    Q_OBJECT

#if QT_VERSION >= 0x040700
    Q_PROPERTY(bool playing
               READ playing
               NOTIFY stateChanged)
    Q_PROPERTY(bool paused
               READ paused
               NOTIFY stateChanged)
    Q_PROPERTY(qint64 position
               READ position
               WRITE setPosition
               NOTIFY positionChanged)
    Q_PROPERTY(qint64 duration
               READ duration
               NOTIFY durationChanged)
    Q_PROPERTY(qreal volume
               READ volume
               WRITE setVolume
               NOTIFY volumeChanged)
    Q_PROPERTY(bool muted
               READ muted
               WRITE setMuted
               NOTIFY mutedChanged)
    Q_PROPERTY(Media::PlaybackMode playbackMode
               READ playbackMode
               WRITE setPlaybackMode
               NOTIFY playbackModeChanged)
    Q_PROPERTY(Media::State state
               READ state
               NOTIFY stateChanged)
    Q_PROPERTY(int currentIndex
               READ currentIndex
               WRITE setCurrentIndex
               NOTIFY currentIndexChanged)
#ifdef QML_USER_INTERFACE
    Q_PROPERTY(TrackItem* currentTrack
               READ currentTrackFromQML
               NOTIFY currentTrackChanged)
#else
    Q_PROPERTY(QSharedPointer<TrackItem> currentTrack
               READ currentTrack
               NOTIFY currentTrackChanged)
#endif
#endif

public:
    explicit MediaPlayer(QObject *parent = 0);
    ~MediaPlayer();
#ifdef QML_USER_INTERFACE
    inline TrackItem* currentTrackFromQML() const { return m_track.data(); }
#endif
    inline QSharedPointer<TrackItem> currentTrack() const { return m_track; }
    inline int currentIndex() const { return m_index; }
    inline Media::PlaybackMode playbackMode() const { return m_playbackMode; }
    bool playing() const;
    bool paused() const;
    qint64 position() const;
    qint64 duration() const;
    qreal volume() const;
    bool muted() const;
    Media::State state() const;

    static MediaPlayer* instance();
        
public slots:
#ifdef QML_USER_INTERFACE
    void playTrack(TrackItem *track, bool playImmediately = true);
    void playTracks(QList<TrackItem*> tracks, bool playImmediately = true);
#else
    void playTrack(QSharedPointer<TrackItem> track, bool playImmediately = true);
    void playTracks(QList< QSharedPointer<TrackItem> > tracks, bool playImmediately = true);
#endif
    void removeTrack(int row);
    void clearTracks();
    void setPlaybackMode(Media::PlaybackMode mode);
    void setSequentialPlaybackMode();
    void setRepeatAllPlaybackMode();
    void setRepeatOnePlaybackMode();
    void togglePlaybackMode();
    void togglePlayPause();
    void play();
    void pause();
    void stop();
    void next();
    void previous();
    void setCurrentIndex(int index, bool playImmediately = true);
    void setPosition(qint64 position);
    void setVolume(qreal volume);
    void setMuted(bool muted);
    void toggleMuted();
    void startTick();
    void stopTick();

private:
    void setCurrentTrack(QSharedPointer<TrackItem> track);
    void setState(Media::State state);
    void onTrackPlayed(QSharedPointer<TrackItem> track);

private slots:
    void checkBusForMessages();
    void onBufferProgressChanged(GstMessage *message);
    void onPositionChanged(GstMessage *message);
    void onDurationChanged(GstMessage *message);
    void onStateChanged(GstMessage *message);
    void onPlaybackFinished();
    void onError(GstMessage *message);
#ifdef MEDIA_PLAYERS_ENABLED
    void onMediaPlayerChanged(const QString &mediaPlayer);
#endif
    void onCountChanged(int count);
#ifdef MEEGO_EDITION_HARMATTAN
    void onResourcesGranted();
    void onResourcesReleased();
    void onResourcesDenied();
    void onResourcesLost();
#endif

signals:
#ifdef QML_USER_INTERFACE
    void currentTrackChanged(TrackItem *track);
#else
    void currentTrackChanged(QSharedPointer<TrackItem> track);
#endif
    void currentIndexChanged(int index);
    void playbackModeChanged(Media::PlaybackMode mode);
    void positionChanged(qint64 position);
    void durationChanged(qint64 duration);
    void volumeChanged(qreal volume);
    void mutedChanged(bool muted);
    void stateChanged(Media::State state);

private:
    static MediaPlayer* self;

    GstElement *m_player;
    GstBus* m_bus;
    QTimer *m_busTimer;
    QSharedPointer<TrackItem> m_track;
    int m_index;
    qint64 m_position;
    qint64 m_duration;
    qreal m_volume;
    bool m_muted;
    Media::State m_state;
    Media::PlaybackMode m_playbackMode;
#ifdef MEEGO_EDITION_HARMATTAN
    ResourcePolicy::ResourceSet *m_resourceSet;
    ResourcePolicy::AudioResource *m_audioResource;
#endif
};

#endif // MEDIAPLAYER_H
