#include "trackinfowindow.h"
#include "separatorlabel.h"
#include "thumbnaillabel.h"
#include "listview.h"
#include "commentdelegate.h"
#include "nowplayingwindow.h"
#include "commentdialog.h"
#include "sharingdialog.h"
#include "channelwindow.h"
#include "nowplayingbutton.h"
#include "scrollarea.h"
#include "../base/commentlistmodel.h"
#include "../base/soundcloud.h"
#include "../base/tracklauncher.h"
#include "../base/transfermanager.h"
#include "../base/utils.h"
#include "../base/thumbnailcache.h"
#include "../base/settings.h"
#include "../base/playbacklistmodel.h"
#include "../mediaplayer/mobility/mediaplayer.h"
#include <QMenuBar>
#include <QGridLayout>
#include <QLabel>
#include <QAction>

TrackInfoWindow::TrackInfoWindow(QSharedPointer<TrackItem> track, QWidget *parent) :
    StackedWindow(parent),
    m_track(track),
    m_commentAction(this->menuBar()->addAction(tr("Add comment"), this, SLOT(showCommentDialog()))),
    m_favouritesAction(this->menuBar()->addAction(m_track.data()->favourite() ? tr("Delete from favourites") : tr("Add to favourites"), this, SLOT(toggleFavourite()))),
    m_queueAction(this->menuBar()->addAction(tr("Add to playback queue"), this, SLOT(queueTrack()))),
    m_downloadAction(this->menuBar()->addAction(tr("Download"), this, SLOT(downloadTrack()))),
    m_shareAction(this->menuBar()->addAction(tr("Share via connections"), this, SLOT(showShareDialog()))),
    m_titleLabel(new QLabel(QString("<p style='font-size:32px;'>%1</p>").arg(m_track.data()->title()), this)),
    m_artistLabel(new QLabel(m_track.data()->service() == Services::NoService ? QString("%1: %2").arg(tr("Artist")).arg(m_track.data()->artist()) : QString("%1: <a href='%2'>%2</a>").arg(tr("Artist")).arg(m_track.data()->artist()), this)),
    m_dateLabel(new QLabel(QString("%1: %2").arg(tr("Date")).arg(m_track.data()->date()), this)),
    m_genreLabel(new QLabel(QString("%1: %2").arg(tr("Genre")).arg(m_track.data()->genre().isEmpty() ? tr("Unknown") : m_track.data()->genre()), this)),
    m_formatLabel(new QLabel(QString("%1: %2").arg(tr("Format")).arg(m_track.data()->format()), this)),
    m_sizeLabel(new QLabel(QString("%1: %2").arg(tr("Size")).arg(Utils::fileSizeFromBytes(m_track.data()->size())), this)),
    m_lengthLabel(new QLabel(QString("%1: %2").arg(tr("Length")).arg(Utils::durationFromMSecs(m_track.data()->duration())), this)),
    m_playCountLabel(new QLabel(QString("%1: %2").arg(tr("Play count")).arg(m_track.data()->playCount()), this)),
    m_downloadCountLabel(new QLabel(QString("%1: %2").arg(tr("Download count")).arg(m_track.data()->downloadCount()), this)),
    m_descriptionLabel(new QLabel(m_track.data()->description().isEmpty() ? tr("No description") : m_track.data()->description(), this)),
    m_thumbnailLabel(new ThumbnailLabel(this)),
    m_waveformLabel(new ThumbnailLabel(this)),
    m_propertiesSeparator(new SeparatorLabel(tr("Properties"), this)),
    m_descriptionSeparator(new SeparatorLabel(tr("Description"), this)),
    m_commentsSeparator(new SeparatorLabel(tr("Comments"), this)),
    m_commentsModel(new CommentListModel(this)),
    m_commentsView(new ListView(this)),
    m_scrollArea(new ScrollArea(this))
{
    this->setWindowTitle(tr("Track info"));
    this->setCentralWidget(m_scrollArea);

    CommentDelegate *delegate = new CommentDelegate(m_commentsView);

    m_commentsView->setModel(m_commentsModel);
    m_commentsView->setItemDelegate(delegate);
    m_thumbnailLabel->setFixedSize(124, 124);
    m_waveformLabel->setFixedHeight(124);
    m_waveformLabel->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    m_titleLabel->setWordWrap(true);
    m_descriptionLabel->setWordWrap(true);

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);

    grid->setContentsMargins(0, 0, 0, 0);
    grid->addItem(new QSpacerItem(10, 10, QSizePolicy::Maximum), 0, 0);
    grid->addWidget(m_titleLabel, 0, 1, 1, 2, Qt::AlignTop);
    grid->addWidget(m_thumbnailLabel, 1, 1, 2, 1, Qt::AlignTop);
    grid->addWidget(m_artistLabel, 1, 2, Qt::AlignTop);
    grid->addWidget(m_dateLabel, 2, 2, Qt::AlignTop);
    grid->addWidget(m_propertiesSeparator, 3, 1, 1, 2);
    grid->addWidget(m_genreLabel, 4, 1, 1, 2);
    grid->addWidget(m_formatLabel, 5, 1, 1, 2);
    grid->addWidget(m_sizeLabel, 6, 1, 1, 2);
    grid->addWidget(m_lengthLabel, 7, 1, 1, 2);
    grid->addWidget(m_playCountLabel, 8, 1, 1, 2);
    grid->addWidget(m_downloadCountLabel, 9, 1, 1, 2);
    grid->addWidget(m_waveformLabel, 10, 1, 1, 2);
    grid->addWidget(m_descriptionSeparator, 11, 1, 1, 2);
    grid->addWidget(m_descriptionLabel, 12, 1, 1, 2);
    grid->addWidget(m_commentsSeparator, 13, 1, 1, 2);
    grid->addWidget(m_commentsView, 14, 1, 1, 2);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_scrollArea->setMonitorScrolling(true);

    this->connect(m_commentsModel, SIGNAL(countChanged(int)), this, SLOT(onCommentsCountChanged(int)));
    this->connect(delegate, SIGNAL(showUserChannel(QModelIndex)), this, SLOT(showUserChannel(QModelIndex)));
    this->connect(m_thumbnailLabel, SIGNAL(clicked()), this, SLOT(playTrack()));
    this->connect(m_waveformLabel, SIGNAL(clicked()), this, SLOT(playTrack()));
    this->connect(m_artistLabel, SIGNAL(linkActivated(QString)), this, SLOT(showUserChannel()));
    this->connect(m_track.data(), SIGNAL(favouriteChanged(bool)), this, SLOT(onFavouriteChanged(bool)));

    if (Settings::instance()->mediaPlayer() == "musikloud") {
        this->connect(m_scrollArea, SIGNAL(scrollingStarted()), NowPlayingButton::instance(), SLOT(hide()));
        this->connect(m_scrollArea, SIGNAL(scrollingStopped()), NowPlayingButton::instance(), SLOT(poke()));
    }

    switch (m_track.data()->service()) {
    case Services::NoService:
        if (Settings::instance()->archiveOnline()) {
            m_commentsModel->getComments(m_track.data()->id());
        }
        else {
            m_commentsSeparator->setText(tr("No comments"));
        }
        break;
    default:
        m_commentsModel->getComments(m_track.data()->id());
        break;
    }

    this->setThumbnails();
    this->setMenuActions();
    this->onOrientationChanged();
}

void TrackInfoWindow::setLandscapeLayout() {
    m_scrollArea->widget()->setMaximumWidth(800);
}

void TrackInfoWindow::setPortraitLayout() {
    m_scrollArea->widget()->setMaximumWidth(480);
}

void TrackInfoWindow::setThumbnails() {
    QImage thumbnail = ThumbnailCache::instance()->thumbnail(m_track.data()->thumbnailUrl(), QSize(124, 124));
    QImage waveform = ThumbnailCache::instance()->thumbnail(m_track.data()->waveformUrl());

    if ((thumbnail.isNull()) || (waveform.isNull())) {
        this->connect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }
    else {
        this->disconnect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }

    if (thumbnail.isNull()) {
        thumbnail = QImage("/usr/share/icons/hicolor/124x124/hildon/mediaplayer_default_album.png");
    }

    m_thumbnailLabel->setImage(thumbnail);

    if (!waveform.isNull()) {
        m_waveformLabel->setImage(waveform);
    }
}

void TrackInfoWindow::setMenuActions() {
    switch (m_track.data()->service()) {
    case Services::NoService:
        m_commentAction->setEnabled((SoundCloud::instance()->userSignedIn()) && (Settings::instance()->archiveOnline()));
        m_favouritesAction->setEnabled((SoundCloud::instance()->userSignedIn()) && (Settings::instance()->archiveOnline()));
        m_shareAction->setEnabled((SoundCloud::instance()->userSignedIn()) && (Settings::instance()->archiveOnline()));
        m_downloadAction->setEnabled(false);
        return;
    default:
        m_commentAction->setEnabled(SoundCloud::instance()->userSignedIn());
        m_favouritesAction->setEnabled(SoundCloud::instance()->userSignedIn());
        m_shareAction->setEnabled(SoundCloud::instance()->userSignedIn());
        return;
    }
}

void TrackInfoWindow::playTrack() {
    if (Settings::instance()->mediaPlayer() == "musikloud") {
        MediaPlayer::instance()->playTrack(m_track);
        NowPlayingWindow *window = new NowPlayingWindow(this);
        window->show();
    }
    else {
        switch (m_track.data()->service()) {
        case Services::NoService:
            TrackLauncher::launchTrack(m_track.data()->url());
            return;
        default:
            TrackLauncher::launchTrack(SoundCloud::instance()->getStreamUrl(m_track.data()->streamUrl()));
            return;
        }
    }
}

void TrackInfoWindow::queueTrack() {
    PlaybackListModel::playbackQueue()->addTrack(m_track);
}

void TrackInfoWindow::downloadTrack() {
    TransferManager::instance()->addDownloadTransfer(m_track);
}

void TrackInfoWindow::toggleFavourite() {
    if (m_track.data()->favourite()) {
        SoundCloud::instance()->deleteFromFavourites(QStringList(m_track.data()->id()));
    }
    else {
        SoundCloud::instance()->addToFavourites(QStringList(m_track.data()->id()));
    }
}

void TrackInfoWindow::showCommentDialog() {
    CommentDialog *dialog = new CommentDialog(m_track, this);
    dialog->open();
}

void TrackInfoWindow::showShareDialog() {
    SharingDialog *dialog = new SharingDialog(m_track, this);
    dialog->open();
}

void TrackInfoWindow::showUserChannel() {
    ChannelWindow *window = new ChannelWindow(m_track.data()->artistId(), this);
    window->show();
}

void TrackInfoWindow::showUserChannel(const QModelIndex &index) {
    ChannelWindow *window = new ChannelWindow(index.data(CommentListModel::ArtistIdRole).toString(), this);
    window->show();
}

void TrackInfoWindow::onFavouriteChanged(bool favourite) {
    m_favouritesAction->setText(favourite ? tr("Delete from favourites") : tr("Add to favourites"));
}

void TrackInfoWindow::onCommentsCountChanged(int count) {
    m_commentsSeparator->setText(!count ? tr("No comments") : tr("Comments"));
    m_commentsView->setFixedHeight(m_commentsModel->rowCount() * m_commentsView->sizeHintForRow(0));
}
