#include "searchdialog.h"
#include "searchhistorydialog.h"
#include "searchtypeselector.h"
#include "searchorderselector.h"
#include "searchhistorydialog.h"
#include "../base/searchlistmodel.h"
#include "../base/settings.h"
#include "../base/clipboardmonitor.h"
#include <QLineEdit>
#include <QPushButton>
#include <QScrollArea>
#include <QDialogButtonBox>
#include <QGridLayout>
#include <QClipboard>
#include <QRegExp>
#include <QApplication>

SearchDialog::SearchDialog(QWidget *parent) :
    AbstractDialog(parent),
    m_searchEdit(new QLineEdit(this)),
    m_searchTypeSelector(new SearchTypeSelector(tr("Search for"), this)),
    m_searchOrderSelector(new SearchOrderSelector(tr("Order by"), this)),
    m_buttonBox(new QDialogButtonBox(Qt::Vertical, this)),
    m_historyButton(m_buttonBox->addButton(tr("History"), QDialogButtonBox::ActionRole)),
    m_searchButton(m_buttonBox->addButton(ClipboardMonitor::instance()->clipboardUrl().isEmpty() ? tr("Search") : tr("From URL"), QDialogButtonBox::ActionRole)),
    m_scrollArea(new QScrollArea(this)),
    m_model(new SearchListModel(this)),
    m_historyDialog(0),
    m_grid(new QGridLayout(this))
{
    this->setWindowTitle(tr("Search"));
    m_searchEdit->setPlaceholderText(tr("Search query"));
    m_searchTypeSelector->setValue(Settings::instance()->searchType());
    m_searchOrderSelector->setValue(Settings::instance()->searchOrder());

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QVBoxLayout *vbox = new QVBoxLayout(scrollWidget);

    vbox->addWidget(m_searchEdit);
    vbox->addWidget(m_searchTypeSelector);
    vbox->addWidget(m_searchOrderSelector);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    this->connect(m_searchEdit, SIGNAL(returnPressed()), this, SLOT(parseSearchQuery()));
    this->connect(m_searchEdit, SIGNAL(textChanged(QString)), this, SLOT(onSearchTextChanged(QString)));
    this->connect(m_historyButton, SIGNAL(clicked()), this, SLOT(showSearchHistory()));
    this->connect(m_searchButton, SIGNAL(clicked()), this, SLOT(parseSearchQuery()));
    this->connect(m_searchTypeSelector, SIGNAL(valueChanged(Queries::QueryType)), Settings::instance(), SLOT(setSearchType(Queries::QueryType)));
    this->connect(m_searchOrderSelector, SIGNAL(valueChanged(Queries::QueryOrder)), Settings::instance(), SLOT(setSearchOrder(Queries::QueryOrder)));

    this->onOrientationChanged();
}

void SearchDialog::setLandscapeLayout() {
    m_buttonBox->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }

    this->setFixedHeight(m_scrollArea->widget()->height());
}

void SearchDialog::setPortraitLayout() {
    m_buttonBox->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }

    this->setFixedHeight(qMin(600, m_scrollArea->widget()->height() + 160));
}

void SearchDialog::showSearchHistory() {
    if (!m_historyDialog) {
        m_historyDialog = new SearchHistoryDialog(m_model, this);
        this->connect(m_historyDialog, SIGNAL(queryChosen(QString)), m_searchEdit, SLOT(setText(QString)));
    }

    m_historyDialog->open();
}

void SearchDialog::onSearchTextChanged(const QString &text) {
    if ((text.isEmpty()) && (!ClipboardMonitor::instance()->clipboardUrl().isEmpty())) {
        m_searchButton->setText(tr("From URL"));
    }
    else {
        m_searchButton->setText(tr("Search"));
    }
}

void SearchDialog::parseSearchQuery() {
    if ((!ClipboardMonitor::instance()->clipboardUrl().isEmpty()) && (m_searchEdit->text().isEmpty())) {
        emit fromUrl(ClipboardMonitor::instance()->clipboardUrl());
    }
    else {
        m_model->addSearch(m_searchEdit->text());
        emit search(m_searchEdit->text(), Settings::instance()->searchType(), Settings::instance()->searchOrder());
    }

    this->accept();
}
