#include "connectionsdialog.h"
#include "listview.h"
#include "../base/connectionlistmodel.h"
#include <QVBoxLayout>
#include <QPainter>
#include <QLabel>
#include <QMaemo5Style>

ConnectionsDialog::ConnectionsDialog(QWidget *parent) :
    AbstractDialog(parent),
    m_model(new ConnectionListModel(this)),
    m_view(new ListView(this)),
    m_noResultsLabel(new QLabel(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No connections found")), this))
{
    this->setAttribute(Qt::WA_DeleteOnClose, false);
    this->setWindowTitle(tr("Connections"));
    this->setFixedHeight(340);

    m_noResultsLabel->hide();
    m_view->setModel(m_model);
    m_view->setItemDelegate(new ConnectionListDelegate(m_view));
    m_model->getConnections();

    QVBoxLayout *vbox = new QVBoxLayout(this);
    vbox->addWidget(m_view);

    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SLOT(toggleBusy(bool)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(m_view, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));

    this->onOrientationChanged();
}

void ConnectionsDialog::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 300);
}

void ConnectionsDialog::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 300);
}

QStringList ConnectionsDialog::connections() const {
    return m_model->connections();
}

QStringList ConnectionsDialog::connectionNames() const {
    return m_model->connectionNames();
}

void ConnectionsDialog::onCountChanged(int count) {
    m_noResultsLabel->setVisible(count == 0);
}

void ConnectionsDialog::onItemClicked(const QModelIndex &index) {
    m_model->toggleSelected(index.row());
    emit connectionNamesChanged(this->connectionNames());
}

ConnectionListDelegate::ConnectionListDelegate(QObject *parent) :
    QStyledItemDelegate(parent)
{
}

void ConnectionListDelegate::paint(QPainter *painter, const QStyleOptionViewItem &option, const QModelIndex &index) const {
    painter->save();

    if (((option.state) & (QStyle::State_Selected)) || index.data(ConnectionListModel::SelectedRole).toBool()) {
        painter->drawImage(option.rect, QImage("/etc/hildon/theme/images/TouchListBackgroundPressed.png"));
    }
    else {
        painter->drawImage(option.rect, QImage("/etc/hildon/theme/images/TouchListBackgroundNormal.png"));
    }

    QRect iconRect = option.rect;
    iconRect.setSize(QSize(32, 32));
    iconRect.moveTopLeft(QPoint(iconRect.left() + 8, iconRect.top() + 19));

    painter->drawImage(iconRect, QImage(":/" + index.data(ConnectionListModel::IconSourceRole).toString()));

    if (index.data(ConnectionListModel::SelectedRole).toBool()) {
        iconRect.setSize(QSize(48, 48));
        iconRect.moveTopRight(QPoint(option.rect.right() - 8, iconRect.top() - 8));

        painter->drawImage(iconRect, QImage("/usr/share/icons/hicolor/48x48/hildon/widgets_tickmark_grid.png"));
    }

    QRect textRect = option.rect;
    textRect.setLeft(textRect.left() + 48);
    textRect.setRight(textRect.right() - 48);

    painter->setClipRect(textRect);
    painter->drawText(textRect, Qt::AlignCenter, index.data(ConnectionListModel::DisplayNameRole).toString());
    painter->restore();
}

QSize ConnectionListDelegate::sizeHint(const QStyleOptionViewItem &option, const QModelIndex &index) const {
    Q_UNUSED(index);

    return QSize(option.rect.width(), 70);
}
