#include "trackinfowindow.h"
#include "commentlistdelegate.h"
#include "mainwindow.h"
#include "channelwindow.h"
#include "commentdialog.h"
#include "sharedialog.h"
#include "waveformlabel.h"
#include "separatorlabel.h"
#include "../base/commentlistmodel.h"
#include "../base/playbacklistmodel.h"
#include "../base/soundcloud.h"
#include "../base/transfermanager.h"
#include "../base/settings.h"
#include "../base/utils.h"
#include "../base/thumbnailcache.h"
#include "../mediaplayer/gstreamer/mediaplayer.h"
#include <QMenu>
#include <QLabel>
#include <QListView>
#include <QScrollArea>
#include <QGridLayout>
#include <QContextMenuEvent>

TrackInfoWindow::TrackInfoWindow(QSharedPointer<TrackItem> track, QWidget *parent) :
    QWidget(parent),
    m_track(track),
    m_contextMenu(new QMenu(this)),
    m_playAction(m_contextMenu->addAction(tr("Play"), this, SLOT(playTrack()))),
    m_queueAction(m_contextMenu->addAction(tr("Queue"), this, SLOT(queueTrack()))),
    m_downloadAction(m_contextMenu->addAction(tr("Download"), this, SLOT(downloadTrack()))),
    m_shareAction(m_contextMenu->addAction(tr("Share"), this, SLOT(showShareDialog()))),
    m_favouritesAction(m_contextMenu->addAction(m_track.data()->favourite() ? tr("Delete from favourites") : tr("Add to favourites"), this, SLOT(toggleFavourite()))),
    m_commentAction(m_contextMenu->addAction(tr("Add comment"), this, SLOT(showCommentDialog()))),
    m_titleLabel(new QLabel(QString("<p style='font-size:32px;'>%1</p>").arg(m_track.data()->title()), this)),
    m_artistLabel(new QLabel(m_track.data()->service() == Services::NoService ? QString("%1: %2").arg(tr("Artist")).arg(m_track.data()->artist()) : QString("%1: <a href='%2'>%2</a>").arg(tr("Artist")).arg(m_track.data()->artist()), this)),
    m_dateLabel(new QLabel(QString("%1: %2").arg(tr("Date")).arg(m_track.data()->date()), this)),
    m_genreLabel(new QLabel(QString("%1: %2").arg(tr("Genre")).arg(m_track.data()->genre().isEmpty() ? tr("Unknown") : m_track.data()->genre()), this)),
    m_formatLabel(new QLabel(QString("%1: %2").arg(tr("Format")).arg(m_track.data()->format()), this)),
    m_sizeLabel(new QLabel(QString("%1: %2").arg(tr("Size")).arg(Utils::fileSizeFromBytes(m_track.data()->size())), this)),
    m_lengthLabel(new QLabel(QString("%1: %2").arg(tr("Length")).arg(Utils::durationFromMSecs(m_track.data()->duration())), this)),
    m_playCountLabel(new QLabel(QString("%1: %2").arg(tr("Play count")).arg(m_track.data()->playCount()), this)),
    m_downloadCountLabel(new QLabel(QString("%1: %2").arg(tr("Download count")).arg(m_track.data()->downloadCount()), this)),
    m_descriptionLabel(new QLabel(m_track.data()->description().isEmpty() ? tr("No description") : m_track.data()->description(), this)),
    m_thumbnailLabel(new ThumbnailLabel(this)),
    m_waveformLabel(new WaveformLabel(this)),
    m_propertiesSeparator(new SeparatorLabel(tr("Properties"), this)),
    m_descriptionSeparator(new SeparatorLabel(tr("Description"), this)),
    m_commentsSeparator(new SeparatorLabel(tr("Comments"), this)),
    m_commentsModel(new CommentListModel(this)),
    m_commentsView(new QListView(this)),
    m_scrollArea(new QScrollArea(this)),
    m_vbox(new QVBoxLayout(this))
{
    this->setWindowTitle(m_track.data()->title());
    this->setAttribute(Qt::WA_DeleteOnClose);
    this->setFocusProxy(m_scrollArea);

    m_commentsView->setModel(m_commentsModel);
    m_commentsView->setItemDelegate(new CommentListDelegate(m_commentsView));
    m_commentsView->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_commentsView->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    m_thumbnailLabel->setFixedSize(100, 100);
    m_waveformLabel->setFixedHeight(100);
    m_waveformLabel->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    m_titleLabel->setWordWrap(true);
    m_descriptionLabel->setWordWrap(true);

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);

    grid->addWidget(m_titleLabel, 0, 0, 1, 2, Qt::AlignTop);
    grid->addWidget(m_thumbnailLabel, 1, 0, 2, 1, Qt::AlignTop);
    grid->addWidget(m_artistLabel, 1, 1, Qt::AlignTop);
    grid->addWidget(m_dateLabel, 2, 1, Qt::AlignTop);
    grid->addWidget(m_propertiesSeparator, 3, 0, 1, 2);
    grid->addWidget(m_genreLabel, 4, 0, 1, 2);
    grid->addWidget(m_formatLabel, 5, 0, 1, 2);
    grid->addWidget(m_sizeLabel, 6, 0, 1, 2);
    grid->addWidget(m_lengthLabel, 7, 0, 1, 2);
    grid->addWidget(m_playCountLabel, 8, 0, 1, 2);
    grid->addWidget(m_downloadCountLabel, 9, 0, 1, 2);
    grid->addWidget(m_waveformLabel, 10, 0, 1, 2);
    grid->addWidget(m_descriptionSeparator, 11, 0, 1, 2);
    grid->addWidget(m_descriptionLabel, 12, 0, 1, 2);
    grid->addWidget(m_commentsSeparator, 13, 0, 1, 2);
    grid->addWidget(m_commentsView, 14, 0, 1, 2);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    this->connect(m_commentsModel, SIGNAL(countChanged(int)), this, SLOT(onCommentsCountChanged(int)));
    this->connect(m_thumbnailLabel, SIGNAL(clicked()), this, SLOT(playTrack()));
    this->connect(m_waveformLabel, SIGNAL(clicked()), this, SLOT(playTrack()));
    this->connect(m_artistLabel, SIGNAL(linkActivated(QString)), this, SLOT(showUserChannel()));
    this->connect(m_track.data(), SIGNAL(favouriteChanged(bool)), this, SLOT(onFavouriteChanged(bool)));

    switch (m_track.data()->service()) {
    case Services::NoService:
        if (Settings::instance()->archiveOnline()) {
            m_commentsModel->getComments(m_track.data()->id());
        }
        else {
            m_commentsSeparator->setText(tr("No comments"));
        }
        break;
    default:
        m_commentsModel->getComments(m_track.data()->id());
        break;
    }

    m_vbox->setContentsMargins(0, 0, 0, 0);
    m_vbox->addWidget(m_scrollArea);

    this->setThumbnails();
    this->setMenuActions();
}

TrackInfoWindow::~TrackInfoWindow() {}

void TrackInfoWindow::contextMenuEvent(QContextMenuEvent *event) {
    m_contextMenu->setActiveAction(m_playAction);
    m_contextMenu->popup(this->mapToGlobal(event->pos()), m_playAction);
}

void TrackInfoWindow::setThumbnails() {
    QImage thumbnail = ThumbnailCache::instance()->thumbnail(m_track.data()->thumbnailUrl(), m_thumbnailLabel->size());
    QImage waveform = ThumbnailCache::instance()->thumbnail(m_track.data()->waveformUrl(), m_waveformLabel->size());

    if ((thumbnail.isNull()) || (waveform.isNull())) {
        this->connect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }
    else {
        this->disconnect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }

    if (thumbnail.isNull()) {
        thumbnail = QImage("/usr/share/icons/hicolor/scalable/hildon/qgn_list_gene_music_file.png");
    }

    m_thumbnailLabel->setImage(thumbnail);

    if (!waveform.isNull()) {
        m_waveformLabel->setImage(waveform);
    }
}

void TrackInfoWindow::setMenuActions() {
    switch (m_track.data()->service()) {
    case Services::NoService:
        m_commentAction->setEnabled((SoundCloud::instance()->userSignedIn()) && (Settings::instance()->archiveOnline()));
        m_favouritesAction->setEnabled((SoundCloud::instance()->userSignedIn()) && (Settings::instance()->archiveOnline()));
        m_shareAction->setEnabled((SoundCloud::instance()->userSignedIn()) && (Settings::instance()->archiveOnline()));
        m_downloadAction->setEnabled(false);
        return;
    default:
        m_commentAction->setEnabled(SoundCloud::instance()->userSignedIn());
        m_favouritesAction->setEnabled(SoundCloud::instance()->userSignedIn());
        m_shareAction->setEnabled(SoundCloud::instance()->userSignedIn());
        return;
    }
}

void TrackInfoWindow::playTrack() {
    MediaPlayer::instance()->playTrack(m_track);
}

void TrackInfoWindow::queueTrack() {
    PlaybackListModel::playbackQueue()->addTrack(m_track);
}

void TrackInfoWindow::downloadTrack() {
    TransferManager::instance()->addDownloadTransfer(m_track);
}

void TrackInfoWindow::toggleFavourite() {
    if (m_track.data()->favourite()) {
        SoundCloud::instance()->deleteFromFavourites(QStringList(m_track.data()->id()));
    }
    else {
        SoundCloud::instance()->addToFavourites(QStringList(m_track.data()->id()));
    }
}

void TrackInfoWindow::showCommentDialog() {
    CommentDialog *dialog = new CommentDialog(m_track, this);
    dialog->open();
}

void TrackInfoWindow::showShareDialog() {
    ShareDialog *dialog = new ShareDialog(m_track, this);
    dialog->open();
}

void TrackInfoWindow::showUserChannel() {
    ChannelWindow *window = new ChannelWindow(m_track.data()->artistId(), this);
    MainWindow::instance()->push(window);
}

void TrackInfoWindow::onFavouriteChanged(bool favourite) {
    m_favouritesAction->setText(favourite ? tr("Delete from favourites") : tr("Add to favourites"));
}

void TrackInfoWindow::onCommentsCountChanged(int count) {
    m_commentsSeparator->setText(!count ? tr("No comments") : tr("Comments"));
    m_commentsView->setFixedHeight(m_commentsModel->rowCount() * m_commentsView->sizeHintForRow(0));
}
