#include "playlisttrackswindow.h"
#include "tracklistdelegate.h"
#include "trackinfowindow.h"
#include "channelwindow.h"
#include "nowplayingwindow.h"
#include "mainwindow.h"
#include "thumbnaillabel.h"
#include "separatorlabel.h"
#include "../base/tracklistmodel.h"
#include "../base/thumbnailcache.h"
#include "../base/playlistitem.h"
#include "../base/playbacklistmodel.h"
#include "../base/transfermanager.h"
#include "../base/settings.h"
#include "../base/soundcloud.h"
#include "../mediaplayer/gstreamer/mediaplayer.h"
#include <QAction>
#include <QLabel>
#include <QPushButton>
#include <QMenu>
#include <QGridLayout>
#include <QScrollArea>
#include <QListView>
#include <QContextMenuEvent>

PlaylistTracksWindow::PlaylistTracksWindow(QSharedPointer<PlaylistItem> playlist, QWidget *parent) :
    QWidget(parent),
    m_playlist(playlist),
    m_contextMenu(new QMenu(this)),
    m_playAction(m_contextMenu->addAction(tr("Play all"), this, SLOT(playPlaylist()))),
    m_queueAction(m_contextMenu->addAction(tr("Queue all"), this, SLOT(queuePlaylist()))),
    m_downloadAction(m_contextMenu->addAction(tr("Download all"), this, SLOT(downloadPlaylist()))),
    m_tracksMenu(new QMenu(this)),
    m_viewTrackAction(m_tracksMenu->addAction(tr("View"), this, SLOT(showSelectedTrack()))),
    m_playTrackAction(m_tracksMenu->addAction(tr("Play"), this, SLOT(playSelectedTrack()))),
    m_queueTrackAction(m_tracksMenu->addAction(tr("Queue"), this, SLOT(queueSelectedTrack()))),
    m_downloadTrackAction(m_tracksMenu->addAction(tr("Download"), this, SLOT(downloadSelectedTrack()))),
    m_favouritesAction(m_tracksMenu->addAction(tr("Add to favourites"), this, SLOT(addSelectedTrackToFavourites()))),
    m_titleLabel(new QLabel(QString("<p style='font-size:32px;'>%1</p>").arg(m_playlist.data()->title()), this)),
    m_descriptionLabel(new QLabel(m_playlist.data()->description().isEmpty() ? tr("No description") : m_playlist.data()->description(), this)),
    m_artistLabel(new QLabel(QString("%1: <a href='%2'>%2</a>").arg(tr("Artist")).arg(m_playlist.data()->artist()), this)),
    m_dateLabel(new QLabel(QString("%1: %2").arg(tr("Date")).arg(m_playlist.data()->date()), this)),
    m_thumbnail(new ThumbnailLabel(this)),
    m_playButton(new QPushButton(tr("Play all"), this)),
    m_separator1(new SeparatorLabel(tr("Description"), this)),
    m_separator2(new SeparatorLabel(QString("-- %1").arg(tr("tracks")), this)),
    m_scrollArea(new QScrollArea(this)),
    m_view(new QListView(this)),
    m_model(new TrackListModel(this)),
    m_vbox(new QVBoxLayout(this))
{
    this->setWindowTitle(m_playlist.data()->title());
    this->setAttribute(Qt::WA_DeleteOnClose);
    this->setFocusProxy(m_scrollArea);

    m_view->setModel(m_model);
    m_view->setItemDelegate(new TrackListDelegate(m_view));
    m_view->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_view->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_view->setContextMenuPolicy(Qt::CustomContextMenu);

    m_titleLabel->setWordWrap(true);
    m_descriptionLabel->setWordWrap(true);

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);

    m_thumbnail->setInteractive(false);
    m_thumbnail->setFixedSize(100, 100);
    m_playButton->setFixedWidth(100);

    grid->addWidget(m_titleLabel, 0, 0, 1, 3, Qt::AlignTop);
    grid->addWidget(m_thumbnail, 1, 0, 2, 1, Qt::AlignTop);
    grid->addWidget(m_artistLabel, 1, 1, Qt::AlignTop);
    grid->addWidget(m_dateLabel, 2, 1, Qt::AlignTop);
    grid->addWidget(m_playButton, 3, 2, Qt::AlignRight);
    grid->addWidget(m_separator1, 4, 0, 1, 4);
    grid->addWidget(m_descriptionLabel, 5, 0, 1, 3);
    grid->addWidget(m_separator2, 6, 0, 1, 4);
    grid->addWidget(m_view, 7, 0, 1, 4);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(m_artistLabel, SIGNAL(linkActivated(QString)), this, SLOT(showUserChannel()));
    this->connect(m_view, SIGNAL(activated(QModelIndex)), this, SLOT(onListItemActivated(QModelIndex)));
    this->connect(m_view, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showTracksContextMenu(QPoint)));
    this->connect(m_playButton, SIGNAL(clicked()), this, SLOT(playPlaylist()));

    m_vbox->addWidget(m_scrollArea);

    m_model->getPlaylistTracks(m_playlist.data()->id());

    this->setThumbnail();
    this->setMenuActions();
}

PlaylistTracksWindow::~PlaylistTracksWindow() {}

void PlaylistTracksWindow::contextMenuEvent(QContextMenuEvent *event) {
    m_contextMenu->setActiveAction(m_playAction);
    m_contextMenu->popup(this->mapToGlobal(event->pos()), m_playAction);
    event->accept();
}

void PlaylistTracksWindow::setThumbnail() {
    QImage thumbnail = ThumbnailCache::instance()->thumbnail(m_playlist.data()->thumbnailUrl(), m_thumbnail->size());

    if (thumbnail.isNull()) {
        m_thumbnail->setImage(QImage("/usr/share/icons/hicolor/scalable/hildon/qgn_list_gene_playlist.png"));
        this->connect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnail()));
    }
    else {
        m_thumbnail->setImage(thumbnail);
        this->disconnect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnail()));
    }
}

void PlaylistTracksWindow::setMenuActions() {
    m_favouritesAction->setEnabled(SoundCloud::instance()->userSignedIn());
}

void PlaylistTracksWindow::onLoadingChanged(bool loading) {
    m_playButton->setEnabled((!loading) && (m_model->rowCount() > 0));

    if (!m_model->loading()) {
        m_separator2->setText(QString("%1 %2").arg(m_model->rowCount()).arg(tr("tracks")));

        if (m_model->rowCount() > 0) {
            m_view->setFixedHeight(m_model->rowCount() * m_view->sizeHintForRow(0));
        }
    }
}

void PlaylistTracksWindow::onCountChanged(int count) {
    if (!m_model->loading()) {
        m_separator2->setText(QString("%1 %2").arg(count).arg(tr("tracks")));
    }
}

void PlaylistTracksWindow::showUserChannel() {
    ChannelWindow *window = new ChannelWindow(m_playlist.data()->artistId(), this);
    window->show();
}

void PlaylistTracksWindow::playPlaylist() {
    MediaPlayer::instance()->playTracks(m_model->allItems());
}

void PlaylistTracksWindow::queuePlaylist() {
    PlaybackListModel::playbackQueue()->addTracks(m_model->allItems());
}

void PlaylistTracksWindow::downloadPlaylist() {
    TransferManager::instance()->addDownloadTransfer(m_playlist, m_model->allItems());
}

void PlaylistTracksWindow::onListItemActivated(const QModelIndex &index) {
    if (index.isValid()) {
        TrackInfoWindow *window = new TrackInfoWindow(m_model->get(index.row()), MainWindow::instance());
        MainWindow::instance()->push(window);
    }
}

void PlaylistTracksWindow::showTracksContextMenu(const QPoint &pos) {
    if (m_view->currentIndex().isValid()) {
        m_tracksMenu->setActiveAction(m_viewTrackAction);
        m_tracksMenu->popup(this->mapToGlobal(pos), m_viewTrackAction);
    }
}

void PlaylistTracksWindow::showSelectedTrack() {
    if (m_view->currentIndex().isValid()) {
        TrackInfoWindow *window = new TrackInfoWindow(m_model->get(m_view->currentIndex().row()), MainWindow::instance());
        MainWindow::instance()->push(window);
    }
}

void PlaylistTracksWindow::playSelectedTrack() {
    if (m_view->currentIndex().isValid()) {
        MediaPlayer::instance()->playTrack(m_model->get(m_view->currentIndex().row()));
    }
}

void PlaylistTracksWindow::queueSelectedTrack() {
    if (m_view->currentIndex().isValid()) {
        PlaybackListModel::playbackQueue()->addTrack(m_model->get(m_view->currentIndex().row()));
    }
}

void PlaylistTracksWindow::downloadSelectedTrack() {
    if (m_view->currentIndex().isValid()) {
        TransferManager::instance()->addDownloadTransfer(m_model->get(m_view->currentIndex().row()));
    }
}

void PlaylistTracksWindow::addSelectedTrackToFavourites() {
    if (m_view->currentIndex().isValid()) {
        SoundCloud::instance()->addToFavourites(QStringList(m_model->data(m_view->currentIndex(), TrackListModel::IdRole).toString()));
    }
}
