#include "nowplayinglisttab.h"
#include "../mediaplayer/gstreamer/mediaplayer.h"
#include "../base/playbacklistmodel.h"
#include "tracklistdelegate.h"
#include <QListView>
#include <QMenu>
#include <QVBoxLayout>
#include <QShowEvent>
#include <QHideEvent>

NowPlayingListTab::NowPlayingListTab(QWidget *parent) :
    QWidget(parent),
    m_view(new QListView(this)),
    m_contextMenu(new QMenu(this)),
    m_playAction(m_contextMenu->addAction(tr("Play"), this, SLOT(playSelectedTrack()))),
    m_removeAction(m_contextMenu->addAction(tr("Remove from playlist"), this, SLOT(removeSelectedTrack()))),
    m_clearAction(m_contextMenu->addAction(tr("Clear playlist"), MediaPlayer::instance(), SLOT(clearTracks()))),
    m_vbox(new QVBoxLayout(this))
{
    this->setWindowTitle(tr("'Now playing' list"));
    this->setFocusProxy(m_view);

    m_view->setModel(PlaybackListModel::playbackQueue());
    m_view->setItemDelegate(new TrackListDelegate(m_view));
    m_view->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_view->setContextMenuPolicy(Qt::CustomContextMenu);

    m_vbox->setContentsMargins(0, 9, 0, 0);
    m_vbox->addWidget(m_view);

    this->connect(m_view, SIGNAL(activated(QModelIndex)), this, SLOT(onListItemActivated(QModelIndex)));
    this->connect(m_view, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
}

NowPlayingListTab::~NowPlayingListTab() {}

void NowPlayingListTab::showEvent(QShowEvent *event) {
    this->connectPlaybackSignals();

    event->accept();
}

void NowPlayingListTab::hideEvent(QHideEvent *event) {
    this->disconnectPlaybackSignals();

    event->accept();
}

void NowPlayingListTab::connectPlaybackSignals() {
    this->connect(MediaPlayer::instance(), SIGNAL(currentIndexChanged(int)), this, SLOT(onCurrentIndexChanged(int)));

    this->onCurrentIndexChanged(MediaPlayer::instance()->currentIndex());
}

void NowPlayingListTab::disconnectPlaybackSignals() {
    this->disconnect(MediaPlayer::instance(), SIGNAL(currentIndexChanged(int)), this, SLOT(onCurrentIndexChanged(int)));
}

void NowPlayingListTab::onListItemActivated(const QModelIndex &index) {
    if (index.isValid()) {
        MediaPlayer::instance()->setCurrentIndex(index.row());
    }
}

void NowPlayingListTab::showContextMenu(const QPoint &pos) {
    if (m_view->currentIndex().isValid()) {
        m_contextMenu->setActiveAction(m_playAction);
        m_contextMenu->popup(this->mapToGlobal(pos), m_playAction);
    }
}

void NowPlayingListTab::playSelectedTrack() {
    if (m_view->currentIndex().isValid()) {
        MediaPlayer::instance()->setCurrentIndex(m_view->currentIndex().row());
    }
}

void NowPlayingListTab::removeSelectedTrack() {
    if (m_view->currentIndex().isValid()) {
        MediaPlayer::instance()->removeTrack(m_view->currentIndex().row());
    }
}

void NowPlayingListTab::onCurrentIndexChanged(int index) {
    m_view->setCurrentIndex(PlaybackListModel::playbackQueue()->index(index));
    m_view->scrollTo(m_view->currentIndex(), QListView::PositionAtCenter);
}
