#include "nowplayingcurrenttab.h"
#include "thumbnaillabel.h"
#include "../mediaplayer/gstreamer/mediaplayer.h"
#include "../base/thumbnailcache.h"
#include "../base/playbacklistmodel.h"
#include <QLabel>
#include <QMenu>
#include <QGridLayout>
#include <QContextMenuEvent>
#include <QShowEvent>
#include <QHideEvent>
#include <QApplication>

NowPlayingCurrentTab::NowPlayingCurrentTab(QWidget *parent) :
    QWidget(parent),
    m_trackCountLabel(new QLabel(this)),
    m_titleLabel(new QLabel(this)),
    m_artistLabel(new QLabel(this)),
    m_thumbnail(new ThumbnailLabel(this)),
    m_contextMenu(new QMenu(this)),
    m_playPauseAction(m_contextMenu->addAction(tr("Play"), MediaPlayer::instance(), SLOT(togglePlayPause()))),
    m_removeAction(m_contextMenu->addAction(tr("Remove from playlist"), this, SLOT(removeFromPlaybackQueue()))),
    m_clearAction(m_contextMenu->addAction(tr("Clear playlist"), MediaPlayer::instance(), SLOT(clearTracks()))),
    m_grid(new QGridLayout(this))
{
    this->setWindowTitle(tr("Now playing"));

    QFont largeFont;
    largeFont.setPixelSize(40);

    m_titleLabel->setFont(largeFont);
    m_titleLabel->setWordWrap(true);
    m_trackCountLabel->setAlignment(Qt::AlignTop);
    m_artistLabel->setAlignment(Qt::AlignBottom);

    m_thumbnail->setFixedSize(230, 230);
    m_thumbnail->setInteractive(false);

    m_grid->setContentsMargins(0, 0, 0, 0);
    m_grid->addWidget(m_trackCountLabel, 0, 0, 1, 1);
    m_grid->addWidget(m_titleLabel, 1, 0, 1, 1);
    m_grid->addWidget(m_thumbnail, 0, 1, 3, 1);
    m_grid->addWidget(m_artistLabel, 2, 0, 1, 1);
}

NowPlayingCurrentTab::~NowPlayingCurrentTab() {}

void NowPlayingCurrentTab::showEvent(QShowEvent *event) {
    this->connectPlaybackSignals();

    event->accept();
}

void NowPlayingCurrentTab::hideEvent(QHideEvent *event) {
    this->disconnectPlaybackSignals();

    event->accept();
}

void NowPlayingCurrentTab::connectPlaybackSignals() {
    this->connect(MediaPlayer::instance(), SIGNAL(currentTrackChanged(QSharedPointer<TrackItem>)), this, SLOT(onCurrentTrackChanged(QSharedPointer<TrackItem>)));
    this->connect(MediaPlayer::instance(), SIGNAL(currentIndexChanged(int)), this, SLOT(onCurrentIndexChanged(int)));
    this->connect(PlaybackListModel::playbackQueue(), SIGNAL(countChanged(int)), this, SLOT(onPlaybackQueueChanged(int)));

    this->onCurrentTrackChanged(MediaPlayer::instance()->currentTrack());
    this->onPlaybackQueueChanged(PlaybackListModel::playbackQueue()->rowCount());
}

void NowPlayingCurrentTab::disconnectPlaybackSignals() {
    this->disconnect(MediaPlayer::instance(), SIGNAL(currentTrackChanged(QSharedPointer<TrackItem>)), this, SLOT(onCurrentTrackChanged(QSharedPointer<TrackItem>)));
    this->disconnect(MediaPlayer::instance(), SIGNAL(currentIndexChanged(int)), this, SLOT(onCurrentIndexChanged(int)));
    this->disconnect(PlaybackListModel::playbackQueue(), SIGNAL(countChanged(int)), this, SLOT(onPlaybackQueueChanged(int)));
}

void NowPlayingCurrentTab::contextMenuEvent(QContextMenuEvent *event) {
    m_playPauseAction->setText(MediaPlayer::instance()->playing() ? tr("Pause") : tr("Play"));
    m_contextMenu->setActiveAction(m_playPauseAction);
    m_contextMenu->popup(event->globalPos(), m_playPauseAction);

    event->accept();
}

void NowPlayingCurrentTab::onCurrentTrackChanged(QSharedPointer<TrackItem> track) {
    if (track.isNull()) {
        m_titleLabel->setText(QString("(%1)").arg(tr("empty")));
        m_artistLabel->setText(QString());
        m_thumbnail->setImage(QImage());
    }
    else {
        m_titleLabel->setText(track.data()->title());
        m_artistLabel->setText(QString("<font color='%1'>%2:</font> %3").arg(QApplication::palette().color(QPalette::Mid).name()).arg(tr("Artist")).arg(track.data()->artist()));

        this->setThumbnail();
    }
}

void NowPlayingCurrentTab::onCurrentIndexChanged(int index) {
    m_trackCountLabel->setText(QString("%1 %2 %3").arg(index + 1).arg(tr("of")).arg(PlaybackListModel::playbackQueue()->rowCount()));
}

void NowPlayingCurrentTab::onPlaybackQueueChanged(int count) {
    m_trackCountLabel->setText(QString("%1 %2 %3").arg(!count ? 0 : MediaPlayer::instance()->currentIndex() + 1).arg(tr("of")).arg(count));
    m_contextMenu->setEnabled(count > 0);
}

void NowPlayingCurrentTab::setThumbnail() {
    QImage thumbnail = ThumbnailCache::instance()->thumbnail(MediaPlayer::instance()->currentTrack().data()->largeThumbnailUrl(), QSize(230, 230));

    if (thumbnail.isNull()) {
        m_thumbnail->setImage(QImage("/usr/share/themes/default/images/qgn_plat_application_media_player_image.png"));

        this->connect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnail()));
    }
    else {
        m_thumbnail->setImage(thumbnail);

        this->disconnect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnail()));
    }
}

void NowPlayingCurrentTab::removeFromPlaybackQueue() {
    MediaPlayer::instance()->removeTrack(MediaPlayer::instance()->currentIndex());
}
