#include "mediaplayertoolbar.h"
#include "../mediaplayer/gstreamer/mediaplayer.h"
#include "../base/playbacklistmodel.h"
#include "../base/utils.h"
#include <QToolButton>
#include <QLabel>
#include <QSlider>
#include <QHBoxLayout>
#include <QTimer>

MediaPlayerToolBar::MediaPlayerToolBar(QWidget *parent) :
    QWidget(parent),
    m_volumeButton(new QToolButton(this)),
    m_playbackModeButton(new QToolButton(this)),
    m_positionLabel(new QLabel("--:--", this)),
    m_durationLabel(new QLabel("--:--", this)),
    m_slider(new QSlider(Qt::Horizontal, this)),
    m_volumeTimer(new QTimer(this)),
    m_hbox(new QHBoxLayout(this))
{
    this->setFocusPolicy(Qt::NoFocus);

    m_volumeButton->setCheckable(true);
    m_volumeButton->setIcon(QIcon("/usr/share/icons/hicolor/26x26/hildon/qgn_medi_volume_unmuted.png"));

    m_playbackModeButton->setIcon(QIcon("/usr/share/icons/hicolor/26x26/hildon/qgn_medi_random_off_repeat_off.png"));

    m_slider->setTracking(false);
    m_slider->setEnabled(false);

    m_volumeTimer->setInterval(5000);
    m_volumeTimer->setSingleShot(true);

    m_hbox->addWidget(m_volumeButton);
    m_hbox->addWidget(m_playbackModeButton);
    m_hbox->addWidget(m_positionLabel);
    m_hbox->addWidget(m_slider);
    m_hbox->addWidget(m_durationLabel);

    this->connect(m_volumeButton, SIGNAL(toggled(bool)), this, SIGNAL(showVolumeControl(bool)));
    this->connect(m_volumeButton, SIGNAL(clicked(bool)), this, SLOT(onVolumeButtonClicked(bool)));
    this->connect(m_playbackModeButton, SIGNAL(clicked()), MediaPlayer::instance(), SLOT(togglePlaybackMode()));
    this->connect(m_slider, SIGNAL(sliderReleased()), this, SLOT(onSliderReleased()));
    this->connect(m_volumeTimer, SIGNAL(timeout()), this, SLOT(onVolumeTimeout()));
    this->connect(MediaPlayer::instance(), SIGNAL(positionChanged(qint64)), this, SLOT(onPlaybackPositionChanged(qint64)));
    this->connect(MediaPlayer::instance(), SIGNAL(durationChanged(qint64)), this, SLOT(onPlaybackDurationChanged(qint64)));
    this->connect(MediaPlayer::instance(), SIGNAL(playbackModeChanged(Media::PlaybackMode)), this, SLOT(onPlaybackModeChanged(Media::PlaybackMode)));
    this->connect(PlaybackListModel::playbackQueue(), SIGNAL(countChanged(int)), this, SLOT(onPlaybackQueueChanged(int)));
}

MediaPlayerToolBar::~MediaPlayerToolBar() {}

void MediaPlayerToolBar::startVolumeTimer() {
    m_volumeTimer->start();
}

void MediaPlayerToolBar::stopVolumeTimer() {
    m_volumeTimer->stop();
}

void MediaPlayerToolBar::onPlaybackQueueChanged(int count) {
    m_slider->setEnabled(count > 0);

    if (!count) {
        m_positionLabel->setText("--:--");
        m_durationLabel->setText("--:--");
    }
}

void MediaPlayerToolBar::onPlaybackPositionChanged(qint64 position) {
    m_positionLabel->setText(Utils::durationFromMSecs(position));

    if (!m_slider->isSliderDown()) {
        m_slider->setValue(position);
    }
}

void MediaPlayerToolBar::onPlaybackDurationChanged(qint64 duration) {
    m_slider->setMaximum(duration);
    m_durationLabel->setText(Utils::durationFromMSecs(duration));
}

void MediaPlayerToolBar::onPlaybackModeChanged(Media::PlaybackMode playbackMode) {
    switch (playbackMode) {
    case Media::RepeatAll:
    case Media::RepeatOne:
        m_playbackModeButton->setIcon(QIcon("/usr/share/icons/hicolor/26x26/hildon/qgn_medi_random_off_repeat_on.png"));
        return;
    default:
        m_playbackModeButton->setIcon(QIcon("/usr/share/icons/hicolor/26x26/hildon/qgn_medi_random_off_repeat_off.png"));
        return;
    }
}

void MediaPlayerToolBar::onSliderReleased() {
    MediaPlayer::instance()->setPosition(m_slider->value());
}

void MediaPlayerToolBar::onVolumeButtonClicked(bool checked) {
    if (checked) {
        this->startVolumeTimer();
    }
    else {
        this->stopVolumeTimer();
    }
}

void MediaPlayerToolBar::onVolumeTimeout() {
    m_volumeButton->setChecked(false);
}
