#include "archivetrackswindow.h"
#include "tracklistdelegate.h"
#include "trackinfowindow.h"
#include "mainwindow.h"
#include "../base/archivetrackmodel.h"
#include "../base/playbacklistmodel.h"
#include "../mediaplayer/gstreamer/mediaplayer.h"
#include <QListView>
#include <QMenu>
#include <QVBoxLayout>

ArchiveTracksWindow::ArchiveTracksWindow(QWidget *parent) :
    QWidget(parent),
    m_model(new ArchiveTrackModel(this)),
    m_view(new QListView(this)),
    m_contextMenu(new QMenu(this)),
    m_viewAction(m_contextMenu->addAction(tr("View"), this, SLOT(showSelectedTrack()))),
    m_playAction(m_contextMenu->addAction(tr("Play"), this, SLOT(playSelectedTrack()))),
    m_queueAction(m_contextMenu->addAction(tr("Queue"), this, SLOT(queueSelectedTrack()))),
    m_vbox(new QVBoxLayout(this))
{
    this->setWindowTitle(tr("Tracks"));
    this->setAttribute(Qt::WA_DeleteOnClose);
    this->setFocusProxy(m_view);

    m_view->setModel(m_model);
    m_view->setItemDelegate(new TrackListDelegate(m_view));
    m_view->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_view->setContextMenuPolicy(Qt::CustomContextMenu);

    m_vbox->setContentsMargins(0, 0, 0, 0);
    m_vbox->addWidget(m_view);

    this->connect(m_view, SIGNAL(activated(QModelIndex)), this, SLOT(onListItemActivated(QModelIndex)));
    this->connect(m_view, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));

    m_model->getTracks();
}

ArchiveTracksWindow::~ArchiveTracksWindow() {}

void ArchiveTracksWindow::onListItemActivated(const QModelIndex &index) {
    if (index.isValid()) {
        TrackInfoWindow *window = new TrackInfoWindow(m_model->get(index.row()), MainWindow::instance());
        MainWindow::instance()->push(window);
    }
}

void ArchiveTracksWindow::showContextMenu(const QPoint &pos) {
    if (m_view->currentIndex().isValid()) {
        m_contextMenu->setActiveAction(m_viewAction);
        m_contextMenu->popup(this->mapToGlobal(pos), m_viewAction);
    }
}

void ArchiveTracksWindow::showSelectedTrack() {
    if (m_view->currentIndex().isValid()) {
        TrackInfoWindow *window = new TrackInfoWindow(m_model->get(m_view->currentIndex().row()), MainWindow::instance());
        MainWindow::instance()->push(window);
    }
}

void ArchiveTracksWindow::playSelectedTrack() {
    if (m_view->currentIndex().isValid()) {
        MediaPlayer::instance()->playTrack(m_model->get(m_view->currentIndex().row()));
    }
}

void ArchiveTracksWindow::queueSelectedTrack() {
    if (m_view->currentIndex().isValid()) {
        PlaybackListModel::playbackQueue()->addTrack(m_model->get(m_view->currentIndex().row()));
    }
}
