#include "grouplistmodel.h"
#include "feedurls.h"
#include "json.h"
#include "utils.h"
#include "notifications.h"
#include "soundcloud.h"
#ifndef QML_USER_INTERFACE
#include "thumbnailcache.h"
#endif
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QTimer>

using namespace QtJson;

GroupListModel::GroupListModel(QObject *parent) :
    QAbstractListModel(parent),
    #ifndef QML_USER_INTERFACE
    m_cache(new ThumbnailCache),
    #endif
    m_loading(true),
    m_moreResults(true),
    m_error(false),
    m_offset(0)
{
#if QT_VERSION >= 0x040600
    m_roleNames[IdRole] = "id";
    m_roleNames[TitleRole] = "title";
    m_roleNames[ArtistRole] = "artist";
    m_roleNames[DateRole] = "date";
#ifndef QML_USER_INTERFACE
    m_roleNames[ThumbnailRole] = "thumbnail";
#endif
    m_roleNames[ThumbnailUrlRole] = "thumbnailUrl";
    m_roleNames[DescriptionRole] = "description";
#if QT_VERSION < 0x050000
    this->setRoleNames(m_roleNames);
#endif
#endif
#ifndef QML_USER_INTERFACE
    this->connect(m_cache, SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
#endif
}

GroupListModel::~GroupListModel() {
    m_list.clear();
#ifndef QML_USER_INTERFACE
    delete m_cache;
    m_cache = 0;
#endif
}

#if QT_VERSION >= 0x050000
QHash<int, QByteArray> GroupListModel::roleNames() const {
    return m_roleNames;
}
#endif

void GroupListModel::clear() {
    this->beginRemoveRows(QModelIndex(), 0, this->rowCount() - 1);
    m_list.clear();
    this->endRemoveRows();
    this->setLoading(false);
    this->setOffset(0);
    this->setMoreResults(true);
    this->setError(false);
}

void GroupListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getMoreGroups();
    }
}

int GroupListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return m_list.size();
}

QVariant GroupListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case IdRole:
        return m_list.at(index.row()).data()->id();
    case TitleRole:
        return m_list.at(index.row()).data()->title();
    case ArtistRole:
        return m_list.at(index.row()).data()->artist();
    case DateRole:
        return m_list.at(index.row()).data()->date();
#ifndef QML_USER_INTERFACE
    case ThumbnailRole:
        return m_cache->thumbnail(m_list.at(index.row()).data()->thumbnailUrl(), QSize(64, 64));
#endif
    case ThumbnailUrlRole:
        return m_list.at(index.row()).data()->thumbnailUrl();
    case DescriptionRole:
        return m_list.at(index.row()).data()->description();
    default:
        return QVariant();
    }
}

#if QT_VERSION >= 0x040600
QVariant GroupListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}
#endif

QSharedPointer<GroupItem> GroupListModel::get(int row) const {
    if ((row >= 0) && (row < m_list.size())) {
        return QSharedPointer<GroupItem>(m_list.at(row));
    }

    return QSharedPointer<GroupItem>();
}

#ifdef QML_USER_INTERFACE
GroupItem* GroupListModel::getFromQML(int row) const {
    if ((row >= 0) && (row < m_list.size())) {
        return m_list.at(row).data();
    }

    return 0;
}
#endif

void GroupListModel::insertGroup(int row, QSharedPointer<GroupItem> group) {
    Q_ASSERT((row >= 0) && (row <= this->rowCount()));

    this->beginInsertRows(QModelIndex(), row, row);
    m_list.insert(row, group);
    this->endInsertRows();
}

void GroupListModel::appendGroup(QSharedPointer<GroupItem> group) {
    this->beginInsertRows(QModelIndex(), this->rowCount(), this->rowCount());
    m_list.append(group);
    this->endInsertRows();
}

bool GroupListModel::removeGroup(const QVariant &value, int role) {
    QModelIndexList indexes = this->match(this->index(0), role, value, 1, Qt::MatchExactly);

    if (indexes.isEmpty()) {
        return false;
    }

    return this->removeRow(indexes.first().row());
}

bool GroupListModel::removeRow(int row, const QModelIndex &parent)
{
    Q_UNUSED(parent)

    if ((row >= 0) && (row < m_list.size())) {
        this->beginRemoveRows(QModelIndex(), row, row);
        m_list.takeAt(row).clear();
        this->endRemoveRows();

        return true;
    }

    return false;
}

#ifndef QML_USER_INTERFACE
void GroupListModel::onThumbnailReady() {
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
}
#endif

void GroupListModel::setFeed(const QUrl &feed) {
    if (feed != this->feed()) {
        m_feed = feed;
    }
}

void GroupListModel::getGroups(QUrl feed) {
    if (!feed.isEmpty()) {
        this->setFeed(feed);
    }
    else {
        feed = this->feed();
    }

    this->setLoading(true);
    QNetworkReply *reply = SoundCloud::instance()->createReply(feed, this->offset());
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addGroups()));
}

void GroupListModel::search(const QString &query, int order) {
    if (!query.isEmpty()) {
        this->setQuery(query);
        this->setOrder(order);
    }

    this->setLoading(true);
    QNetworkReply *reply = SoundCloud::instance()->createSearchReply(Queries::Groups, this->query(), this->order(), this->offset());
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addGroups()));
}

void GroupListModel::addGroups() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setLoading(false);
        this->setError(true);
        Notifications::instance()->onError(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    switch (statusCode) {
    case 200:
    case 201: {
        QString response(reply->readAll());
        bool ok;
        QVariantList entries = Json::parse(response, ok).toList();

        if (!ok) {
            this->setLoading(false);
            this->setError(true);
            Notifications::instance()->onError(tr("Cannot parse server response"));
        }
        else {
            for (int i = 0; i < entries.size(); i++) {
                QMetaObject::invokeMethod(this, "appendGroup", Qt::QueuedConnection, Q_ARG(QSharedPointer<GroupItem>, QSharedPointer<GroupItem>(new GroupItem(entries.at(i).toMap()))));
            }

            this->setMoreResults(!entries.isEmpty());
            this->setOffset(this->offset() + entries.size());
            QTimer::singleShot(1000, this, SLOT(stopLoading()));
        }
    }
        break;
    default:
        this->setLoading(false);
        this->setError(true);
        Notifications::instance()->onError(Utils::httpErrorString(statusCode));
        break;
    }

    reply->deleteLater();
}

void GroupListModel::getMoreGroups() {
    if ((this->moreResults()) && (!this->loading())) {
        if (!this->query().isEmpty()) {
            this->search();
        }
        else {
            this->getGroups();
        }
    }
}
