#include "volumecontrol.h"
#include <QSlider>
#include <QLabel>
#include <QVBoxLayout>
#include <QPaintEvent>
#include <QPainter>

VolumeButton::VolumeButton(Role buttonRole, QWidget *parent) :
    QAbstractButton(parent),
    m_role(buttonRole)
{
    this->setFixedSize(26, 26);

    switch (m_role) {
    case Up:
    case Down:
        this->setAutoRepeat(true);
        this->setAutoRepeatDelay(250);
        this->setAutoRepeatInterval(100);
        break;
    case Mute:
        this->setCheckable(true);
        break;
    default:
        break;
    }
}

VolumeButton::~VolumeButton() {}

void VolumeButton::paintEvent(QPaintEvent *event) {
    QPainter painter(this);

    QString state("normal");

    if (!this->isEnabled()) {
        state = "disabled";
    }
    else if ((this->isDown()) || (this->isChecked())) {
        state = "pressed";
    }

    switch (m_role) {
    case Up:
        painter.drawImage(this->rect(), QImage(QString("/usr/share/themes/default/images/qgn_plat_volume_bar_vertical_regular_more_button_%1.png").arg(state)));
        break;
    case Down:
        painter.drawImage(this->rect(), QImage(QString("/usr/share/themes/default/images/qgn_plat_volume_bar_vertical_regular_less_button_%1.png").arg(state)));
        break;
    case Mute:
        painter.drawImage(this->rect(), QImage(QString("/usr/share/themes/default/images/qgn_plat_volume_bar_vertical_regular_mute_button_%1.png").arg(state)));
        break;
    default:
        this->icon().paint(&painter, this->rect());
        break;
    }

    event->accept();
}

VolumeControl::VolumeControl(QWidget *parent) :
    QWidget(parent),
    m_slider(new QSlider(Qt::Vertical, this)),
    m_upButton(new VolumeButton(VolumeButton::Up, this)),
    m_downButton(new VolumeButton(VolumeButton::Down, this)),
    m_muteButton(new VolumeButton(VolumeButton::Mute, this)),
    m_grid(new QGridLayout(this)),
    m_min(false),
    m_max(false)
{
    this->setFixedWidth(112);

    m_slider->setTracking(true);
    m_slider->setMaximum(100);
    m_slider->setSingleStep(5);

    m_grid->setContentsMargins(0, 30, 0, 30);
    m_grid->setSpacing(20);
    m_grid->addWidget(m_upButton, 0, 0, Qt::AlignHCenter);
    m_grid->addWidget(m_slider, 1, 0, Qt::AlignHCenter);
    m_grid->addWidget(m_downButton, 2, 0, Qt::AlignHCenter);
    m_grid->addWidget(m_muteButton, 3, 0, Qt::AlignHCenter);

    this->connect(m_slider, SIGNAL(valueChanged(int)), this, SLOT(onSliderValueChanged(int)));
    this->connect(m_slider, SIGNAL(sliderPressed()), this, SIGNAL(pressed()));
    this->connect(m_slider, SIGNAL(sliderReleased()), this, SIGNAL(released()));
    this->connect(m_upButton, SIGNAL(pressed()), this, SIGNAL(pressed()));
    this->connect(m_upButton, SIGNAL(released()), this, SIGNAL(released()));
    this->connect(m_upButton, SIGNAL(clicked()), this, SLOT(onVolumeUpClicked()));
    this->connect(m_downButton, SIGNAL(pressed()), this, SIGNAL(pressed()));
    this->connect(m_downButton, SIGNAL(released()), this, SIGNAL(released()));
    this->connect(m_downButton, SIGNAL(clicked()), this, SLOT(onVolumeDownClicked()));
    this->connect(m_muteButton, SIGNAL(pressed()), this, SIGNAL(pressed()));
    this->connect(m_muteButton, SIGNAL(released()), this, SIGNAL(released()));
    this->connect(m_muteButton, SIGNAL(clicked(bool)), m_slider, SLOT(setDisabled(bool)));
    this->connect(m_muteButton, SIGNAL(clicked(bool)), m_upButton, SLOT(setDisabled(bool)));
    this->connect(m_muteButton, SIGNAL(clicked(bool)), m_downButton, SLOT(setDisabled(bool)));
    this->connect(m_muteButton, SIGNAL(clicked(bool)), this, SIGNAL(mutedChanged(bool)));

    m_slider->setValue(50);
}

VolumeControl::~VolumeControl() {}

qreal VolumeControl::value() const {
    return !m_slider->value() ? 0 : qreal(m_slider->value()) / 100;
}

void VolumeControl::setValue(qreal value) {
    m_slider->setValue(value * 100);
}

bool VolumeControl::muted() const {
    return m_muteButton->isChecked();
}

void VolumeControl::setMuted(bool muted) {
    m_muteButton->setChecked(muted);
}

void VolumeControl::onSliderValueChanged(int value) {
    switch (value) {
    case 0:
        if (!m_min) {
            m_min = true;
            m_downButton->setEnabled(false);
        }

        break;
    case 100:
        if (!m_max) {
            m_max = true;
            m_upButton->setEnabled(false);
        }

        break;
    default:
        if (m_max) {
            m_max = false;
            m_upButton->setEnabled(true);
        }
        else if (m_min) {
            m_min = false;
            m_downButton->setEnabled(true);
        }

        break;
    }

    emit valueChanged(qreal(value) / 100);
}

void VolumeControl::onVolumeUpClicked() {
    m_slider->setValue(m_slider->value() + 5);
}

void VolumeControl::onVolumeDownClicked() {
    m_slider->setValue(m_slider->value() - 5);
}
