#include "activitylistwindow.h"
#include "activitylistdelegate.h"
#include "mainwindow.h"
#include "trackinfowindow.h"
#include "playlisttrackswindow.h"
#include "../base/activitylistmodel.h"
#include "../base/soundcloud.h"
#include <QListView>
#include <QMenu>
#include <QVBoxLayout>

ActivityListWindow::ActivityListWindow(QWidget *parent) :
    QWidget(parent),
    m_model(new ActivityListModel(this)),
    m_view(new QListView(this)),
    m_contextMenu(new QMenu(this)),
    m_viewAction(m_contextMenu->addAction(tr("View"), this, SLOT(showSelectedItem()))),
    m_playAction(m_contextMenu->addAction(tr("Play"), this, SLOT(playSelectedItem()))),
    m_separatorAction(m_contextMenu->addSeparator()),
    m_moreResultsAction(m_contextMenu->addAction(tr("Load more activities"), m_model, SLOT(getMoreActivities()))),
    m_vbox(new QVBoxLayout(this))
{
    this->setWindowTitle(tr("Dashboard"));
    this->setAttribute(Qt::WA_DeleteOnClose);
    this->setFocusProxy(m_view);

    m_view->setModel(m_model);
    m_view->setItemDelegate(new ActivityListDelegate(m_view));
    m_view->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_view->setContextMenuPolicy(Qt::CustomContextMenu);

    m_vbox->setContentsMargins(0, 0, 0, 0);
    m_vbox->addWidget(m_view);

    this->connect(m_view, SIGNAL(activated(QModelIndex)), this, SLOT(onListItemActivated(QModelIndex)));
    this->connect(m_view, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));

    m_model->getActivities(ALL_ACTIVITIES_FEED);
}

ActivityListWindow::~ActivityListWindow() {}

void ActivityListWindow::onListItemActivated(const QModelIndex &index) {
    if (index.isValid()) {
        switch (index.data(ActivityListModel::ActivityTypeRole).toInt()) {
        case Activities::Playlist:
            SoundCloud::instance()->getFullPlaylist(index.data(ActivityListModel::IdRole).toString());
            this->connect(SoundCloud::instance(), SIGNAL(gotPlaylist(QSharedPointer<PlaylistItem>)), this, SLOT(showPlaylist(QSharedPointer<PlaylistItem>)));
            return;
        default:
            SoundCloud::instance()->getFullTrack(index.data(ActivityListModel::IdRole).toString());
            this->connect(SoundCloud::instance(), SIGNAL(gotTrack(QSharedPointer<TrackItem>)), this, SLOT(showTrack(QSharedPointer<TrackItem>)));
            return;
        }
    }
}

void ActivityListWindow::showContextMenu(const QPoint &pos) {
    if (m_view->currentIndex().isValid()) {
        switch (m_view->currentIndex().data(ActivityListModel::ActivityTypeRole).toInt()) {
        case Activities::Playlist:
            m_playAction->setEnabled(false);
            break;
        default:
            m_playAction->setEnabled(true);
            break;
        }

        m_moreResultsAction->setEnabled(m_model->moreResults());
        m_contextMenu->setActiveAction(m_viewAction);
        m_contextMenu->popup(this->mapToGlobal(pos), m_viewAction);
    }
}

void ActivityListWindow::showSelectedItem() {
    if (m_view->currentIndex().isValid()) {
        switch (m_view->currentIndex().data(ActivityListModel::ActivityTypeRole).toInt()) {
        case Activities::Playlist:
            SoundCloud::instance()->getFullPlaylist(m_view->currentIndex().data(ActivityListModel::IdRole).toString());
            this->connect(SoundCloud::instance(), SIGNAL(gotPlaylist(QSharedPointer<PlaylistItem>)), this, SLOT(showPlaylist(QSharedPointer<PlaylistItem>)));
            return;
        default:
            SoundCloud::instance()->getFullTrack(m_view->currentIndex().data(ActivityListModel::IdRole).toString());
            this->connect(SoundCloud::instance(), SIGNAL(gotTrack(QSharedPointer<TrackItem>)), this, SLOT(showTrack(QSharedPointer<TrackItem>)));
            return;
        }
    }
}

void ActivityListWindow::playSelectedItem() {
    if (m_view->currentIndex().isValid()) {
        SoundCloud::instance()->getFullTrack(m_view->currentIndex().data(ActivityListModel::IdRole).toString());
        this->connect(SoundCloud::instance(), SIGNAL(gotTrack(QSharedPointer<TrackItem>)), this, SLOT(playTrack(QSharedPointer<TrackItem>)));
    }
}

void ActivityListWindow::showTrack(QSharedPointer<TrackItem> track) {
    this->disconnect(SoundCloud::instance(), SIGNAL(gotTrack(QSharedPointer<TrackItem>)), this, SLOT(showTrackInfo(QSharedPointer<TrackItem>)));

    TrackInfoWindow *window = new TrackInfoWindow(track, MainWindow::instance());
    MainWindow::instance()->push(window);
}

void ActivityListWindow::showPlaylist(QSharedPointer<PlaylistItem> playlist) {
    this->disconnect(SoundCloud::instance(), SIGNAL(gotPlaylist(QSharedPointer<PlaylistItem>)), this, SLOT(showPlaylist(QSharedPointer<PlaylistItem>)));

    PlaylistTracksWindow *window = new PlaylistTracksWindow(playlist, MainWindow::instance());
    MainWindow::instance()->push(window);
}

void ActivityListWindow::playTrack(QSharedPointer<TrackItem> track) {
    this->disconnect(SoundCloud::instance(), SIGNAL(gotTrack(QSharedPointer<TrackItem>)), this, SLOT(playTrack(QSharedPointer<TrackItem>)));

    MediaPlayer::instance()->playTrack(track);
}
