#ifndef SOUNDCLOUD_H
#define SOUNDCLOUD_H

#include "enums.h"
#include "feedurls.h"
#include "trackitem.h"
#include "playlistitem.h"
#include "useritem.h"
#include "commentitem.h"
#include <QObject>
#include <QSharedPointer>

class QNetworkAccessManager;
class QNetworkReply;
class QUrl;

class SoundCloud : public QObject
{
    Q_OBJECT

#if QT_VERSION >= 0x040700
    Q_PROPERTY(QString username
               READ username
               NOTIFY usernameChanged)
    Q_PROPERTY(bool userSignedIn
               READ userSignedIn
               NOTIFY userSignedInChanged)
    Q_PROPERTY(QUrl uploadsFeed
               READ uploadsFeed
               CONSTANT)
    Q_PROPERTY(QUrl favouritesFeed
               READ favouritesFeed
               CONSTANT)
    Q_PROPERTY(QUrl playlistsFeed
               READ playlistsFeed
               CONSTANT)
    Q_PROPERTY(QUrl groupsFeed
               READ groupsFeed
               CONSTANT)
    Q_PROPERTY(QUrl followingsFeed
               READ followingsFeed
               CONSTANT)
    Q_PROPERTY(QUrl allActivitiesFeed
               READ allActivitiesFeed
               CONSTANT)
    Q_PROPERTY(QUrl authUrl
               READ authUrl
               CONSTANT)
    Q_PROPERTY(bool busy
               READ busy
               NOTIFY busyChanged)
#endif

public:
    explicit SoundCloud(QObject *parent = 0);
    ~SoundCloud();
    QUrl authUrl();
    inline QNetworkAccessManager* networkAccessManager() const { return m_nam; }
    inline void setNetworkAccessManager(QNetworkAccessManager *manager) { m_nam = manager; }
    inline QUrl uploadsFeed() const { return UPLOADS_FEED; }
    inline QUrl favouritesFeed() const { return FAVOURITES_FEED; }
    inline QUrl playlistsFeed() const { return PLAYLISTS_FEED; }
    inline QUrl groupsFeed() const { return GROUPS_FEED; }
    inline QUrl followingsFeed() const { return FOLLOWINGS_FEED; }
    inline QUrl allActivitiesFeed() const { return ALL_ACTIVITIES_FEED; }
    inline QString username() const { return m_username; }
    inline QString accessToken() const { return m_accessToken; }
    inline bool userSignedIn() const { return !this->accessToken().isEmpty(); }
    QNetworkReply* createReply(QUrl url, int offset = 0);
    QNetworkReply* createSearchReply(int queryType, const QString &query, int order, int offset = 0);
    Q_INVOKABLE QUrl getStreamUrl(QUrl url);
    Q_INVOKABLE QUrl getDownloadUrl(QUrl url);
    inline bool busy() const { return m_busy; }
    inline bool cancelled() const { return m_cancelled; }

    static SoundCloud* instance();

public slots:
    void signIn(const QUrl &response);
    void signIn(const QString &user, const QString &pass);
    void resolve(const QString &urlToResolve);
    void getFullTrack(const QString &id);
    void getFullPlaylist(const QString &id);
    void getUserProfile(const QString &id);
    void isUserFollowed(const QString &id);
    void getCurrentUserProfile();
    void updateProfile(const QVariantMap &profile);
    void shareTrack(const QString &id, const QStringList &connections, const QString &message);
    void addComment(const QString &id, const QString &comment, int timestamp = 0);
    void setCredentials(const QString &user, const QString &token);
    void setAccessToken(const QString &token);
    void setUsername(const QString &user);
    void addToFavourites(const QStringList &ids);
    void deleteFromFavourites(const QStringList &ids);
    void followUser(const QString &id);
    void unfollowUser(const QString &id);
    void trackIsFavourite(const QString &id);
    void cancelCurrentOperation();

private:
    void addToFavourites();
    void deleteFromFavourites();
    void setBusy(bool isBusy, const QString &message = QString(), int numberOfOperations = 1);
    inline void setCancelled(bool cancelled) { m_cancelled = cancelled; }

private slots:
    void checkAccessToken();
    void postRequest(const QUrl &url, const QByteArray &data);
    void putRequest(const QUrl &url);
    void deleteRequest(const QUrl &url);
    void postFinished();
    void checkResolvedUrl();
    void checkTrack();
    void checkPlaylist();
    void checkUser();
    void checkCurrentUser();
    void checkIfFavourite();
    void checkIfUserFollowed();
    void onAddedToFavourites();
    void onDeletedFromFavourites();
    void onUserFollowed();
    void onUserUnfollowed();
    void onCommentAdded(const QString &response);
    void onTrackShared();
    void onProfileUpdated(const QString &response);
    void onTrackActionError(const QString &errorString);
    void onUserActionError(const QString &errorString);

signals:
    void alert(const QString &message);
    void busy(const QString &message, int numberOfOperations = 1);
    void busyProgressChanged(int progress);
    void busyChanged(bool isBusy);
    void currentOperationCancelled();
    void info(const QString &message);
    void error(const QString &errorString);
    void signedIn(const QString &token);
    void gotUsername(const QString &user);
    void accessTokenChanged();
    void usernameChanged();
    void userSignedInChanged(bool signedIn);
    void postSuccessful(const QString &response);
    void postFailed(const QString &error);
    void commentAdded(QSharedPointer<CommentItem> comment);
    void favouriteChanged(const QString &id, bool favourite);
    void followingChanged(const QString &id, bool following);
    void profileUpdated(const QString &id, const QVariantMap &profile);
    void gotResourceId(const QString &id, Queries::QueryType type);
#ifdef QML_USER_INTERFACE
    void gotTrack(TrackItem *track);
    void gotPlaylist(PlaylistItem *playlist);
    void gotUser(UserItem *userItem);
#else
    void gotTrack(QSharedPointer<TrackItem> track);
    void gotPlaylist(QSharedPointer<PlaylistItem> playlist);
    void gotUser(QSharedPointer<UserItem> user);
#endif

private:
    static SoundCloud* m_instance;
    QNetworkAccessManager *m_nam;
    QString m_username;
    QString m_accessToken;
    QStringList m_actionIds;
    QString m_actionId;
    int m_actionsProcessed;
    bool m_busy;
    bool m_cancelled;
};

#endif // SOUNDCLOUD_H
