#include "playbacklistmodel.h"
#include "transfermanager.h"
#ifndef QML_USER_INTERFACE
#include "thumbnailcache.h"
#endif

PlaybackListModel* PlaybackListModel::m_queue = 0;

PlaybackListModel::PlaybackListModel(QObject *parent) :
    QAbstractListModel(parent)
  #ifndef QML_USER_INTERFACE
  ,m_cache(new ThumbnailCache),
    m_thumbnailPressedRow(-1)
  #endif
{
    if (!m_queue) {
        m_queue = this;
    }

#if QT_VERSION >= 0x040600
    QHash<int, QByteArray> roles;
    roles[IdRole] = "id";
    roles[PlaylistIdRole] = "playlistId";
    roles[TitleRole] = "title";
    roles[ArtistRole] = "artist";
    roles[DateRole] = "date";
    roles[DurationRole] = "duration";
    roles[UrlRole] = "url";
    roles[StreamUrlRole] = "streamUrl";
#ifndef QML_USER_INTERFACE
    roles[ThumbnailRole] = "thumbnail";
    roles[ThumbnailPressedRole] = "thumbnailPressed";
#endif
    roles[ThumbnailUrlRole] = "thumbnailUrl";
    roles[ServiceRole] = "service";
    roles[FavouriteRole] = "favourite";
    roles[SelectedRole] = "selected";
    this->setRoleNames(roles);
#endif
#ifndef QML_USER_INTERFACE
    this->connect(m_cache, SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
#endif
}

PlaybackListModel::~PlaybackListModel() {
    m_list.clear();
#ifndef QML_USER_INTERFACE
    delete m_cache;
    m_cache = 0;
#endif
}

PlaybackListModel* PlaybackListModel::playbackQueue() {
    return !m_queue ? new PlaybackListModel : m_queue;
}

void PlaybackListModel::clear() {
    this->beginRemoveRows(QModelIndex(), 0, this->rowCount() - 1);
    m_list.clear();
    this->endRemoveRows();
    emit countChanged(this->rowCount());
}

int PlaybackListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return m_list.size();
}

QVariant PlaybackListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case IdRole:
        return m_list.at(index.row()).data()->id();
    case PlaylistIdRole:
        return m_list.at(index.row()).data()->playlistId();
    case TitleRole:
        return m_list.at(index.row()).data()->title();
    case ArtistRole:
        return m_list.at(index.row()).data()->artist();
    case DateRole:
        return m_list.at(index.row()).data()->date();
    case DurationRole:
        return m_list.at(index.row()).data()->duration();
    case UrlRole:
        return m_list.at(index.row()).data()->url();
    case StreamUrlRole:
        return m_list.at(index.row()).data()->streamUrl();
#ifndef QML_USER_INTERFACE
    case ThumbnailRole:
        return m_cache->thumbnail(m_list.at(index.row()).data()->thumbnailUrl(), QSize(64, 64));
    case ThumbnailPressedRole:
        return QVariant(m_thumbnailPressedRow == index.row());
#endif
    case ThumbnailUrlRole:
        return m_list.at(index.row()).data()->thumbnailUrl();
    case ServiceRole:
        return m_list.at(index.row()).data()->service();
    case FavouriteRole:
        return m_list.at(index.row()).data()->favourite();
    case SelectedRole:
        return QVariant(m_selectedRows.contains(index.row()));
    default:
        return QVariant();
    }
}

#if QT_VERSION >= 0x040600
QVariant PlaybackListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}
#endif

#ifndef QML_USER_INTERFACE
bool PlaybackListModel::setData(const QModelIndex &index, const QVariant &value, int role) {
    switch (role) {
    case ThumbnailPressedRole:
        m_thumbnailPressedRow = value.toInt();
        break;
    default:
        return false;
    }

    emit dataChanged(index, index);
    return true;
}
#endif

QSharedPointer<TrackItem> PlaybackListModel::get(int row) const {
    if ((row >= 0) && (row < m_list.size())) {
        return QSharedPointer<TrackItem>(m_list.at(row));
    }

    return QSharedPointer<TrackItem>();
}

#ifdef QML_USER_INTERFACE
TrackItem* PlaybackListModel::getFromQML(int row) const {
    if ((row >= 0) && (row < m_list.size())) {
        return m_list.at(row).data();
    }

    return 0;
}
#endif

void PlaybackListModel::insertTrack(int row, QSharedPointer<TrackItem> track) {
    Q_ASSERT((row >= 0) && (row <= this->rowCount()));

    this->beginInsertRows(QModelIndex(), row, row);
    m_list.insert(row, track);
    this->endInsertRows();
}

void PlaybackListModel::appendTrack(QSharedPointer<TrackItem> track) {
    this->beginInsertRows(QModelIndex(), this->rowCount(), this->rowCount());
    m_list.append(track);
    this->endInsertRows();
}

void PlaybackListModel::addTrack(QSharedPointer<TrackItem> track, bool notify) {
    this->appendTrack(QSharedPointer<TrackItem>(track));
    emit countChanged(this->rowCount());

    if (notify) {
        emit alert(tr("Track added to playback queue"));
    }
}

void PlaybackListModel::addTracks(QList< QSharedPointer<TrackItem> > tracks, bool notify) {
    while (!tracks.isEmpty()) {
        this->appendTrack(QSharedPointer<TrackItem>(tracks.takeFirst()));
    }

    emit countChanged(this->rowCount());

    if (notify) {
        emit alert(tr("Tracks added to playback queue"));
    }
}

#ifdef QML_USER_INTERFACE
void PlaybackListModel::addTrackFromQML(TrackItem *track, bool notify) {
    this->appendTrack(QSharedPointer<TrackItem>(new TrackItem(track)));
    emit countChanged(this->rowCount());

    if (notify) {
        emit alert(tr("Track added to playback queue"));
    }
}

void PlaybackListModel::addTracksFromQML(QList<TrackItem *> tracks, bool notify) {
    while (!tracks.isEmpty()) {
        this->appendTrack(QSharedPointer<TrackItem>(new TrackItem(tracks.takeFirst())));
    }

    emit countChanged(this->rowCount());

    if (notify) {
        emit alert(tr("Tracks added to playback queue"));
    }
}
#endif

void PlaybackListModel::removeTrack(int row) {
    if (this->removeRow(row)) {
        emit countChanged(this->rowCount());
    }
}

bool PlaybackListModel::removeRow(int row, const QModelIndex &parent)
{
    Q_UNUSED(parent)

    if ((row >= 0) && (row < m_list.size())) {
        this->beginRemoveRows(QModelIndex(), row, row);
        m_list.takeAt(row).clear();
        this->endRemoveRows();

        return true;
    }

    return false;
}

void PlaybackListModel::toggleSelected(int row) {
    if (!m_selectedRows.contains(row)) {
        m_selectedRows.append(row);
    }
    else {
        m_selectedRows.removeOne(row);
    }

    emit dataChanged(this->index(row), this->index(row));
}

void PlaybackListModel::selectAll() {
    m_selectedRows.clear();

    for (int i = 0; i < this->rowCount(); i++) {
        m_selectedRows.append(i);
    }

    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
}

void PlaybackListModel::selectNone() {
    m_selectedRows.clear();

    if (this->rowCount() > 0) {
        emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
    }
}

QList< QSharedPointer<TrackItem> > PlaybackListModel::selectedItems() const {
    QList< QSharedPointer<TrackItem> > items;

    for (int i = 0; i < m_selectedRows.size(); i++) {
        items.append(this->get(m_selectedRows.at(i)));
    }

    return items;
}

#ifdef QML_USER_INTERFACE
QList<TrackItem*> PlaybackListModel::selectedItemsFromQML() const {
    QList<TrackItem*> items;

    for (int i = 0; i < m_selectedRows.size(); i++) {
        items.append(this->getFromQML(m_selectedRows.at(i)));
    }

    return items;
}
#endif

#ifndef QML_USER_INTERFACE
void PlaybackListModel::onThumbnailReady() {
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
}
#endif

void PlaybackListModel::removeSelectedTracks() {
    qSort(m_selectedRows.begin(), m_selectedRows.end(), qGreater<int>());

    for (int i = 0; i < m_selectedRows.size(); i++) {
        this->removeTrack(m_selectedRows.at(i));
    }

    this->selectNone();
    emit countChanged(this->rowCount());
}

void PlaybackListModel::downloadSelectedTracks() {
    TransferManager::instance()->addDownloadTransfers(this->selectedItems());
    this->selectNone();
}
