#include "oauth.h"
#include "utils.h"
#include <QCryptographicHash>
#include <QStringList>

namespace QtOAuth {

QString OAuth::getTimeStamp() {
    return QString::number(Utils::currentMSecsSinceEpoch() / 1000);
}

QString OAuth::getNonce() {
    QString chars("0123456789ABCDEFGHIJKLMNOPQRSTUVWXTZabcdefghiklmnopqrstuvwxyz");
    QString result;

    int max = chars.size();
    int random;

    for (int i = 0; i < 41; i++) {
        random = qrand() % (max);
        result.append(chars.at(random));
    }

    return result;
}

QByteArray OAuth::createSignature(QByteArray key, QByteArray baseString) {

    int blockSize = 64; // HMAC-SHA-1 block size, defined in SHA-1 standard
    if (key.length() > blockSize) { // if key is longer than block size (64), reduce key length with SHA-1 compression
        key = QCryptographicHash::hash(key, QCryptographicHash::Sha1);
    }

    QByteArray innerPadding(blockSize, char(0x36)); // initialize inner padding with char "6"
    QByteArray outerPadding(blockSize, char(0x5c)); // initialize outer padding with char "\"
    // ascii characters 0x36 ("6") and 0x5c ("\") are selected because they have large
    // Hamming distance (http://en.wikipedia.org/wiki/Hamming_distance)

    for (int i = 0; i < key.length(); i++) {
        innerPadding[i] = innerPadding[i] ^ key.at(i); // XOR operation between every byte in key and innerpadding, of key length
        outerPadding[i] = outerPadding[i] ^ key.at(i); // XOR operation between every byte in key and outerpadding, of key length
    }

    // result = hash ( outerPadding CONCAT hash ( innerPadding CONCAT baseString ) ).toBase64
    QByteArray total = outerPadding;
    QByteArray part = innerPadding;
    part.append(baseString);
    total.append(QCryptographicHash::hash(part, QCryptographicHash::Sha1));
    QByteArray hashed = QCryptographicHash::hash(total, QCryptographicHash::Sha1);

    return hashed.toBase64();
}

QByteArray OAuth::createOAuthHeader(const QString &method, const QUrl &url, QMap<QString, QString> params) {
    QList< QPair<QString, QString> > queryItems = url.queryItems();
    QString nonce(OAuth::getNonce());
    QString timeStamp(OAuth::getTimeStamp());

    for (int i = 0; i < queryItems.size(); i++) {
        params.insert(queryItems.at(i).first, queryItems.at(i).second);
    }

    params.insert("oauth_nonce", nonce);
    params.insert("oauth_timestamp", timeStamp);
    params.insert("oauth_signature_method", "HMAC-SHA1");
    params.insert("oauth_version", "1.0");

    QStringList queryList = params.keys();
    queryList.sort();

    QString baseString;

    while (!queryList.isEmpty()) {
        QString query = queryList.takeFirst();

        if ((!query.isEmpty()) && (query != "oauth_consumer_secret") && (query != "oauth_token_secret")) {
            baseString.append(query.toUtf8().toPercentEncoding());
            baseString.append("=");
            baseString.append(params.value(query).toUtf8().toPercentEncoding());

            if (!queryList.isEmpty()) {
                baseString.append("&");
            }
        }
    }

    baseString = baseString.toUtf8().toPercentEncoding();
    baseString.prepend(method.toUpper() + "&" + url.scheme() + "%3A%2F%2F" + url.host().toUtf8().toPercentEncoding() + url.path().toUtf8().toPercentEncoding() + "&");

    QByteArray key(params.value("oauth_consumer_secret").toUtf8() + "&" + params.value("oauth_token_secret").toUtf8());
    QByteArray signature(OAuth::createSignature(key, baseString.toUtf8()));
    QByteArray header("OAuth ");

    params.insert("oauth_signature", QString(signature));
    params.remove("oauth_consumer_secret");
    params.remove("oauth_token_secret");

    foreach (QString key, params.keys()) {
        if (!key.startsWith("oauth_")) {
            params.remove(key);
        }
    }

    QMapIterator<QString, QString> headerIterator(params);
    headerIterator.toFront();

    while (headerIterator.hasNext()) {
        headerIterator.next();

        if (headerIterator.hasNext()) {
            header.append(headerIterator.key().toUtf8() + "=\"" + headerIterator.value().toUtf8().toPercentEncoding() + "\", ");
        }
        else {
            header.append(headerIterator.key().toUtf8() + "=\"" + headerIterator.value().toUtf8().toPercentEncoding() + "\"");
        }
    }

    return header;
}

}
