#include "commentlistmodel.h"
#include "feedurls.h"
#include "json.h"
#include "soundcloud.h"
#include "utils.h"
#include "notifications.h"
#ifndef QML_USER_INTERFACE
#include "thumbnailcache.h"
#endif
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QTimer>

using namespace QtJson;

CommentListModel::CommentListModel(QObject *parent) :
    QAbstractListModel(parent),
    #ifndef QML_USER_INTERFACE
    m_cache(new ThumbnailCache),
    m_avatarPressedRow(-1),
    #endif
    m_loading(true),
    m_moreResults(true),
    m_error(false),
    m_offset(0)
{
#if QT_VERSION >= 0x040600
    QHash<int, QByteArray> roles;
    roles[IdRole] = "id";
    roles[BodyRole] = "body";
    roles[ArtistRole] = "artist";
    roles[ArtistIdRole] = "artistId";
    roles[ArtistAvatarUrlRole] = "artistAvatarUrl";
    roles[DateRole] = "date";
    roles[TrackIdRole] = "trackId";
#ifndef QML_USER_INTERFACE
    roles[ArtistAvatarRole] = "artistAvatar";
    roles[AvatarPressedRole] = "avatarPressed";
#endif
    this->setRoleNames(roles);
#endif
#ifndef QML_USER_INTERFACE
    this->connect(m_cache, SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
#endif
    this->connect(SoundCloud::instance(), SIGNAL(commentAdded(QSharedPointer<CommentItem>)), this, SLOT(onCommentAdded(QSharedPointer<CommentItem>)));
}

CommentListModel::~CommentListModel() {
    m_list.clear();
#ifndef QML_USER_INTERFACE
    delete m_cache;
    m_cache = 0;
#endif
}

void CommentListModel::clear() {
    this->beginRemoveRows(QModelIndex(), 0, this->rowCount() - 1);
    m_list.clear();
    this->endRemoveRows();
    this->setLoading(false);
    this->setOffset(1);
    this->setMoreResults(true);
    this->setError(false);
}

void CommentListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getMoreComments();
    }
}

int CommentListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return m_list.size();
}

QVariant CommentListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case IdRole:
        return m_list.at(index.row()).data()->id();
    case BodyRole:
        return m_list.at(index.row()).data()->body();
    case ArtistRole:
        return m_list.at(index.row()).data()->artist();
    case ArtistIdRole:
        return m_list.at(index.row()).data()->artistId();
#ifndef QML_USER_INTERFACE
    case ArtistAvatarRole:
        return m_cache->thumbnail(m_list.at(index.row()).data()->artistAvatarUrl(), QSize(48, 48));
    case AvatarPressedRole:
        return m_avatarPressedRow == index.row();
#endif
    case ArtistAvatarUrlRole:
        return m_list.at(index.row()).data()->artistAvatarUrl();
    case DateRole:
        return m_list.at(index.row()).data()->date();
    case TrackIdRole:
        return m_list.at(index.row()).data()->trackId();
    default:
        return QVariant();
    }
}

#if QT_VERSION >= 0x040600
QVariant CommentListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}
#endif

#ifndef QML_USER_INTERFACE
bool CommentListModel::setData(const QModelIndex &index, const QVariant &value, int role) {
    switch (role) {
    case AvatarPressedRole:
        m_avatarPressedRow = value.toInt();
        break;
    default:
        return false;
    }

    emit dataChanged(index, index);
    return true;
}
#endif

QSharedPointer<CommentItem> CommentListModel::get(int row) const {
    if ((row >= 0) && (row < m_list.size())) {
        return QSharedPointer<CommentItem>(m_list.at(row));
    }

    return QSharedPointer<CommentItem>();
}

#ifdef QML_USER_INTERFACE
CommentItem* CommentListModel::getFromQML(int row) const {

    if ((row >= 0) && (row < m_list.size())) {
        return m_list.at(row).data();
    }

    return 0;
}
#endif

void CommentListModel::insertComment(int row, QSharedPointer<CommentItem> comment) {
    Q_ASSERT((row >= 0) && (row <= this->rowCount()));

    this->beginInsertRows(QModelIndex(), row, row);
    m_list.insert(row, comment);
    this->endInsertRows();
}

void CommentListModel::appendComment(QSharedPointer<CommentItem> comment) {
    this->beginInsertRows(QModelIndex(), this->rowCount(), this->rowCount());
    m_list.append(comment);
    this->endInsertRows();
}

bool CommentListModel::removeComment(const QVariant &value, int role) {
    QModelIndexList indexes = this->match(this->index(0), role, value, 1, Qt::MatchExactly);

    if (indexes.isEmpty()) {
        return false;
    }

    return this->removeRow(indexes.first().row());
}

bool CommentListModel::removeRow(int row, const QModelIndex &parent)
{
    Q_UNUSED(parent)

    if ((row >= 0) && (row < m_list.size())) {
        this->beginRemoveRows(QModelIndex(), row, row);
        m_list.takeAt(row).clear();
        this->endRemoveRows();

        return true;
    }

    return false;
}

#ifndef QML_USER_INTERFACE
void CommentListModel::onThumbnailReady() {
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
}
#endif

void CommentListModel::setTrackId(const QString &id) {
    if (id != this->trackId()) {
        m_trackId = id;
        this->setFeed(QUrl(QString("http://api.soundcloud.com/tracks/%1/comments.json?limit=30").arg(id)));
    }
}

void CommentListModel::getComments(const QString &id) {
    if (!id.isEmpty()) {
        this->setTrackId(id);
    }

    this->setLoading(true);
    QNetworkReply *reply = SoundCloud::instance()->createReply(this->feed(), this->offset());
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addComments()));
}

void CommentListModel::addComments() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setLoading(false);
        this->setError(true);
        Notifications::instance()->onError(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    switch (statusCode) {
    case 200:
    case 201: {
        QString response(reply->readAll());
        bool ok;
        QVariantList entries = Json::parse(response, ok).toList();

        if (!ok) {
            this->setLoading(false);
            this->setError(true);
            Notifications::instance()->onError(tr("Cannot parse server response"));
        }
        else {
            for (int i = 0; i < entries.size(); i++) {
                QMetaObject::invokeMethod(this, "appendComment", Qt::QueuedConnection, Q_ARG(QSharedPointer<CommentItem>, QSharedPointer<CommentItem>(new CommentItem(entries.at(i).toMap()))));
            }

            this->setMoreResults(!entries.isEmpty());
            this->setOffset(this->offset() + entries.size());
            QTimer::singleShot(1000, this, SLOT(stopLoading()));
        }
    }
        break;
    default:
        this->setLoading(false);
        this->setError(true);
        Notifications::instance()->onError(Utils::httpErrorString(statusCode));
        break;
    }

    reply->deleteLater();
}

void CommentListModel::getMoreComments() {
    if ((this->moreResults()) && (!this->loading())) {
        this->getComments();
    }
}

void CommentListModel::onCommentAdded(QSharedPointer<CommentItem> comment) {
    if ((!comment.isNull()) && (comment.data()->trackId() == this->trackId())) {
        this->insertComment(0, comment);
        emit countChanged(this->rowCount());
    }
}
