#include "activitylistmodel.h"
#include "feedurls.h"
#include "json.h"
#include "soundcloud.h"
#include "notifications.h"
#include "utils.h"
#ifndef QML_USER_INTERFACE
#include "thumbnailcache.h"
#endif
#include <QNetworkReply>
#include <QTimer>

using namespace QtJson;

ActivityListModel::ActivityListModel(QObject * parent) :
    QAbstractListModel(parent),
    #ifndef QML_USER_INTERFACE
    m_cache(new ThumbnailCache),
    m_artistAvatarPressedRow(-1),
    m_thumbnailPressedRow(-1),
    #endif
    m_loading(true),
    m_moreResults(true),
    m_error(false),
    m_refresh(false)
{
#if QT_VERSION >= 0x040600
    QHash<int, QByteArray> roles;
    roles[IdRole] = "id";
    roles[TitleRole] = "title";
    roles[ArtistRole] = "artist";
    roles[ArtistIdRole] = "artistId";
    roles[ArtistAvatarUrlRole] = "artistAvatarUrl";
    roles[ThumbnailUrlRole] = "thumbnailUrl";
#ifndef QML_USER_INTERFACE
    roles[ArtistAvatarRole] = "artistAvatar";
    roles[ThumbnailRole] = "thumbnail";
    roles[ArtistAvatarPressedRole] = "artistAvatarPressed";
    roles[ThumbnailPressedRole] = "thumbnailPressed";
#endif
    roles[BodyRole] = "body";
    roles[DateRole] = "date";
    roles[ActivityTypeRole] = "activityType";
    this->setRoleNames(roles);
#endif
#ifndef QML_USER_INTERFACE
    this->connect(m_cache, SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
#endif
}

ActivityListModel::~ActivityListModel() {
    m_list.clear();
#ifndef QML_USER_INTERFACE
    delete m_cache;
    m_cache = 0;
#endif
}

void ActivityListModel::clear() {
    this->beginRemoveRows(QModelIndex(), 0, this->rowCount() - 1);
    m_list.clear();
    this->endRemoveRows();
    this->setLoading(false);
    this->setMoreResults(true);
    this->setError(false);
    this->setCursor(QString());
    this->setRefreshRequired(false);
}

void ActivityListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getActivities();
    }
}

int ActivityListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return m_list.size();
}

QVariant ActivityListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case IdRole:
        return m_list.at(index.row()).data()->id();
    case TitleRole:
        return m_list.at(index.row()).data()->title();
    case ArtistRole:
        return m_list.at(index.row()).data()->artist();
    case ArtistIdRole:
        return m_list.at(index.row()).data()->artistId();
    case ArtistAvatarUrlRole:
        return m_list.at(index.row()).data()->artistAvatarUrl();
    case ThumbnailUrlRole:
        return m_list.at(index.row()).data()->thumbnailUrl();
#ifndef QML_USER_INTERFACE
    case ArtistAvatarRole:
        return m_cache->thumbnail(m_list.at(index.row()).data()->artistAvatarUrl(), QSize(48, 48));
    case ThumbnailRole:
        return m_cache->thumbnail(m_list.at(index.row()).data()->thumbnailUrl(), QSize(96, 96));
    case ArtistAvatarPressedRole:
        return QVariant(m_artistAvatarPressedRow == index.row());
    case ThumbnailPressedRole:
        return QVariant(m_thumbnailPressedRow == index.row());
#endif
    case BodyRole:
        return m_list.at(index.row()).data()->body();
    case DateRole:
        return m_list.at(index.row()).data()->date();
    case ActivityTypeRole:
        return m_list.at(index.row()).data()->activityType();
    default:
        return QVariant();
    }
}

#if QT_VERSION >= 0x040600
QVariant ActivityListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}
#endif

#ifndef QML_USER_INTERFACE
bool ActivityListModel::setData(const QModelIndex &index, const QVariant &value, int role) {
    switch (role) {
    case ArtistAvatarPressedRole:
        m_artistAvatarPressedRow = value.toInt();
        break;
    case ThumbnailPressedRole:
        m_thumbnailPressedRow = value.toInt();
        break;
    default:
        return false;
    }

    emit dataChanged(index, index);
    return true;
}
#endif

QSharedPointer<ActivityItem> ActivityListModel::get(int row) const {
    if ((row >= 0) && (row < m_list.size())) {
        return QSharedPointer<ActivityItem>(m_list.at(row));
    }

    return QSharedPointer<ActivityItem>();
}

#ifdef QML_USER_INTERFACE
ActivityItem* ActivityListModel::getFromQML(int row) const {
    if ((row >= 0) && (row < m_list.size())) {
        return m_list.at(row).data();
    }

    return 0;
}
#endif

void ActivityListModel::insertActivity(int row, QSharedPointer<ActivityItem> activity) {
    Q_ASSERT((row >= 0) && (row <= this->rowCount()));

    this->beginInsertRows(QModelIndex(), row, row);
    m_list.insert(row, activity);
    this->endInsertRows();
}

void ActivityListModel::appendActivity(QSharedPointer<ActivityItem> activity) {
    this->beginInsertRows(QModelIndex(), this->rowCount(), this->rowCount());
    m_list.append(activity);
    this->endInsertRows();
}

bool ActivityListModel::removeActivity(const QVariant &value, int role) {
    QModelIndexList indexes = this->match(this->index(0), role, value, 1, Qt::MatchExactly);

    if (indexes.isEmpty()) {
        return false;
    }

    return this->removeRow(indexes.first().row());
}

bool ActivityListModel::removeRow(int row, const QModelIndex &parent)
{
    Q_UNUSED(parent)

    if ((row >= 0) && (row < m_list.size())) {
        this->beginRemoveRows(QModelIndex(), row, row);
        m_list.takeAt(row).clear();
        this->endRemoveRows();

        return true;
    }

    return false;
}

#ifndef QML_USER_INTERFACE
void ActivityListModel::onThumbnailReady() {
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
}
#endif

void ActivityListModel::setFeed(const QUrl &feed) {
    if (feed != this->feed()) {
        m_feed = feed;
        emit feedChanged(feed);
    }
}

void ActivityListModel::getMoreActivities() {
    if ((this->moreResults()) && (!this->loading())) {
        this->getActivities();
    }
}

void ActivityListModel::getActivities(QUrl feed) {
    if (!feed.isEmpty()) {
        this->setFeed(feed);
    }
    else {
        feed = this->feed();
    }

    if (!this->cursor().isEmpty()) {
        feed.addQueryItem("cursor", this->cursor());
    }

    this->setLoading(true);
    QNetworkReply *reply = SoundCloud::instance()->createReply(feed);
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addActivities()));
}

void ActivityListModel::addActivities() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setLoading(false);
        this->setError(true);
        Notifications::instance()->onError(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    switch (statusCode) {
    case 200:
    case 201: {
        QString response(reply->readAll());
        bool ok;
        QVariantMap res = Json::parse(response, ok).toMap();

        if (!ok) {
            this->setLoading(false);
            this->setError(true);
            Notifications::instance()->onError(tr("Cannot parse server response"));
        }
        else {
            QVariantList entries = res.value("collection").toList();

            for (int i = 0; i < entries.size(); i++) {
                QMetaObject::invokeMethod(this, "appendActivity", Qt::QueuedConnection, Q_ARG(QSharedPointer<ActivityItem>, QSharedPointer<ActivityItem>(new ActivityItem(entries.at(i).toMap()))));
            }

            QUrl next = res.value("next_href").toString();

            if (next.hasQueryItem("cursor")) {
                this->setCursor(next.queryItemValue("cursor"));
            }

            this->setMoreResults(!entries.isEmpty());
            QTimer::singleShot(1000, this, SLOT(stopLoading()));
        }
    }
        break;
    default:
        this->setLoading(false);
        this->setError(true);
        Notifications::instance()->onError(Utils::httpErrorString(statusCode));
        break;
    }

    reply->deleteLater();
}
