#include "nowplayingmediatab.h"
#include "thumbnaillabel.h"
#include "mediaplayerbutton.h"
#include "nowplayingdelegate.h"
#include "screen.h"
#include "../base/playbacklistmodel.h"
#include "../base/thumbnailcache.h"
#include "../base/mediaplayer.h"
#include "../base/utils.h"
#include <QListView>
#include <QMenu>
#include <QSlider>
#include <QLabel>
#include <QGridLayout>
#include <QShowEvent>
#include <QHideEvent>
#include <QMaemo5Style>

NowPlayingMediaTab::NowPlayingMediaTab(QWidget *parent) :
    AbstractTab(parent),
    m_thumbnailLabel(new ThumbnailLabel(this)),
    m_view(new QListView(this)),
    m_progressSlider(new QSlider(Qt::Horizontal, this)),
    m_trackNumberLabel(new QLabel(this)),
    m_titleLabel(new QLabel(this)),
    m_artistLabel(new QLabel(this)),
    m_positionLabel(new QLabel(this)),
    m_durationLabel(new QLabel(this)),
    m_contextMenu(new QMenu(this)),
    m_deleteAction(m_contextMenu->addAction(tr("Delete from now playing"), this, SLOT(deleteTrackFromQueue()))),
    m_clearAction(m_contextMenu->addAction(tr("Clear now playing"), MediaPlayer::instance(), SLOT(clearTracks()))),
    m_toolBar(new QWidget(this)),
    m_container(new QWidget(this)),
    m_playPauseButton(new MediaPlayerButton(MediaPlayerButton::PlayPause, this)),
    m_previousButton(new MediaPlayerButton(MediaPlayerButton::Previous, this)),
    m_nextButton(new MediaPlayerButton(MediaPlayerButton::Next, this)),
    m_repeatButton(new MediaPlayerButton(MediaPlayerButton::Repeat, this))
{
    this->addAction(m_clearAction);

    m_view->setModel(PlaybackListModel::playbackQueue());
    m_view->setItemDelegate(new NowPlayingDelegate(m_view));
    m_view->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_view->setContextMenuPolicy(Qt::CustomContextMenu);
    m_view->hide();

    QFont font;
    font.setPixelSize(20);
    m_trackNumberLabel->setFont(font);
    m_positionLabel->setFont(font);
    m_durationLabel->setFont(font);
    m_positionLabel->setFixedWidth(70);
    m_durationLabel->setFixedWidth(70);
    m_positionLabel->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
    m_durationLabel->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
    m_thumbnailLabel->setFixedSize(295, 295);
    m_toolBar->setFixedHeight(70);
    m_progressSlider->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);

    QHBoxLayout *hbox = new QHBoxLayout(m_toolBar);
    hbox->setContentsMargins(0, 0, 0, 0);
    hbox->setSpacing(67);
    hbox->addWidget(m_previousButton);
    hbox->addWidget(m_playPauseButton);
    hbox->addWidget(m_nextButton);
    hbox->addStretch(1);
    hbox->addWidget(m_repeatButton);

    QGridLayout *grid = new QGridLayout(m_container);
    grid->addWidget(m_trackNumberLabel, 0, 0, 1, 3);
    grid->addWidget(m_titleLabel, 1, 0, 1, 3);
    grid->addWidget(m_positionLabel, 2, 0);
    grid->addWidget(m_progressSlider, 2, 1);
    grid->addWidget(m_durationLabel, 2, 2);
    grid->addWidget(m_artistLabel, 3, 0, 1, 3);

    this->connect(m_view, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(m_view, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(m_thumbnailLabel, SIGNAL(clicked()), this, SLOT(onThumbnailClicked()));
    this->connect(m_progressSlider, SIGNAL(sliderReleased()), this, SLOT(onSliderReleased()));
    this->connect(PlaybackListModel::playbackQueue(), SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(Screen::instance(), SIGNAL(screenLockStateChanged(bool)), this, SLOT(onScreenLockStateChanged(bool)));

    this->onOrientationChanged();
}

void NowPlayingMediaTab::showEvent(QShowEvent *event) {
    this->connectPlaybackSignals();
    event->accept();
}

void NowPlayingMediaTab::hideEvent(QHideEvent *event) {
    this->disconnectPlaybackSignals();
    event->accept();
}

void NowPlayingMediaTab::setLandscapeLayout() {
    m_thumbnailLabel->move(30, 30);
    m_container->setGeometry(345, 0, 445, 354);
    m_view->setGeometry(345, 0, 445, 354);
    m_toolBar->setGeometry(15, 354, 770, 70);
}

void NowPlayingMediaTab::setPortraitLayout() {
    m_thumbnailLabel->move(92, 30);
    m_container->setGeometry(10, 335, 460, 354);
    m_view->setGeometry(10, 335, 460, 354);
    m_toolBar->setGeometry(0, 674, 480, 70);
}

void NowPlayingMediaTab::connectPlaybackSignals() {
    this->connect(MediaPlayer::instance(), SIGNAL(positionChanged(qint64)), this, SLOT(onPositionChanged(qint64)));
    this->connect(MediaPlayer::instance(), SIGNAL(durationChanged(qint64)), this, SLOT(onDurationChanged(qint64)));
    this->connect(MediaPlayer::instance(), SIGNAL(currentIndexChanged(int)), this, SLOT(onCurrentIndexChanged(int)));
    this->connect(MediaPlayer::instance(), SIGNAL(currentTrackChanged(QSharedPointer<TrackItem>)), this, SLOT(onCurrentTrackChanged(QSharedPointer<TrackItem>)));

    this->onPositionChanged(MediaPlayer::instance()->position());
    this->onDurationChanged(MediaPlayer::instance()->duration());
    this->onCurrentIndexChanged(MediaPlayer::instance()->currentIndex());
    this->onCurrentTrackChanged(MediaPlayer::instance()->currentTrack());
}

void NowPlayingMediaTab::disconnectPlaybackSignals() {
    this->disconnect(MediaPlayer::instance(), SIGNAL(positionChanged(qint64)), this, SLOT(onPositionChanged(qint64)));
    this->disconnect(MediaPlayer::instance(), SIGNAL(durationChanged(qint64)), this, SLOT(onDurationChanged(qint64)));
    this->disconnect(MediaPlayer::instance(), SIGNAL(currentIndexChanged(int)), this, SLOT(onCurrentIndexChanged(int)));
    this->disconnect(MediaPlayer::instance(), SIGNAL(currentTrackChanged(QSharedPointer<TrackItem>)), this, SLOT(onCurrentTrackChanged(QSharedPointer<TrackItem>)));
}

void NowPlayingMediaTab::setThumbnail() {
    if (MediaPlayer::instance()->currentTrack().isNull()) {
        return;
    }

    QImage thumbnail = ThumbnailCache::instance()->thumbnail(MediaPlayer::instance()->currentTrack().data()->largeThumbnailUrl(), m_thumbnailLabel->size());

    if (thumbnail.isNull()) {
        m_thumbnailLabel->setImage(QImage("/usr/share/icons/hicolor/295x295/hildon/mediaplayer_default_album.png"));
        this->connect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnail()));
    }
    else {
        m_thumbnailLabel->setImage(thumbnail);
        this->disconnect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnail()));
    }
}

void NowPlayingMediaTab::onThumbnailClicked() {
    if (m_view->isVisible()) {
        m_container->show();
        m_view->hide();
    }
    else {
        m_view->setCurrentIndex(PlaybackListModel::playbackQueue()->index(MediaPlayer::instance()->currentIndex()));
        m_view->scrollTo(m_view->currentIndex(), QListView::PositionAtCenter);
        m_view->show();
        m_container->hide();
    }
}

void NowPlayingMediaTab::onSliderReleased() {
    MediaPlayer::instance()->setPosition(m_progressSlider->value());
}

void NowPlayingMediaTab::onPositionChanged(qint64 position) {
    m_positionLabel->setText(Utils::durationFromMSecs(position));

    if (!m_progressSlider->isSliderDown()) {
        m_progressSlider->setValue(position);
    }
}

void NowPlayingMediaTab::onDurationChanged(qint64 duration) {
    m_progressSlider->setMaximum(duration);
}

void NowPlayingMediaTab::onCountChanged(int count) {
    m_trackNumberLabel->setText(QString("<font color='%1'>%2/%3 tracks</font>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(MediaPlayer::instance()->currentIndex() + 1).arg(count));
}

void NowPlayingMediaTab::onCurrentIndexChanged(int index) {
    if (m_view->isVisible()) {
        m_view->setCurrentIndex(PlaybackListModel::playbackQueue()->index(index));
        m_view->scrollTo(m_view->currentIndex(), QListView::PositionAtCenter);
    }

    m_trackNumberLabel->setText(QString("<font color='%1'>%2/%3 tracks</font>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(index + 1).arg(PlaybackListModel::playbackQueue()->rowCount()));
}

void NowPlayingMediaTab::onCurrentTrackChanged(QSharedPointer<TrackItem> track) {
    if (!track.isNull()) {
        m_titleLabel->setText(track.data()->title());
        m_artistLabel->setText(track.data()->artist());
        m_durationLabel->setText(Utils::durationFromMSecs(track.data()->duration()));
        this->setThumbnail();
    }
}

void NowPlayingMediaTab::onScreenLockStateChanged(bool locked) {
    if (locked) {
        this->disconnectPlaybackSignals();
    }
    else if (this->isVisible()) {
        this->connectPlaybackSignals();
    }
}

void NowPlayingMediaTab::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_deleteAction);
}

void NowPlayingMediaTab::onItemClicked(const QModelIndex &index) {
    MediaPlayer::instance()->setCurrentIndex(index.row());
    this->onThumbnailClicked();
}

void NowPlayingMediaTab::deleteTrackFromQueue() {
    PlaybackListModel::playbackQueue()->removeTrack(m_view->currentIndex().row());
}
