#include "nowplayinginfotab.h"
#include "separatorlabel.h"
#include "thumbnaillabel.h"
#include "waveformlabel.h"
#include "listview.h"
#include "commentdelegate.h"
#include "commentdialog.h"
#include "sharingdialog.h"
#include "channelwindow.h"
#include "scrollarea.h"
#include "screen.h"
#include "../base/commentlistmodel.h"
#include "../base/soundcloud.h"
#include "../base/transfermanager.h"
#include "../base/utils.h"
#include "../base/thumbnailcache.h"
#include "../base/settings.h"
#include "../base/playbacklistmodel.h"
#include "../base/mediaplayer.h"
#include <QGridLayout>
#include <QLabel>
#include <QAction>

NowPlayingInfoTab::NowPlayingInfoTab(QWidget *parent) :
    AbstractTab(parent),
    m_commentAction(new QAction(tr("Add comment"), this)),
    m_favouritesAction(new QAction(tr("Add to favourites"), this)),
    m_downloadAction(new QAction(tr("Download"), this)),
    m_shareAction(new QAction(tr("Share via connections"), this)),
    m_titleLabel(new QLabel(this)),
    m_artistLabel(new QLabel(this)),
    m_dateLabel(new QLabel(this)),
    m_genreLabel(new QLabel(this)),
    m_formatLabel(new QLabel(this)),
    m_sizeLabel(new QLabel(this)),
    m_lengthLabel(new QLabel(this)),
    m_playCountLabel(new QLabel(this)),
    m_downloadCountLabel(new QLabel(this)),
    m_descriptionLabel(new QLabel(this)),
    m_thumbnailLabel(new ThumbnailLabel(this)),
    m_waveformLabel(new WaveformLabel(this)),
    m_propertiesSeparator(new SeparatorLabel(tr("Properties"), this)),
    m_descriptionSeparator(new SeparatorLabel(tr("Description"), this)),
    m_commentsSeparator(new SeparatorLabel(tr("Comments"), this)),
    m_commentsModel(new CommentListModel(this)),
    m_commentsView(new ListView(this)),
    m_scrollArea(new ScrollArea(this))
{
    this->addAction(m_commentAction);
    this->addAction(m_favouritesAction);
    this->addAction(m_downloadAction);
    this->addAction(m_shareAction);

    CommentDelegate *delegate = new CommentDelegate(m_commentsView, false);

    m_commentsView->setModel(m_commentsModel);
    m_commentsView->setItemDelegate(delegate);
    m_thumbnailLabel->setFixedSize(124, 124);
    m_thumbnailLabel->setInteractive(false);
    m_waveformLabel->setFixedHeight(124);
    m_waveformLabel->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    m_waveformLabel->setInteractive(false);
    m_titleLabel->setWordWrap(true);
    m_descriptionLabel->setWordWrap(true);

    QVBoxLayout *vbox = new QVBoxLayout(this);
    vbox->setContentsMargins(0, 0, 0, 0);
    vbox->addWidget(m_scrollArea);

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);

    grid->setContentsMargins(0, 0, 0, 0);
    grid->addItem(new QSpacerItem(10, 10, QSizePolicy::Maximum), 0, 0);
    grid->addWidget(m_titleLabel, 0, 1, 1, 2, Qt::AlignTop);
    grid->addWidget(m_thumbnailLabel, 1, 1, 2, 1, Qt::AlignTop);
    grid->addWidget(m_artistLabel, 1, 2, Qt::AlignTop);
    grid->addWidget(m_dateLabel, 2, 2, Qt::AlignTop);
    grid->addWidget(m_propertiesSeparator, 3, 1, 1, 2);
    grid->addWidget(m_genreLabel, 4, 1, 1, 2);
    grid->addWidget(m_formatLabel, 5, 1, 1, 2);
    grid->addWidget(m_sizeLabel, 6, 1, 1, 2);
    grid->addWidget(m_lengthLabel, 7, 1, 1, 2);
    grid->addWidget(m_playCountLabel, 8, 1, 1, 2);
    grid->addWidget(m_downloadCountLabel, 9, 1, 1, 2);
    grid->addWidget(m_waveformLabel, 10, 1, 1, 2);
    grid->addWidget(m_descriptionSeparator, 11, 1, 1, 2);
    grid->addWidget(m_descriptionLabel, 12, 1, 1, 2);
    grid->addWidget(m_commentsSeparator, 13, 1, 1, 2);
    grid->addWidget(m_commentsView, 14, 1, 1, 2);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    this->connect(m_commentAction, SIGNAL(triggered()), this, SLOT(showCommentDialog()));
    this->connect(m_favouritesAction, SIGNAL(triggered()), this, SLOT(toggleFavourite()));
    this->connect(m_downloadAction, SIGNAL(triggered()), this, SLOT(downloadTrack()));
    this->connect(m_shareAction, SIGNAL(triggered()), this, SLOT(showShareDialog()));
    this->connect(m_commentsModel, SIGNAL(countChanged(int)), this, SLOT(onCommentsCountChanged(int)));
    this->connect(Screen::instance(), SIGNAL(screenLockStateChanged(bool)), this, SLOT(onScreenLockStateChanged(bool)));

    this->onOrientationChanged();
}

void NowPlayingInfoTab::showEvent(QShowEvent *event) {
    this->connectPlaybackSignals();
    event->accept();
}

void NowPlayingInfoTab::hideEvent(QHideEvent *event) {
    this->disconnectPlaybackSignals();
    event->accept();
}

void NowPlayingInfoTab::setLandscapeLayout() {
    m_scrollArea->widget()->setMaximumWidth(800);
}

void NowPlayingInfoTab::setPortraitLayout() {
    m_scrollArea->widget()->setMaximumWidth(480);
}

void NowPlayingInfoTab::connectPlaybackSignals() {
    this->connect(MediaPlayer::instance(), SIGNAL(currentTrackChanged(QSharedPointer<TrackItem>)), this, SLOT(onCurrentTrackChanged(QSharedPointer<TrackItem>)));
    this->connect(MediaPlayer::instance(), SIGNAL(positionChanged(qint64)), m_waveformLabel, SLOT(update()));

    this->onCurrentTrackChanged(MediaPlayer::instance()->currentTrack());
    m_waveformLabel->update();
}

void NowPlayingInfoTab::disconnectPlaybackSignals() {
    this->disconnect(MediaPlayer::instance(), SIGNAL(currentTrackChanged(QSharedPointer<TrackItem>)), this, SLOT(onCurrentTrackChanged(QSharedPointer<TrackItem>)));
    this->disconnect(MediaPlayer::instance(), SIGNAL(positionChanged(qint64)), m_waveformLabel, SLOT(update()));
}

void NowPlayingInfoTab::setThumbnails() {
    if (MediaPlayer::instance()->currentTrack().isNull()) {
        return;
    }

    QImage thumbnail = ThumbnailCache::instance()->thumbnail(MediaPlayer::instance()->currentTrack().data()->thumbnailUrl(), QSize(120, 120));
    QImage waveform = ThumbnailCache::instance()->thumbnail(MediaPlayer::instance()->currentTrack().data()->waveformUrl());

    if ((thumbnail.isNull()) || (waveform.isNull())) {
        this->connect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }
    else {
        this->disconnect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }

    if (thumbnail.isNull()) {
        thumbnail = QImage("/usr/share/icons/hicolor/124x124/hildon/mediaplayer_default_album.png");
    }

    m_thumbnailLabel->setImage(thumbnail);

    if (!waveform.isNull()) {
        m_waveformLabel->setImage(waveform);
    }
}

void NowPlayingInfoTab::setMenuActions() {
    if (MediaPlayer::instance()->currentTrack().isNull()) {
        return;
    }

    switch (MediaPlayer::instance()->currentTrack().data()->service()) {
    case Services::NoService:
        m_commentAction->setEnabled((SoundCloud::instance()->userSignedIn()) && (Settings::instance()->archiveOnline()));
        m_favouritesAction->setEnabled((SoundCloud::instance()->userSignedIn()) && (Settings::instance()->archiveOnline()));
        m_shareAction->setEnabled((SoundCloud::instance()->userSignedIn()) && (Settings::instance()->archiveOnline()));
        m_downloadAction->setEnabled(false);
        return;
    default:
        m_commentAction->setEnabled(SoundCloud::instance()->userSignedIn());
        m_favouritesAction->setEnabled(SoundCloud::instance()->userSignedIn());
        m_shareAction->setEnabled(SoundCloud::instance()->userSignedIn());
        return;
    }
}

void NowPlayingInfoTab::downloadTrack() {
    TransferManager::instance()->addDownloadTransfer(MediaPlayer::instance()->currentTrack());
}

void NowPlayingInfoTab::toggleFavourite() {
    if (MediaPlayer::instance()->currentTrack().data()->favourite()) {
        SoundCloud::instance()->deleteFromFavourites(QStringList(MediaPlayer::instance()->currentTrack().data()->id()));
    }
    else {
        SoundCloud::instance()->addToFavourites(QStringList(MediaPlayer::instance()->currentTrack().data()->id()));
    }
}

void NowPlayingInfoTab::showCommentDialog() {
    CommentDialog *dialog = new CommentDialog(MediaPlayer::instance()->currentTrack(), this);
    dialog->open();
}

void NowPlayingInfoTab::showShareDialog() {
    SharingDialog *dialog = new SharingDialog(MediaPlayer::instance()->currentTrack(), this);
    dialog->open();
}

void NowPlayingInfoTab::onFavouriteChanged(bool favourite) {
    m_favouritesAction->setText(favourite ? tr("Delete from favourites") : tr("Add to favourites"));
}

void NowPlayingInfoTab::onCommentsCountChanged(int count) {
    m_commentsSeparator->setText(!count ? tr("No comments") : tr("Comments"));
    m_commentsView->setFixedHeight(m_commentsModel->rowCount() * m_commentsView->sizeHintForRow(0));
}

void NowPlayingInfoTab::onCurrentTrackChanged(QSharedPointer<TrackItem> track) {
    if (!track.isNull()) {
        m_titleLabel->setText(QString("<p style='font-size:32px;'>%1</p>").arg(track.data()->title()));
        m_artistLabel->setText(QString("%1: %2").arg(tr("Artist")).arg(track.data()->artist()));
        m_dateLabel->setText(QString("%1: %2").arg(tr("Date")).arg(track.data()->date()));
        m_genreLabel->setText(QString("%1: %2").arg(tr("Genre")).arg(track.data()->genre().isEmpty() ? tr("Unknown") : track.data()->genre()));
        m_formatLabel->setText(QString("%1: %2").arg(tr("Format")).arg(track.data()->format()));
        m_sizeLabel->setText(QString("%1: %2").arg(tr("Size")).arg(Utils::fileSizeFromBytes(track.data()->size())));
        m_lengthLabel->setText(QString("%1: %2").arg(tr("Length")).arg(Utils::durationFromMSecs(track.data()->duration())));
        m_playCountLabel->setText(QString("%1: %2").arg(tr("Play count")).arg(track.data()->playCount()));
        m_downloadCountLabel->setText(QString("%1: %2").arg(tr("Download count")).arg(track.data()->downloadCount()));
        m_descriptionLabel->setText(track.data()->description().isEmpty() ? tr("No description") : track.data()->description());

        switch (track.data()->service()) {
        case Services::NoService:
            if (Settings::instance()->archiveOnline()) {
                m_commentsModel->getComments(track.data()->id());
            }
            else {
                m_commentsModel->clear();
                m_commentsSeparator->setText(tr("No comments"));
            }
            break;
        default:
            m_commentsModel->getComments(track.data()->id());
            break;
        }

        this->setMenuActions();
        this->setThumbnails();
        this->onFavouriteChanged(track.data()->favourite());

        this->connect(track.data(), SIGNAL(favouriteChanged(bool)), this, SLOT(onFavouriteChanged(bool)));
    }
}

void NowPlayingInfoTab::onScreenLockStateChanged(bool locked) {
    if (locked) {
        this->disconnectPlaybackSignals();
    }
    else if (this->isVisible()) {
        this->connectPlaybackSignals();
    }
}
