#include "nowplayingbutton.h"
#include "nowplayingwindow.h"
#include "windowstack.h"
#include "screen.h"
#include "../base/mediaplayer.h"
#include "../base/playbacklistmodel.h"
#include <QPaintEvent>
#include <QTimer>
#include <QPainter>
#include <QApplication>
#include <QDesktopWidget>

NowPlayingButton* NowPlayingButton::m_instance = 0;

NowPlayingButton::NowPlayingButton(StackedWindow *parent) :
    QWidget(parent),
    m_animationTimer(new QTimer(this)),
    m_currentFrame(0),
    m_ready(false),
    m_moved(false)
{
    if (!m_instance) {
        m_instance = this;
    }

    this->setFixedSize(112, 70);
    this->setAttribute(Qt::WA_OpaquePaintEvent);
    this->setAttribute(Qt::WA_Maemo5NonComposited);
    this->hide();

    m_animationTimer->setInterval(100);
    m_images.reserve(13);
    m_images.append(QPixmap("/usr/share/icons/hicolor/scalable/hildon/mediaplayer_nowplaying_indicator_pause.png"));

    for (int i = 1; i <= 12; i++) {
        m_images.append(QPixmap(QString("/usr/share/icons/hicolor/scalable/hildon/mediaplayer_nowplaying_indicator%1.png").arg(i)));
    }

    this->connect(m_animationTimer, SIGNAL(timeout()), this, SLOT(nextFrame()));
    this->connect(MediaPlayer::instance(), SIGNAL(stateChanged(QMediaPlayer::State)), this, SLOT(onMediaStateChanged(QMediaPlayer::State)));
    this->connect(PlaybackListModel::playbackQueue(), SIGNAL(countChanged(int)), this, SLOT(onPlaybackQueueChanged(int)));
    this->connect(WindowStack::instance(), SIGNAL(currentWindowChanged(StackedWindow*)), this, SLOT(onCurrentWindowChanged(StackedWindow*)));
    this->connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(onScreenOrientationChanged()));
    this->connect(Screen::instance(), SIGNAL(screenLockStateChanged(bool)), this, SLOT(onScreenLockStateChanged(bool)));

    this->onPlaybackQueueChanged(PlaybackListModel::playbackQueue()->rowCount());
}

NowPlayingButton::~NowPlayingButton() {}

NowPlayingButton* NowPlayingButton::instance() {
    return !m_instance ? new NowPlayingButton : m_instance;
}

void NowPlayingButton::showEvent(QShowEvent *event) {
    if (!this->moved()) {
        QRect screen = QApplication::desktop()->screenGeometry();
        this->move(screen.width() - (this->width() + 8), screen.height() - (this->height() + 56));
        this->setMoved(true);
    }

    this->triggerAnimation();
    event->accept();
}

void NowPlayingButton::hideEvent(QHideEvent *event) {
    this->stopAnimation();
    event->accept();
}

bool NowPlayingButton::ready() const {
    return m_ready;
}

void NowPlayingButton::setReady(bool ready) {
    if (ready != this->ready()) {
        m_ready = ready;
        emit readyChanged(ready);
    }
}

bool NowPlayingButton::moved() const {
    return m_moved;
}

void NowPlayingButton::setMoved(bool moved) {
    m_moved = moved;
}

void NowPlayingButton::poke() {
    if (this->ready()) {
        this->show();
        this->raise();
    }
}

void NowPlayingButton::paintEvent(QPaintEvent *event) {
    QPainter painter(this);
    painter.drawPixmap(0, 0, m_images.at(m_currentFrame));
    event->accept();
}

void NowPlayingButton::mousePressEvent(QMouseEvent *event) {
    event->accept();
}

void NowPlayingButton::mouseReleaseEvent(QMouseEvent *event) {
    if (this->rect().contains(event->pos())) {
        this->showNowPlayingWindow();
    }

    event->accept();
}

void NowPlayingButton::onMediaStateChanged(QMediaPlayer::State state) {
    switch (state) {
    case QMediaPlayer::PlayingState:
        this->triggerAnimation();
        return;
    default:
        this->stopAnimation();
        return;
    }
}

void NowPlayingButton::onPlaybackQueueChanged(int count) {
    this->setReady(count > 0);

    if ((this->ready()) && (WindowStack::instance()->currentWindow()->hasNowPlayingButton())) {
        this->poke();
    }
    else {
        this->hide();
    }
}

void NowPlayingButton::onCurrentWindowChanged(StackedWindow *window) {
    if ((window) && (window->hasNowPlayingButton())) {
        this->setParent(window);
        this->setMoved(false);
        this->poke();
    }
    else {
        this->hide();
    }
}

void NowPlayingButton::onScreenLockStateChanged(bool locked) {
    if (locked) {
        this->stopAnimation();
    }
    else {
        this->triggerAnimation();
    }
}

void NowPlayingButton::onScreenOrientationChanged() {
    QRect screen = QApplication::desktop()->screenGeometry();
    this->move(screen.width() - (this->width() + 8), screen.height() - (this->height() + 56));
    this->setMoved(true);
}

void NowPlayingButton::nextFrame() {
    switch (m_currentFrame) {
    case 12:
        m_currentFrame = 1;
        break;
    default:
        m_currentFrame++;
        break;
    }

    this->update();
}

void NowPlayingButton::stopAnimation() {
    m_animationTimer->stop();
    m_currentFrame = 0;
    this->update();
}

void NowPlayingButton::triggerAnimation() {
    if ((this->isVisible()) && (MediaPlayer::instance()->state() == QMediaPlayer::PlayingState) && (!m_animationTimer->isActive())) {
        m_animationTimer->start();
    }
}

void NowPlayingButton::showNowPlayingWindow() {
    NowPlayingWindow *window = new NowPlayingWindow(WindowStack::instance()->currentWindow());
    window->show();
}
