#include "transferitem.h"
#include "../base/utils.h"
#include "../base/soundcloud.h"
#include "../base/settings.h"
#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QDir>
#include <QRegExp>
#include <QDebug>

const QRegExp illegalChars("[\"@&~=\\/:?#!|<>*^]");

TransferItem::TransferItem(QSharedPointer<TrackItem> track, Transfers::TransferStatus status, QObject *parent) :
    QObject(parent),
    m_nam(0),
    m_reply(0),
    m_track(track),
    m_id(m_track.data()->id()),
    m_title(m_track.data()->title()),
    m_service(m_track.data()->service()),
    m_status(status),
    m_priority(Transfers::NormalPriority),
    m_progress(0),
    m_speed(0.0),
    m_resumeSize(0),
    m_size(m_track.data()->size()),
    m_currentIndex(1),
    m_totalFiles(1),
    m_format(AudioFormats::Unknown),
    m_transferType(Transfers::TrackDownload),
    m_retries(0)
{
    m_progressTimer.setInterval(2000);
    m_progressTimer.setSingleShot(false);
    this->connect(&m_progressTimer, SIGNAL(timeout()), this, SLOT(updateProgress()));
}

TransferItem::TransferItem(QSharedPointer<PlaylistItem> playlist, QList< QSharedPointer<TrackItem> > tracks, Transfers::TransferStatus status, QObject *parent) :
    QObject(parent),
    m_nam(0),
    m_reply(0),
    m_playlist(playlist),
    m_playlistTracks(tracks),
    m_track(m_playlistTracks.takeFirst()),
    m_id(m_playlist.data()->id()),
    m_title(m_playlist.data()->title()),
    m_service(m_playlist.data()->service()),
    m_status(status),
    m_priority(Transfers::NormalPriority),
    m_progress(0),
    m_speed(0.0),
    m_resumeSize(0),
    m_size(m_track.data()->size()),
    m_currentIndex(m_track.data()->trackNumber()),
    m_totalFiles(m_playlist.data()->trackCount()),
    m_format(AudioFormats::Unknown),
    m_transferType(Transfers::PlaylistDownload),
    m_retries(0)
{
    m_progressTimer.setInterval(2000);
    m_progressTimer.setSingleShot(false);
    this->connect(&m_progressTimer, SIGNAL(timeout()), this, SLOT(updateProgress()));
}

TransferItem::~TransferItem() {}

void TransferItem::setPriority(Transfers::TransferPriority priority) {
    if (priority != this->priority()) {
        m_priority = priority;
        emit priorityChanged(priority);
    }
}

void TransferItem::setSize(qint64 size) {
    if (size != this->size()) {
        m_size = size;
        emit sizeChanged(size);
    }
}

void TransferItem::setCurrentIndex(int index) {
    if (index != this->currentIndex()) {
        m_currentIndex = index;
        emit currentIndexChanged(index);
    }
}

QString TransferItem::statusText() const {
    switch (this->status()) {
    case Transfers::Queued:
        return tr("Queued");
    case Transfers::Paused:
        return tr("Paused");
    case Transfers::Connecting:
        return tr("Connecting");
    case Transfers::Downloading:
        return tr("Downloading");
    case Transfers::Uploading:
        return tr("Uploading");
    case Transfers::Cancelled:
        return tr("Cancelled");
    case Transfers::Failed:
        return tr("Failed");
    case Transfers::Completed:
        return tr("Completed");
    default:
        return tr("Unknown");
    }
}

QString TransferItem::priorityText() const {
    switch (this->priority()) {
    case Transfers::HighPriority:
        return tr("High");
    case Transfers::NormalPriority:
        return tr("Normal");
    case Transfers::LowPriority:
        return tr("Low");
    default:
        return QString();
    }
}

void TransferItem::setStatus(Transfers::TransferStatus status) {
    if (status == this->status()) {
        return;
    }

    switch (status) {
    case Transfers::Paused:
        switch (this->status()) {
        case Transfers::Uploading:
            return;
        case Transfers::Downloading:
            m_status = status;
            this->pauseTransfer();
            return;
        default:
            break;
        }

        break;
    case Transfers::Cancelled:
        switch (this->status()) {
        case Transfers::Downloading:
            m_status = status;
            this->cancelTransfer();
            return;
        case Transfers::Uploading:
            m_status = status;
            this->cancelTransfer();
            return;
        default:
            break;
        }

        break;
    default:
        break;
    }

    m_status = status;
    emit statusChanged(status);
}

void TransferItem::setStatusInfo(const QString &info) {
    if (info != this->statusInfo()) {
        m_statusInfo = info;
        emit statusInfoChanged(info);
    }
}

void TransferItem::startTransfer(bool resetRetries) {
    if (resetRetries) {
        m_retries = 0;
    }

    this->setStatus(Transfers::Connecting);

    switch (Settings::instance()->downloadFormat()) {
    case AudioFormats::OriginalFormat:
        if (m_track.data()->downloadable()) {
            this->performDownload(SoundCloud::instance()->getDownloadUrl(m_track.data()->downloadUrl()), AudioFormats::OriginalFormat);
            return;
        }

        break;
    default:
        break;
    }

    this->performDownload(SoundCloud::instance()->getStreamUrl(m_track.data()->streamUrl()), AudioFormats::MP3);
}

void TransferItem::performDownload(const QUrl &url, AudioFormats::Format audioFormat) {
    switch (audioFormat) {
    case AudioFormats::Unknown:
        break;
    default:
        m_format = audioFormat;
        break;
    }

    QDir dir;
    dir.mkpath(this->downloadPath());
    m_file.setFileName(this->downloadPath() + m_track.data()->id());
    m_resumeSize = m_file.size();

    if (m_file.exists()) {
        if (!m_file.open(QIODevice::Append)) {
            this->setStatusInfo(tr("Cannot write to file"));
            this->setStatus(Transfers::Failed);
            return;
        }
    }
    else if (!m_file.open(QIODevice::WriteOnly)) {
        this->setStatusInfo(tr("Cannot write to file"));
        this->setStatus(Transfers::Failed);
        return;
    }

    this->setStatus(Transfers::Downloading);
    QNetworkRequest request(url);
    qDebug() << "Download url: " + url.toString();

    if (m_resumeSize > 0) {
        request.setRawHeader("Range", "bytes=" + QByteArray::number(m_resumeSize) + "-"); // Set 'Range' header if resuming a download
    }

    m_reply = this->networkAccessManager()->get(request);
    m_transferTime.start();
    m_progressTimer.start();

    this->connect(m_reply, SIGNAL(metaDataChanged()), this, SLOT(onDownloadMetadataChanged()));
    this->connect(m_reply, SIGNAL(downloadProgress(qint64,qint64)), this, SLOT(onDownloadProgressChanged(qint64,qint64)));
    this->connect(m_reply, SIGNAL(readyRead()), this, SLOT(onDownloadReadyRead()));
    this->connect(m_reply, SIGNAL(finished()), this, SLOT(onDownloadFinished()));
}

void TransferItem::pauseTransfer() {
    if (m_reply) {
        m_reply->abort();
    }
}

void TransferItem::cancelTransfer() {
    if (m_reply) {
        m_reply->abort();
    }
    else if (!m_file.fileName().isEmpty()) {
        m_file.remove();
    }
}

void TransferItem::onDownloadMetadataChanged() {
    qint64 size = m_reply->rawHeader("Content-Length").toLongLong();

    if (size > 0) {
        this->setSize(size + m_resumeSize);
    }
}

void TransferItem::onDownloadProgressChanged(qint64 received, qint64 total) {
    Q_UNUSED(total)

    if (received) {
        this->setProgress((m_resumeSize + received) * 100 / this->size());
        this->setSpeed(received * 1000 / m_transferTime.elapsed());
    }
}

void TransferItem::updateProgress() {
    emit progressChanged(this->progress());
    emit speedChanged(this->speed());
}

void TransferItem::onDownloadReadyRead() {
    m_file.write(m_reply->readAll());
}

void TransferItem::onDownloadFinished() {
    if (!m_reply) {
        this->setStatusInfo(tr("Network error"));
        this->setStatus(Transfers::Failed);
        return;
    }

    m_progressTimer.stop();
    m_file.close();
    QUrl redirect = m_reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();

    if (!redirect.isEmpty()) {
        m_reply->deleteLater();
        this->performDownload(redirect);
        return;
    }

    if (m_reply->error()) {
        if (m_reply->error() != QNetworkReply::OperationCanceledError) {
            if (m_retries > 2) {
                switch (this->transferType()) {
                case Transfers::PlaylistDownload:
                    m_reply->deleteLater();
                    m_file.remove();

                    if (!m_playlistTracks.isEmpty()) {
                        qDebug() << "Download error: skipping track";
                        m_track = m_playlistTracks.takeFirst();
                        this->setSize(m_track.data()->size());
                        this->setCurrentIndex(this->currentIndex() + 1);
                        this->startTransfer();
                    }
                    else {
                        this->downloadPlaylistThumbnail();
                    }

                    break;
                default:
                    break;
                }

                this->setStatusInfo(m_reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString());
                this->setStatus(Transfers::Failed);
                m_reply->deleteLater();
            }
            else {
                m_reply->deleteLater();
                m_retries++;
                qDebug() << "Retry number " + QString::number(m_retries);
                this->startTransfer(false);
            }
        }
        else {
            switch (this->status()) {
            case Transfers::Cancelled:
                m_file.remove();
                m_reply->deleteLater();
                emit statusChanged(Transfers::Cancelled);
                return;
            case Transfers::Paused:
                m_reply->deleteLater();
                emit statusChanged(Transfers::Paused);
                return;
            default:
                m_reply->deleteLater();
                return;
            }
        }
    }
    else {
        m_reply->deleteLater();
        this->downloadThumbnail();
    }
}

void TransferItem::downloadThumbnail() {
    QNetworkRequest request(m_track.data()->largeThumbnailUrl());
    m_reply = this->networkAccessManager()->get(request);
    this->connect(m_reply, SIGNAL(finished()), this, SLOT(onThumbnailDownloadFinished()));
}

void TransferItem::onThumbnailDownloadFinished() {
    if (!m_reply) {
        this->setStatus(Transfers::Completed);
        return;
    }

    if (m_reply->error()) {
        if (m_reply->error() == QNetworkReply::OperationCanceledError) {
            switch (this->status()) {
            case Transfers::Cancelled:
                m_file.remove();
                m_reply->deleteLater();
                emit statusChanged(Transfers::Cancelled);
                return;
            case Transfers::Paused:
                m_reply->deleteLater();
                emit statusChanged(Transfers::Paused);
                return;
            default:
                break;
            }
        }
    }

    QDir dir;
    dir.mkpath(Settings::instance()->downloadPath() + ".artwork/");
    QFile file(QString("%1.artwork/%2.jpg").arg(Settings::instance()->downloadPath()).arg(m_track.data()->id()));
    int num = 1;

    while ((file.exists()) && (num < 100)) {
        file.setFileName(QString("%1.artwork/%2(%3).jpg").arg(Settings::instance()->downloadPath()).arg(m_track.data()->id()).arg(num));
        num++;
    }

    if (file.open(QIODevice::WriteOnly)) {
        file.write(m_reply->readAll());
        file.close();
        m_track.data()->setThumbnailUrl(QUrl::fromLocalFile(file.fileName()));
    }
    else {
        m_track.data()->setThumbnailUrl(QUrl());
    }

    m_reply->deleteLater();

    this->downloadWaveform();
}

void TransferItem::downloadWaveform() {
    QNetworkRequest request(m_track.data()->waveformUrl());
    m_reply = this->networkAccessManager()->get(request);
    this->connect(m_reply, SIGNAL(finished()), this, SLOT(onWaveformDownloadFinished()));
}

void TransferItem::onWaveformDownloadFinished() {
    if (!m_reply) {
        this->setStatus(Transfers::Completed);
        return;
    }

    if (m_reply->error()) {
        if (m_reply->error() == QNetworkReply::OperationCanceledError) {
            switch (this->status()) {
            case Transfers::Cancelled:
                m_file.remove();
                m_reply->deleteLater();
                emit statusChanged(Transfers::Cancelled);
                return;
            case Transfers::Paused:
                m_reply->deleteLater();
                emit statusChanged(Transfers::Paused);
                return;
            default:
                break;
            }
        }
    }

    QDir dir;
    dir.mkpath(Settings::instance()->downloadPath() + ".artwork/");
    QFile file(QString("%1.artwork/%2-waveform.png").arg(Settings::instance()->downloadPath()).arg(m_track.data()->id()));
    int num = 1;

    while ((file.exists()) && (num < 100)) {
        file.setFileName(QString("%1.artwork/%2(%3)-waveform.png").arg(Settings::instance()->downloadPath()).arg(m_track.data()->id()).arg(num));
        num++;
    }

    if (file.open(QIODevice::WriteOnly)) {
        file.write(m_reply->readAll());
        file.close();
        m_track.data()->setWaveformUrl(QUrl::fromLocalFile(file.fileName()));
    }
    else {
        m_track.data()->setWaveformUrl(QUrl());
    }

    m_reply->deleteLater();

    this->moveDownloadedFiles();
}

void TransferItem::downloadPlaylistThumbnail() {
    QNetworkRequest request(m_playlist.data()->thumbnailUrl());
    m_reply = this->networkAccessManager()->get(request);
    this->connect(m_reply, SIGNAL(finished()), this, SLOT(onPlaylistThumbnailDownloadFinished()));
}

void TransferItem::onPlaylistThumbnailDownloadFinished() {
    if (!m_reply) {
        this->setStatus(Transfers::Completed);
        return;
    }

    if (m_reply->error()) {
        if (m_reply->error() == QNetworkReply::OperationCanceledError) {
            switch (this->status()) {
            case Transfers::Cancelled:
                m_file.remove();
                m_reply->deleteLater();
                emit statusChanged(Transfers::Cancelled);
                return;
            case Transfers::Paused:
                m_reply->deleteLater();
                emit statusChanged(Transfers::Paused);
                return;
            default:
                break;
            }
        }
    }

    QDir dir;
    dir.mkpath(Settings::instance()->downloadPath() + ".artwork/");
    QFile file(QString("%1.artwork/%2.jpg").arg(Settings::instance()->downloadPath()).arg(m_playlist.data()->id()));
    int num = 1;

    while ((file.exists()) && (num < 100)) {
        file.setFileName(QString("%1.artwork/%2(%3).jpg").arg(Settings::instance()->downloadPath()).arg(m_playlist.data()->id()).arg(num));
        num++;
    }

    if (file.open(QIODevice::WriteOnly)) {
        file.write(m_reply->readAll());
        file.close();
        m_playlist.data()->setThumbnailUrl(QUrl::fromLocalFile(file.fileName()));
    }
    else {
        m_playlist.data()->setThumbnailUrl(QUrl());
    }

    m_reply->deleteLater();

    emit playlistDownloadCompleted(m_playlist);
    this->setStatus(Transfers::Completed);
}

void TransferItem::moveDownloadedFiles() {
    QString oldFileName = this->downloadPath() + m_track.data()->id();
    QString newFileName = Settings::instance()->createArtistSubfolders() ? QString("%1%2/%3.%4").arg(Settings::instance()->downloadPath()).arg(m_track.data()->artist()).arg(m_track.data()->title().replace(illegalChars, "_")).arg(m_format == AudioFormats::OriginalFormat ? m_track.data()->format().toLower()
                                                                                                                                                                                                                                                                              : QString("mp3"))
                                                                         : QString("%1%2.%3").arg(Settings::instance()->downloadPath()).arg(m_track.data()->title().replace(illegalChars, "_")).arg(m_format == AudioFormats::OriginalFormat ? m_track.data()->format().toLower()
                                                                                                                                                                                                                                             : QString("mp3"));

    if (Settings::instance()->createArtistSubfolders()) {
        QDir(Settings::instance()->downloadPath()).mkdir(m_track.data()->artist());
    }

    qDebug() << "Old fileName: " + oldFileName;
    qDebug() << "New fileName: " + newFileName;
    int num = 1;
    bool fileSaved = QFile::rename(oldFileName, newFileName);

    while ((!fileSaved) && (num < 100)) {
        fileSaved = QFile::rename(oldFileName, QString("%1(%2).%3").arg(newFileName.section('.', 0, -2)).arg(num).arg(newFileName.section('.', -1)));
        num++;
    }

    if (fileSaved) {
        m_track.data()->setUrl(QUrl::fromLocalFile(newFileName));
        emit trackDownloadCompleted(m_track);

        switch (this->transferType()) {
        case Transfers::PlaylistDownload:
            if (m_playlistTracks.isEmpty()) {
                this->downloadPlaylistThumbnail();
                return;
            }
            else {
                m_track = m_playlistTracks.takeFirst();
                this->setSize(m_track.data()->size());
                this->setCurrentIndex(this->currentIndex() + 1);

                switch(this->status()) {
                case Transfers::Paused:
                    return;
                default:
                    this->startTransfer();
                    return;
                }
            }

            break;
        default:
            break;
        }

        this->setStatus(Transfers::Completed);
    }
    else {
        this->setStatusInfo(tr("Unable to rename temporary downloaded file"));
        this->setStatus(Transfers::Failed);
    }
}
