#include "tracklistwindow.h"
#include "trackinfowindow.h"
#include "tracklistdelegate.h"
#include "trackselectionwindow.h"
#include "nowplayingwindow.h"
#include "../base/tracklistmodel.h"
#include "../base/soundcloud.h"
#include "../base/settings.h"
#include "../base/transfermanager.h"
#include "../base/playbacklistmodel.h"
#include "../base/windowstack.h"
#include "../base/tracklauncher.h"
#include "../base/mediaplayer.h"
#include "listview.h"
#include <QAction>
#include <QLabel>
#include <QMenuBar>
#include <QTimer>
#include <QMaemo5Style>

TrackListWindow::TrackListWindow(QWidget *parent) :
    StackedWindow(parent),
    m_listView(new ListView(this)),
    m_model(new TrackListModel(this)),
    m_addToFavouritesAction(this->menuBar()->addAction(tr("Add to favourites"), this, SLOT(addTracksToFavourites()))),
    m_deleteFromFavouritesAction(this->menuBar()->addAction(tr("Delete from favourites"), this, SLOT(deleteTracksFromFavourites()))),
    m_addToQueueAction(this->menuBar()->addAction(tr("Add to playback queue"), this, SLOT(queueTracks()))),
    m_trackDownloadAction(this->menuBar()->addAction(tr("Download"), this, SLOT(addTrackDownloads()))),
    m_contextMenu(new QMenu(this)),
    m_favouritesContextAction(m_contextMenu->addAction(tr("Add to favourites"), this, SLOT(toggleFavourite()))),
    m_addToQueueContextAction(m_contextMenu->addAction(tr("Add to playback queue"), this, SLOT(queueTrack()))),
    m_trackDownloadContextAction(m_contextMenu->addAction(tr("Download"), this, SLOT(addTrackDownload()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No tracks found")), this))
{
    this->setCentralWidget(m_listView);

    TrackListDelegate *delegate = new TrackListDelegate(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(delegate);
    m_noResultsLabel->hide();

    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playTrack(QModelIndex)));
    this->connect(m_listView, SIGNAL(atEnd()), m_model, SLOT(getMoreTracks()));
    this->connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));

    this->onOrientationChanged();
}

void TrackListWindow::showEvent(QShowEvent *event) {
    StackedWindow::showEvent(event);

    if ((m_model) && (m_model->refreshRequired())) {
        QTimer::singleShot(1000, m_model, SLOT(reset()));
    }
}

void TrackListWindow::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void TrackListWindow::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void TrackListWindow::setTrackFeed(const QUrl &feed) {
    m_model->getTracks(feed);
    this->setMenuActions();
}

void TrackListWindow::getGroupTracks(QSharedPointer<GroupItem> group) {
    m_group = group;
    m_model->getGroupTracks(m_group.data()->id());
    this->setMenuActions();
}

void TrackListWindow::search(const QString &query, Queries::QueryOrder order) {
    this->setWindowTitle(tr("Search") + QString(" ('%1')").arg(query));
    m_model->search(query, order);
    this->setMenuActions();
}

void TrackListWindow::setMenuActions() {
    m_addToFavouritesAction->setEnabled((SoundCloud::instance()->userSignedIn()) && (m_model->feed() != FAVOURITES_FEED));
    m_deleteFromFavouritesAction->setEnabled((SoundCloud::instance()->userSignedIn()) && (m_model->feed() == FAVOURITES_FEED));
    m_favouritesContextAction->setVisible((SoundCloud::instance()->userSignedIn()) && (m_model->feed() == FAVOURITES_FEED));
}

void TrackListWindow::onLoadingChanged(bool loading) {
    this->toggleBusy(loading);
    m_noResultsLabel->setVisible((!loading) && (m_model->rowCount() == 0));
}

void TrackListWindow::onCountChanged(int count) {
    m_noResultsLabel->setVisible((count == 0) && (!m_model->loading()));
}

void TrackListWindow::addTracksToFavourites() {
    TrackSelectionWindow *window = new TrackSelectionWindow(m_model, tr("Add to favourites"), SLOT(addTracksToFavourites()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void TrackListWindow::deleteTracksFromFavourites() {
    TrackSelectionWindow *window = new TrackSelectionWindow(m_model, tr("Delete from favourites"), SLOT(deleteTracksFromFavourites()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void TrackListWindow::addTrackDownloads() {
    TrackSelectionWindow *window = new TrackSelectionWindow(m_model, tr("Download"), SLOT(addTrackDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void TrackListWindow::queueTracks() {
    TrackSelectionWindow *window = new TrackSelectionWindow(m_model, tr("Add to queue"), SLOT(queueTracks()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void TrackListWindow::showContextMenu(const QPoint &pos) {
    if (m_model->data(m_listView->currentIndex(), TrackListModel::FavouriteRole).toBool()) {
        m_favouritesContextAction->setText(tr("Delete from favourites"));
    }
    else {
        m_favouritesContextAction->setText(tr("Add to favourites"));
    }

    m_contextMenu->popup(pos, m_favouritesContextAction);
}

void TrackListWindow::toggleFavourite() {
    if (m_model->data(m_listView->currentIndex(), TrackListModel::FavouriteRole).toBool()) {
        SoundCloud::instance()->deleteFromFavourites(QStringList(m_model->data(m_listView->currentIndex(), TrackListModel::IdRole).toString()));
    }
    else {
        SoundCloud::instance()->addToFavourites(QStringList(m_model->data(m_listView->currentIndex(), TrackListModel::IdRole).toString()));
    }
}

void TrackListWindow::addTrackDownload() {
    TransferManager::instance()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()));
}

void TrackListWindow::queueTrack() {
    PlaybackListModel::playbackQueue()->addTrack(m_model->get(m_listView->currentIndex().row()));
}

void TrackListWindow::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        this->showTrackInfo(m_model->get(index.row()));
    }
}

void TrackListWindow::showTrackInfo(QSharedPointer<TrackItem> track) {
    TrackInfoWindow *window = new TrackInfoWindow(track, this);
    window->show();
}

void TrackListWindow::playTrack(const QModelIndex &index) {
    if (Settings::instance()->mediaPlayer() == "musikloud") {
        MediaPlayer::instance()->playTrack(m_model->get(index.row()));
        NowPlayingWindow *window = new NowPlayingWindow(this);
        window->show();
    }
    else {
        TrackLauncher::launchTrack(SoundCloud::instance()->getStreamUrl(index.data(TrackListModel::StreamUrlRole).toUrl()));
    }
}
