#ifndef DAILYMOTIONTrackLISTMODEL_H
#define DAILYMOTIONTrackLISTMODEL_H

#include "enums.h"
#include "trackitem.h"
#include <QSharedPointer>
#include <QAbstractListModel>

#ifndef QML_USER_INTERFACE
class ThumbnailCache;
#endif

class TrackListModel : public QAbstractListModel
{
    Q_OBJECT

    Q_PROPERTY(int count
               READ rowCount
               NOTIFY countChanged)
    Q_PROPERTY(bool loading
               READ loading
               NOTIFY loadingChanged)
    Q_PROPERTY(bool itemsSelected
               READ itemsSelected
               NOTIFY itemsSelectedChanged)
    Q_PROPERTY(QUrl feed
               READ feed
               WRITE setFeed
               NOTIFY feedChanged)
    Q_PROPERTY(bool refreshRequired
               READ refreshRequired
               NOTIFY refreshRequiredChanged)

public:
    enum Roles {
        IdRole = Qt::UserRole + 1,
        PlaylistIdRole,
        TitleRole,
        ArtistRole,
        DateRole,
        DurationRole,
        UrlRole,
        StreamUrlRole,
#ifndef QML_USER_INTERFACE
        ThumbnailRole,
        ThumbnailPressedRole,
#endif
        ThumbnailUrlRole,
        ServiceRole,
        FavouriteRole,
        SelectedRole
    };

public:
    explicit TrackListModel(QObject *parent = 0);
    ~TrackListModel();
    inline QUrl feed() const { return m_feed; }
    void setFeed(const QUrl &feed);
    inline QString playlistId() const { return m_playlistId; }
    inline QString groupId() const { return m_groupId; }
    inline bool loading() const { return m_loading; }
    inline bool moreResults() const { return m_moreResults; }
    inline bool refreshRequired() const { return m_refresh; }
    int rowCount(const QModelIndex &parent = QModelIndex()) const;
    QVariant data(const QModelIndex &index, int role) const;
    Q_INVOKABLE QVariant data(int row, const QByteArray &role) const;
#ifndef QML_USER_INTERFACE
    bool setData(const QModelIndex &index, const QVariant &value, int role);
#endif
    QSharedPointer<TrackItem> get(int row) const;
    Q_INVOKABLE inline QList<int> selectedRows() const { return m_selectedRows; }
    QList< QSharedPointer<TrackItem> > selectedItems() const;
    inline QList< QSharedPointer<TrackItem> > allItems() const { return m_list; }
#ifdef QML_USER_INTERFACE
    Q_INVOKABLE TrackItem* getFromQML(int row) const;
    Q_INVOKABLE QList<TrackItem*> selectedItemsFromQML() const;
    Q_INVOKABLE QList<TrackItem*> allItemsFromQML() const;
#endif
    inline bool itemsSelected() const { return !this->selectedRows().isEmpty(); }

public slots:
    void clear();
    void reset();
    void getTracks(QUrl feed = QUrl());
    void getMoreTracks();
    void getPlaylistTracks(const QString &id);
    void getGroupTracks(const QString &id);
    void search(const QString &query = QString(), int order = Queries::Hotness);
    void addSelectedTracksToFavourites();
    void deleteSelectedTracksFromFavourites();
    void downloadSelectedTracks();
    void queueSelectedTracks();
    void selectAll();
    void selectNone();
    void toggleSelected(int row);
    void insertTrack(int row, QSharedPointer<TrackItem> Track);
    void appendTrack(QSharedPointer<TrackItem> Track);
    bool removeTrack(const QVariant &value, int role = IdRole);

private:
    inline void setPlaylistId(const QString &id) { m_playlistId = id; }
    inline void setGroupId(const QString &id) { m_groupId = id; }
    inline QString query() const { return m_query; }
    inline void setQuery(const QString &query) { m_query = query; }
    inline int order() const { return m_order; }
    inline void setOrder(int order) { m_order = order; }
    inline void setLoading(bool loading) { m_loading = loading; emit loadingChanged(loading); }
    inline void setMoreResults(bool more) { m_moreResults = more; }
    inline int offset() const { return m_offset; }
    inline void setOffset(int offset) { m_offset = offset; }
    inline bool error() const { return m_error; }
    inline void setError(bool error) { m_error = error; }
    inline void setRefreshRequired(bool refresh) { m_refresh = refresh; emit refreshRequiredChanged(refresh); }
    bool removeRow(int row, const QModelIndex &parent = QModelIndex());

private slots:
    void addTracks();
    void onFavouriteChanged(const QString &id, bool favourite);
    inline void onError() { this->setError(true); this->setLoading(false); }
    inline void stopLoading() { this->setLoading(false); emit countChanged(this->rowCount()); }
#ifndef QML_USER_INTERFACE
    void onThumbnailReady();
#endif

signals:
    void loadingChanged(bool loading);
    void countChanged(int count);
    void itemsSelectedChanged(bool selected);
    void refreshRequiredChanged(bool refresh);
    void feedChanged(const QUrl &feed);

private:
#ifndef QML_USER_INTERFACE
    ThumbnailCache *m_cache;
    int m_thumbnailPressedRow;
#endif
    QUrl m_feed;
    QString m_playlistId;
    QString m_groupId;
    QString m_query;
    int m_order;
    bool m_loading;
    bool m_moreResults;
    bool m_error;
    int m_offset;
    bool m_refresh;
    QList< QSharedPointer<TrackItem> > m_list;
    QList<int> m_selectedRows;
};

#endif // DAILYMOTIONTrackLISTMODEL_H
