#include "playlistlistmodel.h"
#include "feedurls.h"
#include "json.h"
#include "utils.h"
#include "notifications.h"
#include "soundcloud.h"
#ifndef QML_USER_INTERFACE
#include "thumbnailcache.h"
#endif
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QTimer>

using namespace QtJson;

PlaylistListModel::PlaylistListModel(QObject *parent) :
    QAbstractListModel(parent),
    #ifndef QML_USER_INTERFACE
    m_cache(new ThumbnailCache),
    #endif
    m_loading(true),
    m_moreResults(true),
    m_error(false),
    m_offset(0)
{
    QHash<int, QByteArray> roles;
    roles[IdRole] = "id";
    roles[TitleRole] = "title";
    roles[ArtistRole] = "artist";
    roles[DateRole] = "date";
    roles[TrackCountRole] = "trackCount";
#ifndef QML_USER_INTERFACE
    roles[ThumbnailRole] = "thumbnail";
#endif
    roles[ThumbnailUrlRole] = "thumbnailUrl";
    roles[DescriptionRole] = "description";
    this->setRoleNames(roles);
#ifndef QML_USER_INTERFACE
    this->connect(m_cache, SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
#endif
}

PlaylistListModel::~PlaylistListModel() {
    m_list.clear();
#ifndef QML_USER_INTERFACE
    delete m_cache;
    m_cache = 0;
#endif
}

void PlaylistListModel::clear() {
    this->beginResetModel();
    m_list.clear();
    this->endResetModel();
    this->setLoading(false);
    this->setOffset(1);
    this->setMoreResults(true);
    this->setError(false);
}

void PlaylistListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getMorePlaylists();
    }
}

int PlaylistListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return m_list.size();
}

QVariant PlaylistListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case IdRole:
        return m_list.at(index.row()).data()->id();
    case TitleRole:
        return m_list.at(index.row()).data()->title();
    case ArtistRole:
        return m_list.at(index.row()).data()->artist();
    case DateRole:
        return m_list.at(index.row()).data()->date();
    case TrackCountRole:
        return m_list.at(index.row()).data()->trackCount();
#ifndef QML_USER_INTERFACE
    case ThumbnailRole:
        return m_cache->thumbnail(m_list.at(index.row()).data()->thumbnailUrl(), QSize(64, 64));
#endif
    case ThumbnailUrlRole:
        return m_list.at(index.row()).data()->thumbnailUrl();
    case DescriptionRole:
        return m_list.at(index.row()).data()->description();
    default:
        return QVariant();
    }
}

QVariant PlaylistListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}

QSharedPointer<PlaylistItem> PlaylistListModel::get(int row) const {
    if ((row >= 0) && (row < m_list.size())) {
        return QSharedPointer<PlaylistItem>(m_list.at(row));
    }

    return QSharedPointer<PlaylistItem>();
}

#ifdef QML_USER_INTERFACE
PlaylistItem* PlaylistListModel::getFromQML(int row) const {
    if ((row >= 0) && (row < m_list.size())) {
        return m_list.at(row).data();
    }

    return 0;
}
#endif

void PlaylistListModel::insertPlaylist(int row, QSharedPointer<PlaylistItem> playlist) {
    Q_ASSERT((row >= 0) && (row <= this->rowCount()));

    this->beginInsertRows(QModelIndex(), row, row);
    m_list.insert(row, playlist);
    this->endInsertRows();
}

void PlaylistListModel::appendPlaylist(QSharedPointer<PlaylistItem> playlist) {
    this->beginInsertRows(QModelIndex(), this->rowCount(), this->rowCount());
    m_list.append(playlist);
    this->endInsertRows();
}

bool PlaylistListModel::removePlaylist(const QVariant &value, int role) {
    QModelIndexList indexes = this->match(this->index(0), role, value, 1, Qt::MatchExactly);

    if (indexes.isEmpty()) {
        return false;
    }

    return this->removeRow(indexes.first().row());
}

bool PlaylistListModel::removeRow(int row, const QModelIndex &parent)
{
    Q_UNUSED(parent)

    if ((row >= 0) && (row < m_list.size())) {
        this->beginRemoveRows(QModelIndex(), row, row);
        m_list.takeAt(row).clear();
        this->endRemoveRows();

        return true;
    }

    return false;
}

#ifndef QML_USER_INTERFACE
void PlaylistListModel::onThumbnailReady() {
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
}
#endif

void PlaylistListModel::setFeed(const QUrl &feed) {
    if (feed != this->feed()) {
        m_feed = feed;
    }
}

void PlaylistListModel::getPlaylists(QUrl feed) {
    if (!feed.isEmpty()) {
        this->setFeed(feed);
    }
    else {
        feed = this->feed();
    }

    this->setLoading(true);
    QNetworkReply *reply = SoundCloud::instance()->createReply(feed, this->offset());
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addPlaylists()));
}

void PlaylistListModel::search(const QString &query, int order) {
    if (!query.isEmpty()) {
        this->setQuery(query);
        this->setOrder(order);
    }

    this->setLoading(true);
    QNetworkReply *reply = SoundCloud::instance()->createSearchReply(Queries::Playlists, this->query(), this->order(), this->offset());
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addPlaylists()));
}

void PlaylistListModel::addPlaylists() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setLoading(false);
        this->setError(true);
        Notifications::instance()->onError(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    switch (statusCode) {
    case 200:
    case 201: {
        QString response(reply->readAll());
        bool ok;
        QVariantList entries = Json::parse(response, ok).toList();

        if (!ok) {
            this->setLoading(false);
            this->setError(true);
            Notifications::instance()->onError(tr("Cannot parse server response"));
        }
        else {
            for (int i = 0; i < entries.size(); i++) {
                QMetaObject::invokeMethod(this, "appendPlaylist", Qt::QueuedConnection, Q_ARG(QSharedPointer<PlaylistItem>, QSharedPointer<PlaylistItem>(new PlaylistItem(entries.at(i).toMap()))));
            }

            this->setMoreResults(!entries.isEmpty());
            this->setOffset(this->offset() + entries.size());
            QTimer::singleShot(1000, this, SLOT(stopLoading()));
        }
    }
        break;
    default:
        this->setLoading(false);
        this->setError(true);
        Notifications::instance()->onError(Utils::httpErrorString(statusCode));
        break;
    }

    reply->deleteLater();
}

void PlaylistListModel::getMorePlaylists() {
    if ((this->moreResults()) && (!this->loading())) {
        if (!this->query().isEmpty()) {
            this->search();
        }
        else {
            this->getPlaylists();
        }
    }
}
