#include "mediaplayer.h"
#include "playbacklistmodel.h"
#include "session.h"
#include "soundcloud.h"
#include "notifications.h"
#include "settings.h"
#include "lastfm.h"
#include "database.h"
#include "utils.h"

MediaPlayer::MediaPlayer* mediaplayerInstance = 0;

MediaPlayer::MediaPlayer(QObject *parent) :
    QMediaPlayer(parent),
    m_index(0),
    m_playbackMode(PlaybackMode::Sequential)
{
    if (!mediaplayerInstance) {
        mediaplayerInstance = this;
    }

    this->connect(this, SIGNAL(stateChanged(QMediaPlayer::State)), this, SLOT(onStateChanged(QMediaPlayer::State)));
    this->connect(this, SIGNAL(mediaStatusChanged(QMediaPlayer::MediaStatus)), this, SLOT(onMediaStatusChanged(QMediaPlayer::MediaStatus)));
    this->connect(this, SIGNAL(error(QMediaPlayer::Error)), this, SLOT(onError(QMediaPlayer::Error)));
#ifdef MEDIA_PLAYERS_ENABLED
    this->connect(Settings::instance(), SIGNAL(mediaPlayerChanged(QString)), this, SLOT(onMediaPlayerChanged(QString)));
#endif
    this->connect(PlaybackListModel::playbackQueue(), SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
#if (defined (MEDIA_PLAYERS_ENABLED) && (defined (QML_USER_INTERFACE)))
    this->onMediaPlayerChanged(Settings::instance()->mediaPlayer());
#else
    if (PlaybackListModel::playbackQueue()->rowCount() > 0) {
        this->setCurrentTrack(PlaybackListModel::playbackQueue()->get(0));
    }
#endif
}

MediaPlayer::~MediaPlayer() {}

MediaPlayer* MediaPlayer::instance() {
    return !mediaplayerInstance ? new MediaPlayer(Session::instance()) : mediaplayerInstance;
}

#ifdef QML_USER_INTERFACE
void MediaPlayer::playTrack(TrackItem *track, bool playImmediately) {
    PlaybackListModel::playbackQueue()->clear();
    PlaybackListModel::playbackQueue()->addTrackFromQML(track, false);
    this->setCurrentIndex(0, playImmediately);
}

void MediaPlayer::playTracks(QList<TrackItem*> tracks, bool playImmediately) {
    PlaybackListModel::playbackQueue()->clear();
    PlaybackListModel::playbackQueue()->addTracksFromQML(tracks, false);
    this->setCurrentIndex(0, playImmediately);
}
#else
void MediaPlayer::playTrack(QSharedPointer<TrackItem> track, bool playImmediately) {
    PlaybackListModel::playbackQueue()->clear();
    PlaybackListModel::playbackQueue()->addTrack(track, false);
    this->setCurrentIndex(0, playImmediately);
}

void MediaPlayer::playTracks(QList<QSharedPointer<TrackItem> > tracks, bool playImmediately) {
    PlaybackListModel::playbackQueue()->clear();
    PlaybackListModel::playbackQueue()->addTracks(tracks, false);
    this->setCurrentIndex(0, playImmediately);
}
#endif
void MediaPlayer::removeTrack(int row) {
    PlaybackListModel::playbackQueue()->removeTrack(row);

    if (row == this->currentIndex()) {
        if (PlaybackListModel::playbackQueue()->rowCount() > 0) {
            this->setCurrentIndex(row);
        }
    }
    else if (row < this->currentIndex()) {
        m_index--;
        emit currentIndexChanged(this->currentIndex());
    }
}

void MediaPlayer::clearTracks() {
    PlaybackListModel::playbackQueue()->clear();
}

void MediaPlayer::setCurrentTrack(QSharedPointer<TrackItem> track) {
    m_track = track;

    if (track.isNull()) {
        return;
    }
#ifdef QML_USER_INTERFACE
    emit currentTrackChanged(track.data());
#else
    emit currentTrackChanged(track);
#endif
    switch (track.data()->service()) {
    case Services::NoService:
        if ((Settings::instance()->archiveOnline()) && (!track.data()->favourite()) && (!track.data()->id().isEmpty()) && (SoundCloud::instance()->userSignedIn())) {
            SoundCloud::instance()->trackIsFavourite(track.data()->id());
        }

        return;
    default:
        return;
    }
}

void MediaPlayer::setCurrentIndex(int index, bool playImmediately) {
    m_index = index;
    emit currentIndexChanged(index);

    if ((index >= 0) && (index < PlaybackListModel::playbackQueue()->rowCount())) {
        this->setCurrentTrack(PlaybackListModel::playbackQueue()->get(index));

        if (playImmediately) {
            this->play(true);
        }
    }
}

void MediaPlayer::play(bool resetSource) {
    if (((this->media().isNull()) || (resetSource)) && (!this->currentTrack().isNull())) {
        switch (this->currentTrack().data()->service()) {
        case Services::NoService:
            this->setMedia(this->currentTrack().data()->url());
            break;
        default:
            this->setMedia(SoundCloud::instance()->getStreamUrl(this->currentTrack().data()->streamUrl()));
            break;
        }
    }

    QMediaPlayer::play();
}

void MediaPlayer::togglePlayPause() {
    switch (this->state()) {
    case PlayingState:
        this->pause();
        return;
    default:
        this->play();
        return;
    }
}

void MediaPlayer::togglePlaybackMode() {
    switch (this->playbackMode()) {
    case PlaybackMode::Sequential:
        this->setPlaybackMode(PlaybackMode::RepeatAll);
        return;
    case PlaybackMode::RepeatAll:
        this->setPlaybackMode(PlaybackMode::RepeatOne);
        return;
    default:
        this->setPlaybackMode(PlaybackMode::Sequential);
        return;
    }
}

void MediaPlayer::setPlaybackMode(PlaybackMode::Mode mode) {
    if (mode != this->playbackMode()) {
        m_playbackMode = mode;
        emit playbackModeChanged(mode);
    }
}

bool MediaPlayer::playing() const {
    switch (this->state()) {
    case PlayingState:
        return true;
    default:
        return false;
    }
}

bool MediaPlayer::paused() const {
    switch (this->state()) {
    case PausedState:
        return true;
    default:
        return false;
    }
}

void MediaPlayer::next() {
    switch (this->playbackMode()) {
    case PlaybackMode::RepeatAll:
        if (this->currentIndex() < PlaybackListModel::playbackQueue()->rowCount() - 1) {
            this->setCurrentIndex(this->currentIndex() + 1);
        }
        else {
            this->setCurrentIndex(0);
        }

        return;
    case PlaybackMode::RepeatOne:
        this->play();
        return;
    default:
        if (this->currentIndex() < PlaybackListModel::playbackQueue()->rowCount() - 1) {
            this->setCurrentIndex(this->currentIndex() + 1);
        }

        return;
    }
}

void MediaPlayer::previous() {
    if (this->currentIndex() > 0) {
        this->setCurrentIndex(this->currentIndex() - 1);
    }
}

void MediaPlayer::onStateChanged(QMediaPlayer::State state) {
    switch (state) {
    case PlayingState:
        emit playingChanged(true);
        emit pausedChanged(false);
        return;
    case PausedState:
        emit playingChanged(false);
        emit pausedChanged(true);
        return;
    default:
        emit playingChanged(false);
        emit pausedChanged(false);
        return;
    }
}

void MediaPlayer::onMediaStatusChanged(QMediaPlayer::MediaStatus status) {
    switch (status) {
    case EndOfMedia:
        this->onTrackPlayed(this->currentTrack());
        this->setPosition(0);
        this->next();
        return;
    default:
        return;
    }
}

void MediaPlayer::onError(QMediaPlayer::Error error) {
    switch (error) {
    case NoError:
        return;
    default:
        this->setMedia(QMediaContent());
        Notifications::instance()->onError(this->errorString());
        return;
    }
}

#ifdef MEDIA_PLAYERS_ENABLED
void MediaPlayer::onMediaPlayerChanged(const QString &mediaPlayer) {
    if (mediaPlayer != "musikloud") {
        this->stop();
#ifdef QML_USER_INTERFACE
        this->setCurrentIndex(0);
        this->setCurrentTrack(QSharedPointer<TrackItem>());
        this->disconnect(PlaybackListModel::playbackQueue(), SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
#else
        this->deleteLater();
#endif
    }
#ifdef QML_USER_INTERFACE
    else {
        this->onCountChanged(PlaybackListModel::playbackQueue()->rowCount());
        this->connect(PlaybackListModel::playbackQueue(), SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    }
#endif
}
#endif

void MediaPlayer::onTrackPlayed(QSharedPointer<TrackItem> track) {
    track.data()->setPlayCount(track.data()->playCount() + 1);
    track.data()->setLastPlayed(QDateTime::currentMSecsSinceEpoch());

    switch (track.data()->service()) {
    case Services::NoService:
        QMetaObject::invokeMethod(Database::instance(), "updateTrack", Q_ARG(QUrl, track.data()->url()), Q_ARG(QString, "playCount"), Q_ARG(QVariant, track.data()->playCount()));
        QMetaObject::invokeMethod(Database::instance(), "updateTrack", Q_ARG(QUrl, track.data()->url()), Q_ARG(QString, "lastPlayed"), Q_ARG(QVariant, track.data()->lastPlayed()));
        break;
    default:
        break;
    }

    if ((Settings::instance()->scrobbleTracks()) && (Lastfm::instance()->userSignedIn()) && (!track.data()->artist().isEmpty())) {
        Lastfm::instance()->scrobbleTrack(track.data()->artist(), track.data()->title());
    }
}

void MediaPlayer::onCountChanged(int count) {
    switch (count) {
    case 0:
        this->stop();
        this->setMedia(QMediaContent());
        this->setCurrentIndex(0);
        this->setCurrentTrack(QSharedPointer<TrackItem>());
        return;
    default:
        if (this->currentTrack().isNull()) {
            this->setCurrentTrack(PlaybackListModel::playbackQueue()->get(0));
        }

        return;
    }
}
