#ifndef DATABASE_H
#define DATABASE_H

#include <QObject>
#include <QSqlDatabase>
#include <qplatformdefs.h>
#ifdef MEEGO_EDITION_HARMATTAN
#include "../harmattan/transferitem.h"
#elif (defined (Q_WS_MAEMO_5))
#include "../maemo5/transferitem.h"
#elif (defined (SYMBIAN_OS))
#include "../symbian/transferitem.h"
#endif

class QUrl;
class QStringList;
class QThread;

class Database : public QObject
{
    Q_OBJECT

public:
    explicit Database();
    ~Database();
    void restoreAccounts();
    inline bool busy() const { return m_busy; }
    inline bool cancelled() const { return m_cancelled; }

public slots:
    QList< QSharedPointer<TransferItem> > getStoredDownloads();
    bool storeDownload(QSharedPointer<TransferItem> transfer);
    bool removeStoredDownload(QSharedPointer<TransferItem> transfer);
    bool storeSoundCloudToken(const QString &token);
    bool storeSoundCloudUsername(const QString &user);
    bool storeLastfmAccount(const QString &user, const QString &token);
    bool deleteSoundCloudAccount();
    bool deleteLastfmAccount();
    bool addTrack(QSharedPointer<TrackItem> track);
    bool addPlaylist(QSharedPointer<PlaylistItem> playlist);
    bool updateTrack(const QUrl &url, const QString &key, const QVariant &value);
    bool updatePlaylist(const QString &id, const QString &key, const QVariant &value);
    bool deleteTracks(QList< QSharedPointer<TrackItem> > tracks, bool deleteFromFS = false, bool updatePlaylistTrackCount = true);
    bool deletePlaylist(const QString &id, bool deleteTracksFromFS = false);
    QList< QSharedPointer<TrackItem> > getTracks(const QString &playlistId = QString());
    QList< QSharedPointer<PlaylistItem> > getPlaylists();
    void cancelCurrentOperation();

    static Database* instance();

private:
    void initialize();
    bool storeTrackDownload(QSharedPointer<TrackItem> track);
    bool storePlaylistDownload(QSharedPointer<PlaylistItem> playlist);
    bool removeStoredTrackDownload(const QString &id, const QString &key = "id");
    bool removeStoredPlaylistDownload(const QString &id);
    QList< QSharedPointer<TrackItem> > getStoredTrackDownloads();
    QList< QSharedPointer<PlaylistItem> > getStoredPlaylistDownloads();
    QSharedPointer<PlaylistItem> getPlaylist(const QString &id);
    bool updatePlaylist(const QString &id, int trackCountChange, int durationChange);
    bool deleteTrack(QSharedPointer<TrackItem> track, bool deleteFromFS = false, bool updatePlaylistTrackCount = true);
    void restoreSoundCloudAccount();
    void restoreLastfmAccount();
    void restoreFacebookAccount();
    void restoreTwitterAccount();
    void setBusy(bool isBusy, const QString &message = QString(), int numberOfOperations = 1);
    inline void setCancelled(bool cancelled) { m_cancelled = cancelled; }

signals:
    void alert(const QString &message);
    void busy(const QString &message, int numberOfOperations = 1);
    void busyProgressChanged(int progress);
    void busyChanged(bool isBusy);
    void info(const QString &message);
    void error(const QString &errorString);
    void tracksUpdated();
    void playlistsUpdated();
    void gotSoundCloudAccount(const QString &user, const QString &token);
    void gotSoundCloudUsername(const QString &user);
    void gotSoundCloudToken(const QString &token);
    void gotLastfmAccount(const QString &user, const QString &token);
    void currentOperationCancelled();

private:
    QThread *m_thread;
    QSqlDatabase m_database;
    bool m_busy;
    bool m_cancelled;
};

#endif // DATABASE_H
