#include "userlistmodel.h"
#include "feedurls.h"
#include "json.h"
#include "soundcloud.h"
#include "utils.h"
#include "notifications.h"
#ifndef QML_USER_INTERFACE
#include "thumbnailcache.h"
#endif
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QTimer>

using namespace QtJson;

UserListModel::UserListModel(QObject *parent) :
    QAbstractListModel(parent),
    #ifndef QML_USER_INTERFACE
    m_cache(new ThumbnailCache),
    #endif
    m_loading(true),
    m_moreResults(true),
    m_error(false),
    m_offset(0)
{
    QHash<int, QByteArray> roles;
    roles[IdRole] = "id";
    roles[UsernameRole] = "username";
#ifndef QML_USER_INTERFACE
    roles[AvatarRole] = "avatar";
#endif
    roles[AvatarUrlRole] = "avatarUrl";
    roles[TrackCountRole] = "trackCount";
    roles[PlaylistCountRole] = "playlistCount";
    this->setRoleNames(roles);
#ifndef QML_USER_INTERFACE
    this->connect(m_cache, SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
#endif
}

UserListModel::~UserListModel() {
    m_list.clear();
#ifndef QML_USER_INTERFACE
    delete m_cache;
    m_cache = 0;
#endif
}

void UserListModel::clear() {
    this->beginResetModel();
    m_list.clear();
    this->endResetModel();
    this->setLoading(false);
    this->setOffset(1);
    this->setMoreResults(true);
    this->setError(false);
}

void UserListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getMoreUsers();
    }
}

int UserListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return m_list.size();
}

QVariant UserListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case IdRole:
        return m_list.at(index.row()).data()->id();
    case UsernameRole:
        return m_list.at(index.row()).data()->username();
#ifndef QML_USER_INTERFACE
    case AvatarRole:
        return m_cache->thumbnail(m_list.at(index.row()).data()->avatarUrl(), QSize(64, 64));
#endif
    case AvatarUrlRole:
        return m_list.at(index.row()).data()->avatarUrl();
    case TrackCountRole:
        return m_list.at(index.row()).data()->trackCount();
    case PlaylistCountRole:
        return m_list.at(index.row()).data()->playlistCount();
    default:
        return QVariant();
    }
}

QVariant UserListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}

QSharedPointer<UserItem> UserListModel::get(int row) const {
    if ((row >= 0) && (row < m_list.size())) {
        return QSharedPointer<UserItem>(m_list.at(row));
    }

    return QSharedPointer<UserItem>();
}

#ifdef QML_USER_INTERFACE
UserItem* UserListModel::getFromQML(int row) const {
    if ((row >= 0) && (row < m_list.size())) {
        return m_list.at(row).data();
    }

    return 0;
}
#endif

void UserListModel::insertUser(int row, QSharedPointer<UserItem> user) {
    Q_ASSERT((row >= 0) && (row <= this->rowCount()));

    this->beginInsertRows(QModelIndex(), row, row);
    m_list.insert(row, user);
    this->endInsertRows();
}

void UserListModel::appendUser(QSharedPointer<UserItem> user) {
    this->beginInsertRows(QModelIndex(), this->rowCount(), this->rowCount());
    m_list.append(user);
    this->endInsertRows();
}

bool UserListModel::removeUser(const QVariant &value, int role) {
    QModelIndexList indexes = this->match(this->index(0), role, value, 1, Qt::MatchExactly);

    if (indexes.isEmpty()) {
        return false;
    }

    return this->removeRow(indexes.first().row());
}

bool UserListModel::removeRow(int row, const QModelIndex &parent)
{
    Q_UNUSED(parent)

    if ((row >= 0) && (row < m_list.size())) {
        this->beginRemoveRows(QModelIndex(), row, row);
        m_list.takeAt(row).clear();
        this->endRemoveRows();

        return true;
    }

    return false;
}

#ifndef QML_USER_INTERFACE
void UserListModel::onThumbnailReady() {
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
}
#endif

void UserListModel::setFeed(const QUrl &feed) {
    if (feed != this->feed()) {
        m_feed = feed;
    }
}

void UserListModel::getUsers(QUrl feed) {
    if (!feed.isEmpty()) {
        this->setFeed(feed);
    }
    else {
        feed = this->feed();
    }

    this->setLoading(true);
    QNetworkReply *reply = SoundCloud::instance()->createReply(feed, this->offset());
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addUsers()));
}

void UserListModel::search(const QString &query) {
    if (!query.isEmpty()) {
        this->setQuery(query);
    }

    this->setLoading(true);
    QNetworkReply *reply = SoundCloud::instance()->createSearchReply(Queries::Users, this->query(), Queries::Hotness, this->offset());
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addUsers()));
}

void UserListModel::addUsers() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setLoading(false);
        this->setError(true);
        Notifications::instance()->onError(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    switch (statusCode) {
    case 200:
    case 201: {
        QString response(reply->readAll());
        bool ok;
        QVariantList entries = Json::parse(response, ok).toList();

        if (!ok) {
            this->setLoading(false);
            this->setError(true);
            Notifications::instance()->onError(tr("Cannot parse server response"));
        }
        else {
            for (int i = 0; i < entries.size(); i++) {
                QMetaObject::invokeMethod(this, "appendUser", Qt::QueuedConnection, Q_ARG(QSharedPointer<UserItem>, QSharedPointer<UserItem>(new UserItem(entries.at(i).toMap()))));
            }

            this->setMoreResults(!entries.isEmpty());
            this->setOffset(this->offset() + entries.size());
            QTimer::singleShot(1000, this, SLOT(stopLoading()));
        }
    }
        break;
    default:
        this->setLoading(false);
        this->setError(true);
        Notifications::instance()->onError(Utils::httpErrorString(statusCode));
        break;
    }

    reply->deleteLater();
}

void UserListModel::getMoreUsers() {
    if ((this->moreResults()) && (!this->loading())) {
        if (!this->query().isEmpty()) {
            this->search();
        }
        else {
            this->getUsers();
        }
    }
}
