#include "mediaplayer.h"
#include "playbacklistmodel.h"
#include "session.h"
#include "soundcloud.h"
#include "notifications.h"
#include "settings.h"
#include "lastfm.h"
#include "database.h"
#include "utils.h"

MediaPlayer* mediaplayerInstance = 0;

MediaPlayer::MediaPlayer(QObject *parent) :
    QMediaPlayer(parent),
    m_index(0),
    m_playbackMode(Sequential)
{
    if (!mediaplayerInstance) {
        mediaplayerInstance = this;
    }

    if (PlaybackListModel::playbackQueue()->rowCount() > 0) {
        this->setCurrentTrack(PlaybackListModel::playbackQueue()->get(0));
    }

    this->connect(this, SIGNAL(mediaStatusChanged(QMediaPlayer::MediaStatus)), this, SLOT(onMediaStatusChanged(QMediaPlayer::MediaStatus)));
    this->connect(this, SIGNAL(error(QMediaPlayer::Error)), this, SLOT(onError(QMediaPlayer::Error)));
    this->connect(Settings::instance(), SIGNAL(mediaPlayerChanged(QString)), this, SLOT(onMediaPlayerChanged(QString)));
    this->connect(PlaybackListModel::playbackQueue(), SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
}

MediaPlayer::~MediaPlayer() {}

MediaPlayer* MediaPlayer::instance() {
    return !mediaplayerInstance ? new MediaPlayer(Session::instance()) : mediaplayerInstance;
}

#ifdef QML_USER_INTERFACE
void MediaPlayer::playTrack(TrackItem *track) {
    PlaybackListModel::playbackQueue()->clear();
    PlaybackListModel::playbackQueue()->addTrackFromQML(track, false);
    this->setCurrentIndex(0);
}

void MediaPlayer::playTracks(QList<TrackItem*> tracks) {
    PlaybackListModel::playbackQueue()->clear();
    PlaybackListModel::playbackQueue()->addTracksFromQML(tracks, false);
    this->setCurrentIndex(0);
}
#else
void MediaPlayer::playTrack(QSharedPointer<TrackItem> track) {
    PlaybackListModel::playbackQueue()->clear();
    PlaybackListModel::playbackQueue()->addTrack(track, false);
    this->setCurrentIndex(0);
}

void MediaPlayer::playTracks(QList<QSharedPointer<TrackItem> > tracks) {
    PlaybackListModel::playbackQueue()->clear();
    PlaybackListModel::playbackQueue()->addTracks(tracks, false);
    this->setCurrentIndex(0);
}
#endif
void MediaPlayer::removeTrack(int row) {
    PlaybackListModel::playbackQueue()->removeTrack(row);

    if (row == this->currentIndex()) {
        this->setCurrentIndex(row);
    }
    else if (row < this->currentIndex()) {
        m_index--;
        emit currentIndexChanged(this->currentIndex());
    }
}

void MediaPlayer::clearTracks() {
    PlaybackListModel::playbackQueue()->clear();
}

void MediaPlayer::setCurrentTrack(QSharedPointer<TrackItem> track) {
    m_track = track;

    if (track.isNull()) {
        return;
    }
#ifdef QML_USER_INTERFACE
    emit currentTrackChanged(track.data());
#else
    emit currentTrackChanged(track);
#endif
    switch (track.data()->service()) {
    case Services::NoService:
        this->setMedia(track.data()->url());
        return;
    default:
        this->setMedia(SoundCloud::instance()->getStreamUrl(track.data()->streamUrl()));
        return;
    }
}

void MediaPlayer::setCurrentIndex(int index) {
    m_index = index;
    emit currentIndexChanged(index);

    if ((index >= 0) && (index < PlaybackListModel::playbackQueue()->rowCount())) {
        this->setCurrentTrack(PlaybackListModel::playbackQueue()->get(index));
        this->play();
    }
}

void MediaPlayer::togglePlayPause() {
    switch (this->state()) {
    case PlayingState:
        this->pause();
        return;
    default:
        this->play();
        return;
    }
}

void MediaPlayer::togglePlaybackMode() {
    switch (this->playbackMode()) {
    case Sequential:
        this->setPlaybackMode(RepeatAll);
        return;
    case RepeatAll:
        this->setPlaybackMode(RepeatOne);
        return;
    default:
        this->setPlaybackMode(Sequential);
        return;
    }
}

void MediaPlayer::setPlaybackMode(PlaybackMode mode) {
    if (mode != this->playbackMode()) {
        m_playbackMode = mode;
        emit playbackModeChanged(mode);
    }
}

void MediaPlayer::next() {
    switch (this->playbackMode()) {
    case RepeatAll:
        if (this->currentIndex() < PlaybackListModel::playbackQueue()->rowCount() - 1) {
            this->setCurrentIndex(this->currentIndex() + 1);
        }
        else {
            this->setCurrentIndex(0);
        }

        return;
    case RepeatOne:
        this->play();
        return;
    default:
        if (this->currentIndex() < PlaybackListModel::playbackQueue()->rowCount() - 1) {
            this->setCurrentIndex(this->currentIndex() + 1);
        }

        return;
    }
}

void MediaPlayer::previous() {
    if (this->currentIndex() > 0) {
        this->setCurrentIndex(this->currentIndex() - 1);
    }
}

void MediaPlayer::onMediaStatusChanged(QMediaPlayer::MediaStatus status) {
    switch (status) {
    case EndOfMedia:
        this->onTrackPlayed(this->currentTrack());
        this->next();
        return;
    default:
        return;
    }
}

void MediaPlayer::onError(QMediaPlayer::Error error) {
    switch (error) {
    case NoError:
        return;
    default:
        Notifications::instance()->onError(this->errorString());
        return;
    }
}

void MediaPlayer::onMediaPlayerChanged(const QString &mediaPlayer) {
    if (mediaPlayer != "musikloud") {
        this->stop();
        this->deleteLater();
    }
}

void MediaPlayer::onTrackPlayed(QSharedPointer<TrackItem> track) {
    track.data()->setPlayCount(track.data()->playCount() + 1);
    track.data()->setLastPlayed(Utils::dateFromMSecs(QDateTime::currentMSecsSinceEpoch()));

    switch (track.data()->service()) {
    case Services::NoService:
        QMetaObject::invokeMethod(Database::instance(), "updateTrack", Q_ARG(QUrl, track.data()->url()), Q_ARG(QString, "playCount"), Q_ARG(QVariant, track.data()->playCount()));
        QMetaObject::invokeMethod(Database::instance(), "updateTrack", Q_ARG(QUrl, track.data()->url()), Q_ARG(QString, "lastPlayed"), Q_ARG(QVariant, QDateTime::currentMSecsSinceEpoch()));
        break;
    default:
        break;
    }

    if ((Settings::instance()->scrobbleTracks()) && (Lastfm::instance()->userSignedIn()) && (!track.data()->artist().isEmpty())) {
        Lastfm::instance()->scrobbleTrack(track.data()->artist(), track.data()->title());
    }
}

void MediaPlayer::onCountChanged(int count) {
    if (count == 0) {
        this->stop();
        this->setCurrentIndex(0);
    }
}
