#include "lastfm.h"
#include "utils.h"
#include <QRegExp>
#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QUrl>
#include <QCryptographicHash>
#include <QXmlStreamReader>
#include <QStringList>
#include <QDateTime>

const QString API_KEY("ac4c4586cc1441c74011bf89b28af6b3");
const QString API_SECRET("05e69588c52c6da942f56d3faed3c45f");

Lastfm *lastfmInstance = 0;

Lastfm::Lastfm(QObject *parent) :
    QObject(parent),
    m_nam(0)
{
    if (!lastfmInstance) {
        lastfmInstance = this;
    }
}

Lastfm::~Lastfm() {}

Lastfm* Lastfm::instance() {
    return lastfmInstance;
}

void Lastfm::setBusy(bool isBusy, const QString &message, int numberOfOperations) {
    if (isBusy != this->busy()) {
        m_busy = isBusy;

        if (isBusy) {
            this->setCancelled(false);
            emit busy(message, numberOfOperations);
        }
        else if (!this->cancelled()) {
            emit busyProgressChanged(numberOfOperations);
        }

        emit busyChanged(isBusy);
    }
}

void Lastfm::cancelCurrentOperation() {
    this->setCancelled(true);
    this->setBusy(false);
    emit currentOperationCancelled();
}

QString Lastfm::getMd5Hash(const QString &value) {
    QByteArray hash(QCryptographicHash::hash(value.toUtf8(), QCryptographicHash::Md5));
    QString signature(hash.toHex());

    return signature;
}

QString Lastfm::getSignature(const QMap<QString, QString> &params, const QString &secret) {
    QString signature;
    QMapIterator<QString, QString> iterator(params);

    while (iterator.hasNext()) {
        iterator.next();
        signature.append(iterator.key());
        signature.append(iterator.value());
    }

    signature.append(secret);

    return this->getMd5Hash(signature);
}

QByteArray Lastfm::getPostData(const QMap<QString, QString> &params) {
    QString postData;
    postData.append("method=" + params.value("method"));
    QMapIterator<QString, QString> iterator(params);

    while (iterator.hasNext()) {
        iterator.next();

        if(iterator.key() != "method") {
            postData.append("&" + iterator.key() + "=" + iterator.value().toUtf8().toPercentEncoding());
        }
    }

    return postData.toAscii();
}

QString Lastfm::getAuthToken(const QString &user, const QString &pass) {
    QString hashPass = this->getMd5Hash(pass);
    QString userHash = user.toLower() + hashPass;

    return this->getMd5Hash(userHash);
}

void Lastfm::signIn(const QString &user, const QString &pass) {
    this->setBusy(true, tr("Signing in"));
    QString authToken = this->getAuthToken(user, pass);
    QMap<QString, QString> params;
    params.insert("username", user);
    params.insert("api_key", API_KEY);
    params.insert("authToken", authToken);
    params.insert("method", "auth.getMobileSession");
    params.insert("api_sig", this->getSignature(params, API_SECRET));
    QNetworkRequest request(QUrl("http://ws.audioscrobbler.com/2.0/?"));
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    request.setRawHeader("User-Agent", QString("MusiKloud/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->post(request, this->getPostData(params));
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkAccessToken()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void Lastfm::checkAccessToken() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QByteArray response(reply->readAll());
    QXmlStreamReader xmlReader;
    xmlReader.addData(response);
    QString status;
    QString user;
    QString key;
    bool keyFound = false;

    while (!((keyFound) || (xmlReader.atEnd()))) {
        xmlReader.readNext();
        if (xmlReader.name() == "lfm") {
            status = xmlReader.attributes().value("status").toString();
        }
        else {
            if (status == "ok") {
                if (xmlReader.name() == "name") {
                    user = xmlReader.readElementText();
                }
                if (xmlReader.name() == "key") {
                    key = xmlReader.readElementText();
                }
                if (!((user.isEmpty()) || (key.isEmpty()))) {
                    keyFound = true;
                }
            }
        }
    }

    if ((status == "ok") && (keyFound)) {
        emit signedIn(user, key);
        emit alert(tr("You are signed in to your Last.fm account"));
    }
    else {
        emit error(reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString());
    }

    reply->deleteLater();
}

void Lastfm::setCredentials(const QString &user, const QString &token) {
    m_username = user;
    m_token = token;
    emit usernameChanged();
    emit userSignedInChanged(this->userSignedIn());
}

void Lastfm::postRequest(const QUrl &url, const QByteArray &data) {
    QNetworkRequest request(url);
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    request.setRawHeader("User-Agent", QString("MusiKloud/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->post(request, data);
    this->connect(reply, SIGNAL(finished()), this, SLOT(postFinished()));
}

void Lastfm::postFinished() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    switch (statusCode) {
    case 200:
    case 201:
        emit postSuccessful();
        break;
    case 403:
        emit signedIn(QString(), QString());
        break;
    default:
        emit error(reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString());
        break;
    }

    this->disconnect(this, SIGNAL(postSuccessful()), 0, 0);
    reply->deleteLater();
}

void Lastfm::scrobbleTrack(const QString &artist, const QString &title) {
    if ((!this->userSignedIn()) || (artist.isEmpty()) || (title.isEmpty())) {
        emit error(tr("Cannot scrobble track"));
        return;
    }

    QUrl url("http://ws.audioscrobbler.com/2.0/?");
    QMap<QString, QString> params;
    params.insert("api_key", API_KEY);
    params.insert("username", m_username);
    params.insert("sk", m_token);
    params.insert("method", "track.scrobble");
    params.insert("timestamp", QString::number(QDateTime::currentMSecsSinceEpoch() / 1000));
    params.insert("artist", artist);
    params.insert("track", title);
    params.insert("api_sig", this->getSignature(params, API_SECRET));
    this->postRequest(url, this->getPostData(params));
    this->connect(this, SIGNAL(postSuccessful()), this, SLOT(onTrackScrobbled()));
}

void Lastfm::onTrackScrobbled() {
    emit alert(tr("Track scrobbled to Last.fm"));
}
