/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   Mupen64plus - audio.h                                                 *
 *   Mupen64Plus homepage: http://code.google.com/p/mupen64plus/           *
 *   Copyright (C) 2008 Richard42                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <stdio.h>
#include <stdint.h>
#include <math.h>

#include <pulse/simple.h>
#include <pulse/error.h>

#include "Audio_1.1.h"

#define MODULE_NAME "pa64"
/* This sets default frequency what is used if rom doesn't want to change it.
Probably only game that needs this is Zelda: Ocarina Of Time Master Quest */
#define DEFAULT_FREQUENCY 33600

/** Read header for type definition */
static AUDIO_INFO AudioInfo;
/** Audio frequency, this is usually obtained from the game, but for compatibility we set default value */
static int GameFreq = DEFAULT_FREQUENCY;
/** Connection to pulseaudio server */
static pa_simple *pa = NULL; 

static void CloseAudio()
{
	if (pa) {
		printf("[%s]: Closing pulse connection\n", MODULE_NAME);
		pa_simple_free(pa);
		pa = NULL;
	}
	GameFreq = DEFAULT_FREQUENCY;
}

static void InitializeAudio(int freq)
{
	pa_sample_spec ss = {
		.format = PA_SAMPLE_S16LE,
		.rate = freq,
		.channels = 2
	};
	int error;

	CloseAudio();

	GameFreq = freq;

	printf("[%s]: Starting pulse connection with rate: %d Hz\n", 
	       MODULE_NAME, freq);
	
	pa = pa_simple_new(NULL, MODULE_NAME, PA_STREAM_PLAYBACK, NULL,
	                   "pa64 Playback", &ss, NULL, NULL, &error);
	if (!pa) {
		fprintf(stderr, "[%s]: Error connecting to pulse: %s\n", 
		        MODULE_NAME, pa_strerror(error));
		return;
	}
}

EXPORT void CALL AiDacrateChanged( int SystemType )
{
    int f = GameFreq;
    switch (SystemType)
    {
        case SYSTEM_NTSC:
            f = 48681812 / (*AudioInfo.AI_DACRATE_REG + 1);
            break;
        case SYSTEM_PAL:
            f = 49656530 / (*AudioInfo.AI_DACRATE_REG + 1);
            break;
        case SYSTEM_MPAL:
            f = 48628316 / (*AudioInfo.AI_DACRATE_REG + 1);
            break;
    }
    /* Round sample rate a bit */
    InitializeAudio(roundf(f / 100.0f) * 100);
}

EXPORT void CALL AiLenChanged(void)
{
	size_t len = *AudioInfo.AI_LEN_REG;
	uint8_t *buf = (uint8_t *)(AudioInfo.RDRAM +
	                          (*AudioInfo.AI_DRAM_ADDR_REG & 0xFFFFFF));
	int error;

	if (!pa) return;

	if (pa_simple_write(pa, buf, len, &error) < 0) {
		fprintf(stderr, "[%s]: Error writing to pulse: %s\n", 
		        MODULE_NAME, pa_strerror(error));
	}
}

EXPORT DWORD CALL
AiReadLength( void )
{
    return 0;
}

EXPORT void CALL
AiUpdate( BOOL Wait )
{
}

EXPORT void CALL
CloseDLL( void )
{
	CloseAudio();
}

EXPORT void CALL
DllAbout( HWND hParent )
{
    printf ("Simple-API Pulseaudio plugin\n" );
}

EXPORT void CALL
DllConfig ( HWND hParent )
{
}

EXPORT void CALL
DllTest ( HWND hParent )
{
}

EXPORT void CALL
GetDllInfo( PLUGIN_INFO * PluginInfo )
{
    PluginInfo->Version = 0x0102;
    PluginInfo->Type    = PLUGIN_TYPE_AUDIO;
    sprintf(PluginInfo->Name, "pa64");
    PluginInfo->NormalMemory  = TRUE;
    PluginInfo->MemoryBswaped = TRUE;
}

EXPORT BOOL CALL
InitiateAudio( AUDIO_INFO Audio_Info )
{
    AudioInfo = Audio_Info;
    return TRUE;
}

EXPORT void CALL RomOpen(void)
{
}

EXPORT void CALL
RomClosed( void )
{
}

EXPORT void CALL
ProcessAList( void )
{
}

