#include "channelsform.h"
#include "ui_channelsform.h"
#include "ui_playercontrols.h"
#include "ui_channelplayer.h"
#include "channelplayer.h"
#include "channelsreader.h"
#include "constants.h"
#include "playlistreader.h"
#include "songsreader.h"

#include <QUrl>
#include <QWebFrame>
#include <QHostInfo>
#include <QDateTime>
#include <QSignalMapper>
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QTimer>

// qt-mobility
#include <QMediaPlaylist>

ChannelsForm::ChannelsForm(QWidget *parent) :
    QWidget(parent),
    ui(new Ui::ChannelsForm),
    channelsFeedReply(0)
{
    ui->setupUi(this);

    ui->channelInfo->setHtml(tr("Welcome to mSoma!"));

    ui->playWidget->setVisible(false);

    ui->channelPlayer->setVisible(false);

    ui->progressBar->hide();

    connect(ui->updateButton, SIGNAL(clicked()), this, SLOT(updateChannels()));

    //connect(ui->channelsWidget, SIGNAL(currentRowChanged(int)), this, SLOT(updateChannelInfo(int)));
    connect(ui->channelsWidget, SIGNAL(itemActivated(QListWidgetItem*)), this, SLOT(updateChannelInfo(QListWidgetItem*)));
    connect(ui->channelsWidget, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(updateChannelInfo(QListWidgetItem*)));

    connect(ui->abortButton, SIGNAL(clicked()), this, SLOT(abortFetching()));

    // maps for playlist buttons
    signalMapper = new QSignalMapper(this);
    connect(ui->mp3FastButton, SIGNAL(clicked()), signalMapper, SLOT(map()));
    connect(ui->mp3SlowButton, SIGNAL(clicked()), signalMapper, SLOT(map()));
    connect(ui->aacpFastButton, SIGNAL(clicked()), signalMapper, SLOT(map()));
    connect(ui->aacpSlowButton, SIGNAL(clicked()), signalMapper, SLOT(map()));
    connect(signalMapper, SIGNAL(mapped(const QString &)), this, SLOT(playPlaylist(const QString &)));

    // maps for the player
    connect(ui->channelPlayer, SIGNAL(stopPlayer()), this, SLOT(stopPlaylist()));

    // Set autorotation on
    #if defined(Q_WS_MAEMO_5)
    setAttribute(Qt::WA_Maemo5AutoOrientation, true);
    #endif

    channelsFeedNetworkReader = new QNetworkAccessManager(this);
    connect(channelsFeedNetworkReader, SIGNAL(finished(QNetworkReply*)), this, SLOT(finishReadingChannelsFeed(QNetworkReply*)));
}

ChannelsForm::~ChannelsForm()
{
    delete ui;
}

bool ChannelsForm::isShowRecentlyPlayedSongs() const
{
    return ui->channelPlayer->isShowRecentlyPlayedSongs();
}

void ChannelsForm::setShowRecentlyPlayedSongs(bool b)
{
    ui->channelPlayer->setShowRecentlyPlayedSongs(b);
}

void ChannelsForm::updateChannels()
{
    // disable button while updating
    startUpdating();

    channelsFeedData.clear();
    ui->channelsWidget->clear();

    QUrl url(channelsFeedUrl);

    // clear the channel info that we'll use for displaying status
    ui->channelInfo->setHtml("");

    if (url.isValid()) {
        QHostInfo info = QHostInfo::fromName(url.host());
        if (info.error() == QHostInfo::NoError) {
            startUpdating();
            channelsFeedReply = channelsFeedNetworkReader->get(QNetworkRequest(url));
            connect(channelsFeedReply, SIGNAL(downloadProgress(qint64,qint64)), this, SLOT(channelsFeedDownloadProgress(qint64,qint64)));

            setStatus(tr("fetching from %1 ...").arg(channelsFeedUrl));
        } else {
            setErrorStatus(tr("Host resolution error: %1").arg(info.errorString()));
            endUpdating();
        }
    } else {
        setErrorStatus(tr("Invalid URL: %1").arg(url.toString()));
        endUpdating();
    }
}

void ChannelsForm::finishReadingChannelsFeed(QNetworkReply *reply)
{
    channelsFeedData = reply->readAll();
    //qDebug() << "channels data: " << data;

    if (reply->error() != QNetworkReply::NoError) {
        qDebug() << "error: " << reply->errorString();
        setErrorStatus(reply->errorString());
    }

    endUpdating();

    // it's our responsibility to delete it, but after the slot!
    reply->deleteLater();

    if (channelsFeedData.size() == 0 || reply->error() != QNetworkReply::NoError)
        return;

    setStatus(tr("finished fetching from %1").arg(channelsFeedUrl));

    ui->channelsWidget->clear();

    ChannelsReader channelsReader;
    channelsReader.addData(channelsFeedData);
    // replace channels
    channels = channelsReader.readChannels();
    for (MSomaChannels::ConstIterator it = channels.begin(); it != channels.end(); ++it)
    {
        ui->channelsWidget->addItem(it->values["title"]);
    }
}

void ChannelsForm::channelsFeedDownloadProgress(qint64 bytesReceived, qint64 bytesTotal)
{
    //setStatus(tr("download progress %1 of %2").arg(bytesReceived).arg(bytesTotal));
    ui->progressBar->setMinimum(0);
    ui->progressBar->setMaximum(bytesTotal);
    ui->progressBar->setValue(bytesReceived);
}

void ChannelsForm::fillChannelList()
{
    if (channelsFeedData.size() == 0)
        return;

    ui->channelsWidget->clear();

    ChannelsReader channelsReader;
    channelsReader.addData(channelsFeedData);
    // replace channels
    channels = channelsReader.readChannels();
    for (MSomaChannels::ConstIterator it = channels.begin(); it != channels.end(); ++it)
    {
        ui->channelsWidget->addItem(it->values["title"]);
    }
}

void ChannelsForm::abortFetching()
{
    if (channelsFeedReply)
        channelsFeedReply->abort();
    setStatus(tr("Aborted"));
}

void ChannelsForm::updateStatus(const QString &s)
{
    ui->channelInfo->setHtml
            (ui->channelInfo->page()->mainFrame()->toHtml() +
             "<br>" + s);
}

void ChannelsForm::setStatus(const QString &s)
{
    updateStatus(s);
}

void ChannelsForm::setErrorStatus(const QString &s)
{
    updateStatus("<b>" + s + "</b>");
}

void ChannelsForm::setErrorStatus(const QString &s, int error)
{
    setErrorStatus(s + " (" + QString::number(error) + ")");
}

void ChannelsForm::startUpdating()
{
    ui->progressBar->reset();
    ui->progressBar->show();
    // setting min and max to 0 shows a busy indicator
    ui->progressBar->setMinimum(0);
    ui->progressBar->setMaximum(0);
    ui->updateButton->setEnabled(false);
    ui->abortButton->setEnabled(true);
}

void ChannelsForm::endUpdating()
{
    ui->updateButton->setEnabled(true);
    ui->abortButton->setEnabled(false);
    // after 1 second hide the progress bar
    QTimer::singleShot(1000, ui->progressBar, SLOT(hide()));
}

bool channelInfoAlreadyShown(const QString &value)
{
    return (value == "image" || value == "title");
}

void ChannelsForm::updateChannelInfo(QListWidgetItem *item)
{
    updateChannelInfo(item->listWidget()->row(item));
}

void ChannelsForm::updateChannelInfo(int position)
{
    if (position < 0)
        return;

    if (position < channels.size()) {
        //ui->channelInfo->clear();
        QString info;

        channel = channels[position];

        info += "<img src=\""
                +
                channel.values["image"]
                + "\"><br>" +
                "<b>title</b>: " +
                channel.values["title"] + "<br>\n";
                ;

        for (MSomaChannel::ValueIterator value = channel.values.begin(); value != channel.values.end(); ++value)
        {
            if (value.key() == "updated") {
                info += "<b>" + value.key() + "</b>: " + QDateTime::fromTime_t((value.value()).toUInt()).toString() + "<br>\n";
            } else if (!channelInfoAlreadyShown(value.key())) {
                info += "<b>" + value.key() + "</b>: " + value.value() + "<br>\n";
            }
        }

        ui->channelInfo->setHtml(info);

        enablePlayButtons(channel);
    }
}

void ChannelsForm::enablePlayButtons(const MSomaChannel &channel)
{
    disablePlayButtons();
    ui->playWidget->setVisible(true);

    for (MSomaChannel::PlayListIterator playlist = channel.playlists.begin(); playlist != channel.playlists.end(); ++playlist)
    {
        if (playlist->name == "fastpls")
        {
            if (playlist->format == "mp3") {
                ui->mp3FastButton->setVisible(true);
                signalMapper->setMapping(ui->mp3FastButton, playlist->url);
            } else if (playlist->format == "aacp") {
                ui->aacpFastButton->setVisible(true);
                signalMapper->setMapping(ui->aacpFastButton, playlist->url);
            }
        }
        else if (playlist->name == "slowpls")
        {
            if (playlist->format == "mp3") {
                ui->mp3SlowButton->setVisible(true);
                signalMapper->setMapping(ui->mp3SlowButton, playlist->url);
            } else if (playlist->format == "aacp") {
                ui->aacpSlowButton->setVisible(true);
                signalMapper->setMapping(ui->aacpSlowButton, playlist->url);
            }
        }
    }
}

void ChannelsForm::disablePlayButtons()
{
    ui->playWidget->setVisible(false);
    ui->mp3FastButton->setVisible(false);
    ui->mp3SlowButton->setVisible(false);
    ui->aacpFastButton->setVisible(false);
    ui->aacpSlowButton->setVisible(false);
}

void ChannelsForm::playPlaylist(const QString &url)
{
    ui->channelPlayer->setCurrentChannel(channel);
    ui->channelPlayer->show();
    ui->channelPlayer->playPlaylist(url);
}

void ChannelsForm::stopPlaylist()
{
    ui->channelPlayer->hide();
}

void ChannelsForm::readSettings(QSettings &settings)
{
    channelsFeedUrl = settings.value("feedurl", CHANNELS_FEED_URL).toString();
    ui->splitter->restoreState(settings.value("splitterSizes").toByteArray());
    ui->channelPlayer->setShowRecentlyPlayedSongs(settings.value("showRecentlyPlayedSongs", false).toBool());
    channelsFeedData = settings.value("channelFeed").toByteArray();
}

void ChannelsForm::writeSettings(QSettings &settings)
{
    settings.setValue("feedurl", channelsFeedUrl);
    settings.setValue("splitterSizes", ui->splitter->saveState());
    settings.setValue("showRecentlyPlayedSongs", ui->channelPlayer->isShowRecentlyPlayedSongs());
    settings.setValue("channelFeed", channelsFeedData);
}
