/********************
 * Callback functions
 ********************/
/*
 * Include declarations
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <glob.h>

#include <libosso.h>
#include <hildon/hildon.h>
#include <hildon/hildon-defines.h>
#include <gtk/gtk.h>
#include <gdk/gdkpixbuf.h>
#include <gdk/gdkx.h>
#include <X11/Xatom.h>
#include <glib/gprintf.h>
#include <glib/gfileutils.h>
#include <gconf/gconf-client.h>

#include <libraw/libraw.h>

#include "appdata.h"
#include "interface.h"
#include "callbacks.h"
#include "helpers.h"
#ifdef HAVE_CONFIG_H
#include "../config.h"
#endif

/*
 * Load image from filename
 */
void load_image(app_data_t *myapp, gchar *imgfname) {
	GtkWidget *errdialog = NULL;
	GdkPixbufLoader *imgloader = NULL;
	GdkPixbuf *tmpimgbuffer = NULL;
	libraw_data_t *rawdata = NULL;
	libraw_processed_image_t *imgdata = NULL;
	int rescode = LIBRAW_UNSPECIFIED_ERROR;
	gboolean imgerr = FALSE, imgthumb = TRUE;
	gint shutternum = 0;
	gchar *shutter = NULL;
	const gchar *paramformat = "<span size=\"medium\" face=\"arial\">Filename: <b>%s</b>\nCamera model: <b>%s %s</b>\nShutter speed: <b>%s s</b>\nAperture: <b>f/%-.1f</b>\nISO: <b>%-.0f</b>\nFocal length: <b>%-.0f mm</b></span>";
	GError *error = NULL;

	g_free(myapp->currfname);
	myapp->currfname = g_strndup(g_path_get_basename(imgfname),MAXFNLEN);
	gtk_window_set_title(GTK_WINDOW (myapp->mainwin),g_strconcat(g_get_application_name()," - ",myapp->currfname,NULL));
	hildon_gtk_window_set_progress_indicator(GTK_WINDOW (myapp->mainwin),1);
	if (gdk_window_get_state(gtk_widget_get_window(GTK_WIDGET (myapp->mainwin))) & GDK_WINDOW_STATE_FULLSCREEN) hildon_banner_show_informationf(GTK_WIDGET (myapp->mainwin),NULL,"Please wait while loading image:\n%s",myapp->currfname);
	while (gtk_events_pending()) gtk_main_iteration();
	rawdata = libraw_init(0);
	rawdata->params.use_camera_wb = 1;
	rescode = libraw_open_file(rawdata,imgfname);
	if (rescode == LIBRAW_SUCCESS) {
		/* First try to retrieve embedded thumbnail */
		rescode = libraw_unpack_thumb(rawdata);
		/* DEBUG */
		/* g_print("DEBUG: Thumbnail extraction result: %d, format: %d, lenghth: %d, colors: %d\n",rescode,rawdata->thumbnail.tformat,rawdata->thumbnail.tlength,rawdata->thumbnail.tcolors); */
		if ((rescode == LIBRAW_SUCCESS) || ((rescode == LIBRAW_UNSUPPORTED_THUMBNAIL) && (rawdata->thumbnail.thumb != NULL))) {
			switch (rawdata->thumbnail.tformat) {
				case LIBRAW_THUMBNAIL_JPEG:
					imgloader = gdk_pixbuf_loader_new();
					if (!gdk_pixbuf_loader_write(imgloader,(guchar *) rawdata->thumbnail.thumb,(gsize) rawdata->thumbnail.tlength,&error)) {
						imgerr = TRUE;
						rescode = LIBRAW_UNSPECIFIED_ERROR;
					} else tmpimgbuffer = gdk_pixbuf_loader_get_pixbuf(imgloader);
					gdk_pixbuf_loader_close(imgloader,NULL);
					break;
				case LIBRAW_THUMBNAIL_BITMAP:
					tmpimgbuffer = gdk_pixbuf_new_from_data((guchar *) rawdata->thumbnail.thumb,GDK_COLORSPACE_RGB,FALSE,8,(int) rawdata->thumbnail.twidth,(int) rawdata->thumbnail.theight,(rawdata->thumbnail.tlength / rawdata->thumbnail.theight),NULL,NULL);
					break;
				default:
					break;
			}
			if (tmpimgbuffer == NULL) {
				imgerr = TRUE;
				rescode = LIBRAW_UNSPECIFIED_ERROR;
			}
		} else if ((rescode == LIBRAW_NO_THUMBNAIL) || ((rescode == LIBRAW_UNSUPPORTED_THUMBNAIL) && (rawdata->thumbnail.thumb == NULL))) {
			/* Then try to process RAW data */
			imgthumb = FALSE;
			rescode = libraw_unpack(rawdata);
			if (rescode == LIBRAW_SUCCESS) {
				rescode = libraw_dcraw_process(rawdata);
				if (rescode == LIBRAW_SUCCESS) {
					imgdata = libraw_dcraw_make_mem_image(rawdata,&rescode);
					if ((rescode == LIBRAW_SUCCESS) & (imgdata != NULL)) {
						if ((imgdata->type != LIBRAW_IMAGE_BITMAP) || !(tmpimgbuffer = gdk_pixbuf_new_from_data((guchar *) imgdata->data,GDK_COLORSPACE_RGB,FALSE,imgdata->bits,(int) imgdata->width,(int) imgdata->height,(imgdata->data_size / imgdata->height),NULL,NULL))) {
							imgerr = TRUE;
							rescode = LIBRAW_UNSPECIFIED_ERROR;
						}
					} else imgerr = TRUE;
				} else imgerr = TRUE;
			} else imgerr = TRUE;
		} else imgerr = TRUE;
		if (!imgerr) {
			if (imgthumb) {
				switch(rawdata->sizes.flip) {
					case 3:
						myapp->imgbuffer = gdk_pixbuf_rotate_simple(tmpimgbuffer,GDK_PIXBUF_ROTATE_UPSIDEDOWN);
						break;
					case 5:
						myapp->imgbuffer = gdk_pixbuf_rotate_simple(tmpimgbuffer,GDK_PIXBUF_ROTATE_COUNTERCLOCKWISE);
						break;
					case 6:
						myapp->imgbuffer = gdk_pixbuf_rotate_simple(tmpimgbuffer,GDK_PIXBUF_ROTATE_CLOCKWISE);
						break;
					case 0:
					default:
						myapp->imgbuffer = gdk_pixbuf_copy(tmpimgbuffer);
						break;
				}
			} else {
				myapp->imgbuffer = gdk_pixbuf_copy(tmpimgbuffer);
			}
			g_object_unref(tmpimgbuffer);
			if (imgdata) g_free(imgdata);
		}
	} else imgerr = TRUE;
	while (gtk_events_pending()) gtk_main_iteration();
	hildon_gtk_window_set_progress_indicator(GTK_WINDOW (myapp->mainwin),0);
	g_free(myapp->imgparam);
	if (!imgerr) {
		shutternum = (gint) ((rawdata->other.shutter < 1) ? round(1 / rawdata->other.shutter) : round(rawdata->other.shutter));
		shutter = g_strdup_printf(((rawdata->other.shutter < 1) ? "1/%d" : "%d"),shutternum);
		myapp->imgparam = g_strdup_printf(paramformat,myapp->currfname,rawdata->idata.make,rawdata->idata.model,shutter,rawdata->other.aperture,rawdata->other.iso_speed,rawdata->other.focal_len);
		g_free(shutter);
		myapp->zoomlevel = FIT_SCREEN;
		scale_image(GTK_WIDGET (myapp->mainwin),myapp);
		g_object_set(G_OBJECT (myapp->imgparamwin),"markup",myapp->imgparam,NULL);
	} else {
		g_object_unref(myapp->imgbuffer);
		gtk_image_set_from_pixbuf(GTK_IMAGE (myapp->image),NULL);
		g_object_set(G_OBJECT (myapp->imgparamwin),"markup",NULL,NULL);
		imgparam_window_hide(IMGPARAM_WINDOW (myapp->imgparamwin));
		errdialog = gtk_message_dialog_new(GTK_WINDOW (myapp->mainwin),
											GTK_DIALOG_DESTROY_WITH_PARENT,
											GTK_MESSAGE_ERROR,
											GTK_BUTTONS_CLOSE,
											"There was an error while loading \"%s\"\n\nError message: \"%s\"",
											g_path_get_basename(imgfname),
											libraw_strerror(rescode));
		gtk_window_set_title(GTK_WINDOW (errdialog),"Error loading image");
		gtk_dialog_run(GTK_DIALOG (errdialog));
		gtk_widget_destroy(errdialog);
	}
	libraw_close(rawdata);
}

/*
 * Compute dimensions of scaled image according to selected zoom factor
 */
void scaled_image_compute_size(app_data_t *myapp,guint *scaledwidth,guint *scaledheight) {
	gint dispwidth, dispheight, imgwidth, imgheight;
	gdouble dispratio, imgratio;

	/* Get actual image widget dimensions */
	gtk_window_get_size(GTK_WINDOW (myapp->mainwin),&dispwidth,&dispheight);
	dispratio = (gdouble) dispwidth / (gdouble) dispheight;
	imgwidth = gdk_pixbuf_get_width(myapp->imgbuffer);
	imgheight = gdk_pixbuf_get_height(myapp->imgbuffer);
	imgratio = (gdouble) imgwidth / (gdouble) imgheight;
	*scaledwidth = *scaledheight = 0;
	/* Compute scaled image size */
	if ((imgwidth > dispwidth) || (imgheight > dispheight)) {
		switch (myapp->zoomlevel) {
			case FIT_SCREEN:
				if (imgratio >= dispratio) {
					*scaledwidth = dispwidth;
					*scaledheight = (guint) round((double) dispwidth * (1 / imgratio));
				} else {
					*scaledwidth = (guint) round((double) (dispheight * imgratio));
					*scaledheight = dispheight;
				}
				break;
			case ONE_THIRD_RESOLUTION:
				*scaledwidth = (guint) round((double) imgwidth * 0.33);
				*scaledheight = (guint) round((double) imgheight * 0.33);
				break;
			case HALF_RESOLUTION:
				*scaledwidth = (guint) round((double) imgwidth * 0.5);
				*scaledheight = (guint) round((double) imgheight * 0.5);
				break;
			case FULL_RESOLUTION:
			default:
				break;
		}
	}
	/* DEBUG */
	/* g_print("\n\nOriginal dim.: %dx%d (%f) - Window dim.: %dx%d (%f) - Scaled dim.: %dx%d\n\n",imgwidth,imgheight,imgratio,dispwidth,dispheight,dispratio,*scaledwidth,*scaledheight); */
}

/*
 * Scale current image according to selected zoom factor
 */
gboolean scale_image(GtkWidget *callerobj,app_data_t *myapp) {
	gchar *zoomtext = NULL;
	GdkPixbuf *tmpbuffer = NULL;
	guint scaledwidth, scaledheight;
	GdkInterpType scalingmode = GDK_INTERP_BILINEAR;

	/* DEBUG */
	/* g_print("Scale image with buffer: %d (%d)\n",myapp->imgbuffer,G_IS_OBJECT (G_OBJECT (myapp->imgbuffer))); */
	if (myapp->imgbuffer != NULL) {
		hildon_gtk_window_set_progress_indicator(GTK_WINDOW (myapp->mainwin),1);
		if (gdk_window_get_state(gtk_widget_get_window(GTK_WIDGET (myapp->mainwin))) & GDK_WINDOW_STATE_FULLSCREEN) hildon_banner_show_informationf(GTK_WIDGET (myapp->mainwin),NULL,"Scaling image %s",zoomtext);
		while (gtk_events_pending()) gtk_main_iteration();
		/* Eventually scale image */
		scaled_image_compute_size(myapp,&scaledwidth,&scaledheight);
		if ((scaledwidth > 0) & (scaledheight > 0)) {
			switch (myapp->zoomlevel) {
				case FIT_SCREEN:
					/* scalingmode = GDK_INTERP_NEAREST; */
					zoomtext = " (Fit Scr.)";
					break;
				case ONE_THIRD_RESOLUTION:
					zoomtext = " (33%)";
					break;
				case HALF_RESOLUTION:
					zoomtext = " (50%)";
					break;
				case FULL_RESOLUTION:
					zoomtext = " (Full Res.)";
					break;
				default:
					zoomtext = "";
					break;
			}
			tmpbuffer = gdk_pixbuf_scale_simple(myapp->imgbuffer,scaledwidth,scaledheight,scalingmode);
		} else zoomtext = " (Full Res.)";
		gtk_image_set_from_pixbuf(GTK_IMAGE (myapp->image), (tmpbuffer ? tmpbuffer : myapp->imgbuffer));
		while (gtk_events_pending()) gtk_main_iteration();
		hildon_gtk_window_set_progress_indicator(GTK_WINDOW (myapp->mainwin),0);
		gtk_window_set_title(GTK_WINDOW (myapp->mainwin),g_strconcat(g_get_application_name()," - ",myapp->currfname,zoomtext,NULL));
		if (tmpbuffer) g_object_unref(tmpbuffer);
	}
	return TRUE;
}

/*
 * Store image file names in current directory into application's data structure
 */
void store_filenames_in_dir(app_data_t *myapp) {
	gchar *tmpstr;
	glob_t allimages;
	guint i;
	GList *li = NULL;

	/* First free the allocated memory in the list */
	if (myapp->allimgindir != NULL) {
		for(li = myapp->allimgindir;li != NULL;li = g_list_next(li))
			g_free(li->data);
		g_list_free(li);
		g_list_free(myapp->allimgindir);
		myapp->allimgindir = NULL;
	}
	/* Then search for other image files */
	if (myapp->currdir != NULL) {
		tmpstr = g_strconcat(myapp->currdir,"/*.{",g_strjoinv(",",(gchar**) myapp->rawext),"}",NULL);
		if (glob(tmpstr,GLOB_ERR | GLOB_BRACE,NULL,&allimages) == 0) {
			for(i = 0;i < allimages.gl_pathc;i++) {
				myapp->allimgindir = g_list_prepend(myapp->allimgindir,g_strndup(g_path_get_basename((gchar*) allimages.gl_pathv[i]),MAXFNLEN));
			}
			if (myapp->allimgindir != NULL) myapp->allimgindir = g_list_reverse(myapp->allimgindir);
		}
		if (allimages.gl_pathc <= 1) {
			arrow_button_disable(myapp->prevarrowbut);
			arrow_button_disable(myapp->nextarrowbut);
		} else {
			arrow_button_enable(myapp->prevarrowbut);
			arrow_button_enable(myapp->nextarrowbut);
		}
		globfree(&allimages);
		g_free(tmpstr);
	}
	/* DEBUG */
	/*for(li = myapp->allimgindir;li != NULL;li = g_list_next(li))
		g_print("File in list: %s\n",li->data);
	g_list_free(li);*/
}

/*
 * Ungrab volume keys
 */
void ungrab_volume_keys(GtkWidget *widget) {
    /* Tell maemo-status-volume daemon to ungrab keys */
    unsigned long val = 1; /* ungrab, use 0 to grab */
    Atom atom;
    GdkDisplay *display = NULL;
    display = gtk_widget_get_display(widget);
    atom = gdk_x11_get_xatom_by_name_for_display(display,"_HILDON_ZOOM_KEY_ATOM");
    XChangeProperty (GDK_DISPLAY_XDISPLAY (display),
                     GDK_WINDOW_XID (GDK_WINDOW (widget->window)),atom,XA_INTEGER,32,
                     PropModeReplace,(unsigned char *)&val,1);
}

/*
 * Store configuration values into GConf
 */
void save_configuration(app_data_t *myapp) {
	GConfClient* gcClient = NULL;

	gcClient = gconf_client_get_default();
	/* Store the values. */
	if (GCONF_IS_CLIENT (gcClient)) {
		if (!gconf_client_set_string(gcClient,g_strconcat(GC_ROOT,"currdir",NULL),(const gchar *)myapp->currdir,NULL)) {
			g_printerr("WARNING: Failed to set %scurrdir to %s\n",GC_ROOT,myapp->currdir);
		}
		if (!gconf_client_set_string(gcClient,g_strconcat(GC_ROOT,"currfname",NULL),(const gchar *)myapp->currfname,NULL)) {
			g_printerr("WARNING: Failed to set %scurrfname to %s\n",GC_ROOT,myapp->currfname);
		}
		if (!gconf_client_set_int(gcClient,g_strconcat(GC_ROOT,"currfindex",NULL),myapp->currfindex,NULL)) {
			g_printerr("WARNING: Failed to set %scurrfindex to %d\n",GC_ROOT,myapp->currfindex);
		}
		if (!gconf_client_set_int(gcClient,g_strconcat(GC_ROOT,"zoomlevel",NULL),myapp->zoomlevel,NULL)) {
			g_printerr("WARNING: Failed to set %szoomlevel to %d\n",GC_ROOT,myapp->zoomlevel);
		}
		if (!gconf_client_set_bool(gcClient,g_strconcat(GC_ROOT,"showimgparam",NULL),myapp->showimgparam,NULL)) {
			g_printerr("WARNING: Failed to set %sshowimgparam to %d\n",GC_ROOT,myapp->showimgparam);
		}
		g_object_unref(gcClient);
	} else g_printerr("WARNING: Failed to get default GConf client.\n");
}

/*
 * Restore configuration from GConf values
 */
void load_configuration(app_data_t *myapp) {
	GConfClient* gcClient = NULL;

	gcClient = gconf_client_get_default();
	/* Load the values. */
	if (GCONF_IS_CLIENT (gcClient)) {
		myapp->currdir = gconf_client_get_string(gcClient,g_strconcat(GC_ROOT,"currdir",NULL),NULL);
		myapp->currfname = gconf_client_get_string(gcClient,g_strconcat(GC_ROOT,"currfname",NULL),NULL);
		myapp->currfindex = gconf_client_get_int(gcClient,g_strconcat(GC_ROOT,"currfindex",NULL),NULL);
		myapp->zoomlevel = gconf_client_get_int(gcClient,g_strconcat(GC_ROOT,"zoomlevel",NULL),NULL);
		myapp->showimgparam = gconf_client_get_bool(gcClient,g_strconcat(GC_ROOT,"showimgparam",NULL),NULL);
		g_object_unref(gcClient);
	} else g_printerr("WARNING: Failed to get default GConf client.\n");
}
