#include "PrayTime.h"
#include <QDate>
#include <QDateTime>
#include "iostream"
#include <QString>
#include "time.h"
#include "math.h"

class Date {

        int day;
        int month;
        int year;
        time_t currentTime;
        time_t gmttime;
        struct tm * timeinfo;

public:

        Date() {
                time(&currentTime);
                timeinfo = localtime(&currentTime);
                day = timeinfo->tm_mday;
                month = timeinfo->tm_mon + 1;
                year = timeinfo->tm_year + 1900;
                currentTime = mktime(timeinfo);

        }

        Date(int y, int m, int d) {
                time(&currentTime);
                timeinfo = localtime(&currentTime);
                timeinfo->tm_mday = d;
                timeinfo->tm_mon = m - 1;
                timeinfo->tm_year = y - 1900;
                day = d;
                month = m;
                year = y;
                currentTime = mktime(timeinfo);
        }

        int getDay() {
                return day;
        }

        int getMonth() {
                return month;
        }

        int getYear() {
                return year;
        }

        time_t getTime() {
                return currentTime;
        }

        float getTimeZone() {
                tm* ptm = gmtime(&currentTime);
                gmttime = mktime(ptm);
                float diff = currentTime - gmttime;
                return diff / 3600;
        }
};

static QString timeNames[7];/* = {
 'Fajr',
 'Sunrise',
 'Dhuhr',
 'Asr',
 'Sunset',
 'Maghrib',
 'Isha'
 };
 */
QString InvalidTime;// = '-----';	 // The string used for invalid times


//---------------------- Global Variables --------------------


static int dhuhrMinutes;// = 0;		// minutes after mid-day for Dhuhr
enum JuristicMethod {shafii, hanafi};
enum LatitudeAdjustment {none, midNight, oneSeventh, angleBased};
enum TimeFormat {time24, time12, time12NS, floatFormat};

JuristicMethod asrJuristic;
LatitudeAdjustment adjustHighLats;
TimeFormat timeFormat;

float lat; // latitude
float lng; // longitude
float timeZone; // time-zone
float jDate; // Julian date

float resultArray[7];
QString stringArray[7];


//--------------------- Technical Settings --------------------


int numIterations;// = 1;		// number of iterations needed to compute times


//------------------- Calc Method Parameters --------------------


CalculationMethod calculationMethod;
        // convert a calendar date to julian date (second method)
float PrayTime::calcJD(int year, int month, int day) {
                float J1970 = 2440588.0;
                Date date(year, month, day);
                int ms = date.getTime(); // # of milliseconds since midnight Jan 1, 1970
              //  cout << "ms="<<ms << endl;
                float days = floor(ms / (1 * 60 * 60 * 24));
              //  cout << "days="<<days<<endl;
                return J1970 + days - 0.5;
        }

        //---------------------- Trigonometric Functions -----------------------

        // degree sin
        float PrayTime::dsin(float d) {
                return sin(dtr(d));
        }

        // degree cos
        float PrayTime::dcos(float d) {
                return cos(dtr(d));
        }

        // degree tan
        float PrayTime::dtan(float d) {
                return tan(dtr(d));
        }

        // degree arcsin
        float PrayTime::darcsin(float x) {
                return rtd(asin(x));
        }

        // degree arccos
        float PrayTime::darccos(float x) {
                return rtd(acos(x));
        }

        // degree arctan
        float PrayTime::darctan(float x) {
                return rtd(atan(x));
        }

        // degree arctan2
        float PrayTime::darctan2(float y, float x) {
                return rtd(atan2(y, x));
        }

        // degree arccot
        float PrayTime::darccot(float x) {
                return rtd(atan(1 / x));
        }

        // degree to radian
        float PrayTime::dtr(float d) {
                return (d * 3.14159265358979323846264338327950288) / 180.0;
        }

        // radian to degree
        float PrayTime::rtd(float r) {
                return (r * 180.0) / 3.14159265358979323846264338327950288;
        }

        // range reduce angle in degrees.
        float PrayTime::fixangle(float a) {
                a = a - 360.0 * (floor(a / 360.0));
                a = a < 0 ? a + 360.0 : a;
                return a;
        }

        // range reduce hours to 0..23
        float PrayTime::fixhour(float a) {
                a = a - 24.0 * (floor(a / 24.0));
                a = a < 0 ? a + 24.0 : a;
                return a;
        }

        // set the calculation method
        void PrayTime::setCalcMethod(int methodID) {
                if (methodID == 0) {
                    calculationMethod.setMakkah();
                }
                if (methodID == 1) {
                    calculationMethod.setIsna();
                }
                if (methodID == 2) {
                    calculationMethod.setMwl();
                }
                if (methodID == 3) {
                    calculationMethod.setEgypt();
                }
                if (methodID == 4) {
                    calculationMethod.setKarachi();
                }
                if (methodID == 5) {
                        calculationMethod.setTehran();
                }

        }

        // set the juristic method for Asr
        void PrayTime::setAsrMethod(int methodID) {
                if (methodID < 0 || methodID > 1)
                        return;
                if (methodID == 0){
                        asrJuristic = shafii;
                } else {
                        asrJuristic = hanafi;
                }
        }

        // set the minutes after mid-day for calculating Dhuhr
        void PrayTime::setDhuhrMinutes(int minutes) {
            dhuhrMinutes = minutes;
        }

        // set adjusting method for higher latitudes
        void PrayTime::setHighLatsMethod(int methodID) {
                if (methodID == 0){
                        adjustHighLats = none;
                } else if (methodID == 1){
                        adjustHighLats = midNight;
                } else if (methodID == 2){
                        adjustHighLats = oneSeventh;
                } else {
                        adjustHighLats = angleBased;
                }
        }

        // set the time format
        void PrayTime::setTimeFormat(int tFormat) {
                if (tFormat == 0){
                        timeFormat = time24;
                } else if (tFormat == 1){
                        timeFormat = time12;
                } else if (tFormat == 2){
                        timeFormat = time12NS;
                } else {
                        timeFormat = floatFormat;
                }
        }

        QString PrayTime::floatToString(float in) {
            QString ss;
            return ss.setNum(in, 'f', 0 );
        }

        // convert float hours to 24h format
        QString PrayTime::floatToTime24(float time) {
                //if (isNaN(time)) return InvalidTime;
                time = fixhour(time + 0.5 / 60); // add 0.5 minutes to round
                double hours = floor(time);
                double minutes = floor((time - hours) * 60);

                return twoDigitsFormat(hours) + ':' + twoDigitsFormat(minutes);
        }

        // convert float hours to 12h format
        QString PrayTime::floatToTime12(float time, bool noSuffix) {
                //if (isNaN(time)) return InvalidTime;
                time = fixhour(time + 0.5 / 60); // add 0.5 minutes to round
                int hours = (int) floor(time);
                int minutes = (int) floor((time - hours) * 60);
                QString suffix;
                if (hours >= 12) {
                        suffix = " pm";
                } else {
                        suffix = " am";
                }
                hours = (hours + 12 - 1) % 12 + 1;

                QString result = floatToString(hours) + ':' + twoDigitsFormat(minutes);

                if (!noSuffix) {
                        result = result + suffix;
                }
                return result;

        }

        // convert float hours to 12h format with no suffix
        QString PrayTime::floatToTime12NS(float time) {
                return floatToTime12(time, true);
        }

        //---------------------- Calculation Functions -----------------------

        // References:
        // http://www.ummah.net/astronomy/saltime
        // http://aa.usno.navy.mil/faq/docs/SunApprox.html
        //******************something is fishy in this method ***************************************

        // compute declination angle of sun and equation of time
        float PrayTime::sunPosition(float jd, int index) {
                float D = jd - 2451545.0;
                float g = fixangle(357.529 + 0.98560028 * D);
                float q = fixangle(280.459 + 0.98564736 * D);
                float L = fixangle(q + 1.915 * dsin(g) + 0.020 * dsin(2 * g));

                //float R = 1.00014 - 0.01671 * dcos(g) - 0.00014 * dcos(2 * g);
                float e = 23.439 - 0.00000036 * D;

                float d = darcsin(dsin(e) * dsin(L));
                float RA = darctan2(dcos(e) * dsin(L), dcos(L)) / 15;
                RA = fixhour(RA);
                float EqT = q / 15 - RA;

                float result[2] = { d, EqT };
                return result[index];
        }
        // compute equation of time
        float PrayTime::equationOfTime(float jd) {
                return sunPosition(jd,1);
        }

        // compute declination angle of sun
        float PrayTime::sunDeclination(float jd) {
                return sunPosition(jd, 0);
        }

        //---------------------- Compute Prayer Times -----------------------

        // convert hours to day portions
        float * PrayTime::dayPortion(float times[]) {
                for (int i = 0; i < 7; i++) {
                        times[i] /= 24;
                }
                return times;
        }

        // compute mid-day (Dhuhr, Zawal) time
        float PrayTime::computeMidDay(float t) {
                float T = equationOfTime(jDate + t);
                float Z = fixhour(12 - T);
                return Z;
        }

        // compute time for a given angle G
        float PrayTime::computeTime(float G, float t) {
                float D = sunDeclination(jDate + t);
                float Z = computeMidDay(t);
                float V = (1.0 /15.0) * darccos((-dsin(G) - dsin(D) * dsin(lat)) / (dcos(D)* dcos(lat)));
                return Z + (G > 90 ? -V : V);
        }

        // compute the time of Asr
        float PrayTime::computeAsr(int step, float t) // Shafii: step=1, Hanafi: step=2
        {
                float D = sunDeclination(jDate + t);
                float latD = lat - D;
                float G = -darccot(step + dtan(fabs(latD)));
                return computeTime(G, t);
        }

        // compute prayer times at given julian date
        void PrayTime::computeTimes(float times[]) {
                float * t = dayPortion(times);
                float Fajr = computeTime(180 - calculationMethod.getFajrAngle(), t[0]);
                float Sunrise = computeTime(180 - 0.833, t[1]);
                float Dhuhr = computeMidDay(t[2]);
                float Asr = computeAsr(1 + asrJuristic, t[3]);
                float Sunset = computeTime(0.833, t[4]);
                float Maghrib = computeTime(calculationMethod.getMaghribParam(), t[5]);
                float Isha = computeTime(calculationMethod.getIshaParam(), t[6]);

                float results[7] = { Fajr, Sunrise, Dhuhr, Asr, Sunset, Maghrib, Isha };
                for (int i = 0; i < 7; i++){
                        resultArray[i] = results[i];
                }
        }

        // adjust times in a prayer time array
        void PrayTime::adjustTimes(float * array) {
                float *times = array;
                for (int i = 0; i < 7; i++)
                        times[i] += timeZone - lng / 15.0;
                times[2] += 1 / 60; //Dhuhr
                if (calculationMethod.getMaghribSelector() == 1) // Maghrib
                        times[5] = times[4] + calculationMethod.getMaghribParam() / 60;
                if (calculationMethod.getIshaSelector() == 1) // Isha
                        times[6] = times[5] + calculationMethod.getIshaParam() / 60;
                for (int i = 0; i < 7; i++){
                        resultArray[i] = times[i];
                }
        }

        // convert times array to given time format
        void PrayTime::adjustTimesFormat(float times[]) {
                QString sTimes[7];
                for (int i = 0; i < 7; i++) {
                        if (timeFormat == time12) {
                                sTimes[i] = floatToTime12(times[i], false);
                        } else if (timeFormat == floatFormat) {
                                sTimes[i] = floatToString(times[i]);
                        } else if (timeFormat == time12NS) {
                                sTimes[i] = floatToTime12(times[i], true);
                        } else {
                                sTimes[i] = floatToTime24(times[i]);
                        }
                }
                for (int i = 0; i < 7; i++){
                        stringArray[i] = sTimes[i];
                }
        }

        // compute prayer times at given julian date
        void PrayTime::computeDayTimes() {

                float times[] = { 5, 6, 12, 13, 18, 18, 18 }; //default times
                computeTimes(times);

                adjustTimes(resultArray);
                adjustTimesFormat(resultArray);
        }

        // adjust Fajr, Isha and Maghrib for locations in higher latitudes
        float * PrayTime::adjustHighLatTimes(float times[]) {
                float nightTime = timeDiff(times[4], times[1]); // sunset to sunrise

                // Adjust Fajr
                float FajrDiff = nightPortion(calculationMethod.getFajrAngle()) * nightTime;
                if (timeDiff(times[0], times[1]) > FajrDiff)
                        times[0] = times[1] - FajrDiff;

                // Adjust Isha
                float IshaAngle = (calculationMethod.getIshaSelector() == 0) ? calculationMethod.getIshaParam(): 18;
                float IshaDiff = nightPortion(IshaAngle) * nightTime;

                if (timeDiff(times[4], times[6]) > IshaDiff)
                        times[6] = times[4] + IshaDiff;

                // Adjust Maghrib
                float MaghribAngle = (calculationMethod.getMaghribSelector() == 0) ? calculationMethod.getIshaParam(): 4;
                float MaghribDiff = nightPortion(MaghribAngle) * nightTime;
                //		if (isNaN(times[5]) || timeDiff(times[4], times[5]) > MaghribDiff)
                if (timeDiff(times[4], times[5]) > MaghribDiff)
                        times[5] = times[4] + MaghribDiff;

                return times;
        }

        // the night portion used for adjusting times in higher latitudes
        float PrayTime::nightPortion(float angle) {
            if (adjustHighLats == angleBased){
                        return 1 / 60 * angle;
                    }
                if (adjustHighLats == midNight){
                        return 1 / 2;
                    }
                if (adjustHighLats == oneSeventh){
                        return 1 / 7;
                    }
        }

        //---------------------- Misc Functions -----------------------


        // compute the difference between two times
        float PrayTime::timeDiff(float time1, float time2) {
                return fixhour(time2 - time1);
        }

        // add a leading 0 if necessary
        QString PrayTime::twoDigitsFormat(float num) {
                return (num < 10) ? '0' + floatToString(num) : floatToString(num);
        }

        //-------------------- Interface Functions --------------------

        // return prayer times for a given date
        void PrayTime::getDatePrayerTimes(int year, int month, int day, float latitude,
                        float longitude, float tZ) {
                float jd;
                jd = calcJD(year, month, day);
                //Date currentDate;
                lat = latitude;
                lng = longitude;
                //timeZone = currentDate.getTimeZone();
                timeZone = tZ;
                jDate = jd - longitude / (15 * 24);
                computeDayTimes();
        }

        // return prayer times for a given date
        void PrayTime::getPrayerTimes(QDateTime dateTime, float latitude, float longitude, float timeZone) {
            getDatePrayerTimes(dateTime.date().year(), dateTime.date().month(), dateTime.date().day(), latitude, longitude, timeZone);
        }

        PrayTime::PrayTime(int calcMethod, int juristicMethod, int latMethod, int timeFormat) {
                setCalcMethod(calcMethod);
                setAsrMethod(juristicMethod);
                setHighLatsMethod(latMethod);
                setTimeFormat(timeFormat);
        }

        QString * PrayTime::getPrayTimesArray(){
            return stringArray;
        }

        /*PrayTime::~PrayTime(){
            delete(stringArray);
        }*/

