// Copyright 2010 Jochen Becher
//
// This file is part of MovieSchedule.
//
// MovieSchedule is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// MovieSchedule is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with MovieSchedule.  If not, see <http://www.gnu.org/licenses/>.

#include "contextdialog.h"
#include "ui_contextdialog.h"

#include "data/cinemaschedule.h"
#include "data/scheduleentrykey.h"
#include "data/cinema.h"
#include "data/movie.h"
#include "utils/assertedlocker.h"
#include "uiutils.h"

#include <QLayoutItem>
#include <QDesktopWidget>

ContextDialog::ContextDialog(CinemaSchedule *cinema_schedule, QWidget *parent) :
        QDialog(parent),
        ui(new Ui::ContextDialog),
        _cinema_schedule(cinema_schedule),
        _next_row(0),
        _next_column(0)
{
    ui->setupUi(this);
    removeAllWidgets();
    connect(ui->add_to_calendar, SIGNAL(clicked()), this, SLOT(AddToCalendar()));
    connect(ui->call_theater_by_phone, SIGNAL(clicked()), this, SLOT(CallTheaterByPhone()));
    connect(ui->find_route_to_theater, SIGNAL(clicked()), this, SLOT(FindRouteToTheater()));
    connect(ui->search_movie_in_web, SIGNAL(clicked()), this, SLOT(SearchMovieInWeb()));
    connect(ui->search_theater_in_web, SIGNAL(clicked()), this, SLOT(SearchTheaterInWeb()));
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(Rotate()));
    Rotate();
}

ContextDialog::~ContextDialog()
{
    delete ui;
}

void ContextDialog::Show(ScheduleEntryKey schedule_entry_key)
{
    removeAllWidgets();
    _schedule_entry_key = schedule_entry_key;
    AssertedReadLocker locker(_cinema_schedule->GetLock());
    ScheduleEntry schedule_entry = _cinema_schedule->FindScheduleEntry(_schedule_entry_key);
    if (schedule_entry.IsValid()) {
        addWidget(ui->add_to_calendar);
        if (!schedule_entry.GetCinema()->GetTelephone().isEmpty()) {
            addWidget(ui->call_theater_by_phone);
        }
        if (!schedule_entry.GetCinema()->GetAddress().isEmpty()) {
            // TODO enable if find route is implemented
            // addWidget(ui->find_route_to_theater);
        }
        addWidget(ui->search_movie_in_web);
        if (!schedule_entry.GetCinema()->GetAddress().isEmpty()) {
            addWidget(ui->search_theater_in_web);
        }
        adjustSize();
        show();
    }
}

void ContextDialog::changeEvent(QEvent *e)
{
    QDialog::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void ContextDialog::removeAllWidgets()
{
    QLayoutItem *item;
    while ((item = ui->gridLayout->itemAt(0)) != 0) {
        if (item->widget() != 0) {
            item->widget()->setVisible(false);
        }
        ui->gridLayout->removeItem(item);
    }
    _next_row = 0;
    _next_column = 0;
}

void ContextDialog::addWidget(QWidget *widget)
{
    bool landscape = UiUtils::IsLandscape();
    widget->setVisible(true);
    ui->gridLayout->addWidget(widget, _next_row, _next_column, 1, 1);
    ++_next_column;
    if (_next_column > (landscape ? 1 : 0)) {
        ++_next_row;
        _next_column = 0;
    }
}

void ContextDialog::AddToCalendar()
{
    hide();
    emit AddToCalendar(_schedule_entry_key);
}

void ContextDialog::CallTheaterByPhone()
{
    hide();
    emit CallTheaterByPhone(_schedule_entry_key.GetCinemaKey());
}

void ContextDialog::FindRouteToTheater()
{
    hide();
    emit FindRouteToTheater(_schedule_entry_key.GetCinemaKey());
}

void ContextDialog::SearchMovieInWeb()
{
    hide();
    emit SearchMovieInWeb(_schedule_entry_key.GetMovieKey());
}

void ContextDialog::SearchTheaterInWeb()
{
    hide();
    emit SearchTheaterInWeb(_schedule_entry_key.GetCinemaKey());
}

void ContextDialog::Rotate()
{
    if (isVisible()) {
        Show(_schedule_entry_key);
    }
}
